﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Reactive.Linq;
using BezelEditor.Mvvm;
using Nintendo.Authoring.AuthoringEditor.Controls;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Pages;
using Nintendo.Authoring.AuthoringEditor.Properties;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using SimpleInjector;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params
{
    public class AocContentVm : ViewModelBase
    {
        public string Title =>
            string.IsNullOrEmpty(Model.Tag)
                ? Model.Index.ToString()
                : $"{Model.Index} ({Model.Tag})";

        public ulong Index => Model.Index;

        public AocContent Model { get; }
        public ParamVm[] Params { get; }
        public ReactiveProperty<bool> HasErrors { get; protected set; }

        public AocContentVm(Container diContainer, AocContent aocContent, Project project)
        {
            Model = aocContent;

            /////////////////////////////////////////////////////////
            HasErrors = aocContent
                .ToReactivePropertyAsSynchronized(x => x.HasErrors)
                .AddTo(CompositeDisposable);

            var isReadOnlyRp = project.Meta.ToReactivePropertyAsSynchronized(x => x.IsReadOnly);
            var appMode = diContainer.GetInstance<AppProfile>().AppMode;

            var paramVms = new List<ParamVm>();

            /////////////////////////////////////////////////////////
            var indexRp = aocContent.ToReactivePropertyAsSynchronized(x => x.Index);
            {
                indexRp
                    .SetValidateNotifyError(
                        s =>
                            diContainer.GetInstance<PageValidations>().AocIndex(s, aocContent, project.AocMeta.Contents));

                indexRp
                    // ReSharper disable once ExplicitCallerInfoArgument
                    .Subscribe(_ => RaisePropertyChanged(nameof(Index)))
                    .AddTo(CompositeDisposable);

                var p = new CapacityParamVm(
                    nameof(Resources.AocIndex_Caption),
                    nameof(Resources.AocIndex_Comment),
                    indexRp,
                    8,
                    Constants.AocIndexMinimum,
                    Constants.AocIndexMaximum,
                    null
                ).AddTo(CompositeDisposable);
                p.IsReadOnly = isReadOnlyRp;
                p.IsVisibleStyleSelector = false;
                p.EditingStyle = CapacityEditBox.EditingStyles.Bytes;



                paramVms.Add(p);
            }

            /////////////////////////////////////////////////////////
            var tagRp = aocContent.ToReactivePropertyAsSynchronized(x => x.Tag);
            {
                tagRp
                    .SetValidateNotifyError(
                        s => diContainer.GetInstance<PageValidations>().AocTag(s, aocContent, project.AocMeta.Contents));

                var p = new StringParamVm(
                    nameof(Resources.AocContnetTag_Caption),
                    nameof(Resources.AocContnetTag_Comment),
                    tagRp
                ).AddTo(CompositeDisposable);
                p.IsReadOnly = isReadOnlyRp;
                p.IsExpandParentWidth.Value = true;
                p.MaxLength = AocContent.MaxTagLength;

                paramVms.Add(p);

                project.AocMeta.Contents.CollectionChangedAsObservable()
                    .Subscribe(_ => tagRp.ForceValidate())
                    .AddTo(CompositeDisposable);
            }

            /////////////////////////////////////////////////////////
            {
                var p = new CapacityParamVm(
                    nameof(Resources.ReleaseVersion_Caption),
                    nameof(Resources.ReleaseVersion_Comment),
                    aocContent.ToReactivePropertyAsSynchronized(x => x.ReleaseVersion),
                    4,
                    0,
                    0xFFFF,
                    null
                ).AddTo(CompositeDisposable);
                p.IsReadOnly = isReadOnlyRp;
                p.IsVisibleStyleSelector = false;
                p.EditingStyle = CapacityEditBox.EditingStyles.Bytes;

                paramVms.Add(p);
            }

            /////////////////////////////////////////////////////////
            {
                var p = new CapacityParamVm(
                    nameof(Resources.RequiredApplicationReleaseVersion_Caption),
                    nameof(Resources.RequiredApplicationReleaseVersion_Comment),
                    aocContent.ToReactivePropertyAsSynchronized(x => x.RequiredApplicationReleaseVersion),
                    4,
                    0,
                    0xFFFF,
                    null
                ).AddTo(CompositeDisposable);
                p.IsReadOnly = isReadOnlyRp;
                p.IsVisibleStyleSelector = false;
                p.EditingStyle = CapacityEditBox.EditingStyles.Bytes;

                paramVms.Add(p);
            }

            /////////////////////////////////////////////////////////
            if (appMode == AppModeType.AocMeta)
            {
                var p = Utilities.CreateExpandablePathVm(
                    project,
                    aocContent.DataPath,
                    pathRp => new DirPathStringParamVm(
                        nameof(Resources.DataPath_Caption),
                        nameof(Resources.DataPath_Comment),
                        pathRp,
                        pathRp.Select(x => diContainer.GetInstance<Project>().ToAbsolutePath(aocContent.DataPath))
                    )
                    {
                        IsReadOnly = isReadOnlyRp
                    },
                    _ => diContainer.GetInstance<PageValidations>().AocDataPath(aocContent),
                    CompositeDisposable);

                paramVms.Add(p);
            }

            /////////////////////////////////////////////////////////
            if (appMode == AppModeType.AocNsp)
            {
                var p = new StringParamVm(
                    nameof(Resources.RomDigest_Caption),
                    null,
                    aocContent.ToReactivePropertyAsSynchronized(x => x.Digest)
                ).AddTo(CompositeDisposable);
                p.IsReadOnly = isReadOnlyRp;

                paramVms.Add(p);
            }

            /////////////////////////////////////////////////////////
            if (appMode == AppModeType.AocNsp)
            {
                var p = new CapacityParamVm(
                    nameof(Resources.ContentDataSize),
                    null,
                    aocContent.ToReactivePropertyAsSynchronized(x => x.ContentDataSize),
                    16,
                    0x0000000000000000,
                    0x7FFFFFFFFFFFFFFF,
                    null
                ).AddTo(CompositeDisposable);
                p.IsReadOnly = isReadOnlyRp;

                paramVms.Add(p);
            }

            /////////////////////////////////////////////////////////
            Params = paramVms.ToArray();

            Observable
                .Merge(indexRp.ToUnit())
                .Merge(tagRp.ToUnit())
                // ReSharper disable once ExplicitCallerInfoArgument
                .Subscribe(_ => RaisePropertyChanged(nameof(Title)))
                .AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////
            aocContent.ObserveProperty(x => x.HasErrors)
                .Subscribe(_ =>
                {
                    indexRp.ForceValidate();
                    tagRp.ForceValidate();
                }).AddTo(CompositeDisposable);

            aocContent.Parent.Contents.CollectionChangedAsObservable()
                .Subscribe(_ =>
                {
                    indexRp.ForceValidate();
                    tagRp.ForceValidate();
                }).AddTo(CompositeDisposable);
        }
    }
}
