﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Reactive.Linq;
using Nintendo.Authoring.AuthoringEditor.Controls;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Controls;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using SimpleInjector;
using static Nintendo.Authoring.AuthoringEditor.Properties.Resources;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Pages
{
    public class SaveDataPageVm : PageVmBase
    {
        public CapacityParamVm SaveDataSizeMax { get; private set; }
        public CapacityParamVm SaveDataJournalSizeMax { get; private set; }


        public SaveDataPageVm(Container diContainer, Project project)
            : base(nameof(SaveData))
        {
            var isReadOnlyRp = project.Meta.ToReactivePropertyAsSynchronized(x => x.IsReadOnly).AddTo(CompositeDisposable);

            var validations = diContainer.GetInstance<PageValidations>();
            var appMeta = project.Meta;
            var app = appMeta.Application;

            /////////////////////////////////////////////////////////
            var summary = new SaveDataSummryParamVm(
                nameof(TotalSaveData_Caption),
                nameof(TotalSaveData_Comment),
                appMeta
            ).AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////
            var saveDataSizeRp = app.ToReactivePropertyAsSynchronized(x => x.SaveDataSize);
            var saveDataSize = CreateCapacityParamVm(
                nameof(SaveDataSize_Caption),
                nameof(SaveDataSize_Comment),
                saveDataSizeRp,
                nameof(SaveData_Use),
                app.ToReactivePropertyAsSynchronized(x => x.IsUseSaveData),
                s => validations.SaveDataSize(app.IsUseSaveData, s));

            saveDataSize.IsReadOnly = isReadOnlyRp;
            saveDataSize.Additional = new RunSaveDataSizeSimulatorVm(appMeta).AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////
            var journalSizeRp = app.ToReactivePropertyAsSynchronized(x => x.SaveDataJournalSize);
            var journalSize = CreateCapacityParamVm(
                nameof(SaveDataJournalSize_Caption),
                nameof(SaveDataJournalSize_Comment),
                journalSizeRp,
                nameof(SaveDataJournal_Specified),
                app.ToReactivePropertyAsSynchronized(x => x.IsSpecifiedSaveDataJournal),
                s => validations.SaveDataJournalSize(app.IsSpecifiedSaveDataJournal, s));

            journalSize.IsReadOnly = isReadOnlyRp;

            /////////////////////////////////////////////////////////
            var dataLossConf = EnumParamVm.Factory(
                nameof(DataLossConfirmation_Caption),
                nameof(DataLossConfirmation_Comment),
                app.ToReactivePropertyAsSynchronized(x => x.DataLossConfirmation)
            ).AddTo(CompositeDisposable);

            dataLossConf.IsReadOnly = isReadOnlyRp;

            /////////////////////////////////////////////////////////
            if (project.AppCapability.IsSupportSaveDataSizeExtend)
            {
                CreateSaveDataSizeExtendParams(app, validations, isReadOnlyRp);
            }

            /////////////////////////////////////////////////////////

            // ジャーナルサイズが指定されていなければジャーナルサイズをセーブデータと同期させる
            Observable
                .Merge(saveDataSizeRp.ToUnit())
                .Merge(journalSize.IsUse.ToUnit())
                .Where(_ => app.IsSpecifiedSaveDataJournal == false)
                .Subscribe(_ => app.SaveDataJournalSize = app.SaveDataSize)
                .AddTo(CompositeDisposable);

            var pageParams = new List<ParamVm>
            {
                summary,
                saveDataSize,
                journalSize,
                dataLossConf,
                SaveDataSizeMax,
                SaveDataJournalSizeMax
            };

            /////////////////////////////////////////////////////////
            CultureService.Instance.ObserveProperty(x => x.Resources)
                .Subscribe(_ =>
                {
                    saveDataSizeRp.ForceNotify();
                    journalSizeRp.ForceNotify();
                })
                .AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////
            Params = pageParams.Where(x => x != null).ToArray();
        }

        private void CreateSaveDataSizeExtendParams(Application app, PageValidations validations, ReactiveProperty<bool> isReadOnlyRp)
        {
            /////////////////////////////////////////////////////////
            SaveDataSizeMax = CreateCapacityParamVm(
                nameof(SaveDataSizeMax_Caption),
                nameof(SaveDataSizeMax_Comment),
                app.ToReactivePropertyAsSynchronized(x => x.UserAccountSaveDataSizeMax)
                    .SetValidateNotifyError(s => validations.SaveDataSizeMax(app.IsUseUserAccountSaveDataSizeMax, s)),
                nameof(IsUse),
                app.ToReactivePropertyAsSynchronized(x => x.IsUseUserAccountSaveDataSizeMax));

            SaveDataSizeMax.EditingStyle = CapacityEditBox.EditingStyles.MegaBytes;
            SaveDataSizeMax.IsReadOnly = isReadOnlyRp;

            /////////////////////////////////////////////////////////
            SaveDataJournalSizeMax = CreateCapacityParamVm(
                nameof(SaveDataJournalSizeMax_Caption),
                nameof(SaveDataJournalSizeMax_Comment),
                app.ToReactivePropertyAsSynchronized(x => x.UserAccountSaveDataJournalSizeMax)
                    .SetValidateNotifyError(s => validations.SaveDataSizeMax(app.IsUseUserAccountSaveDataJournalSizeMax, s)),
                nameof(IsUse),
                app.ToReactivePropertyAsSynchronized(x => x.IsUseUserAccountSaveDataJournalSizeMax));

            SaveDataJournalSizeMax.EditingStyle = CapacityEditBox.EditingStyles.MegaBytes;
            SaveDataJournalSizeMax.IsReadOnly = isReadOnlyRp;

            /////////////////////////////////////////////////////////
            var saveDataSizeMaxRp = (ReactiveProperty<long>) SaveDataSizeMax.Property;
            var saveDataJournalSizeMaxRp = (ReactiveProperty<long>) SaveDataJournalSizeMax.Property;

            CultureService.Instance.ObserveProperty(x => x.Resources)
                .Subscribe(_ =>
                {
                    saveDataSizeMaxRp.ForceNotify();
                    saveDataJournalSizeMaxRp.ForceNotify();
                })
                .AddTo(CompositeDisposable);

            Observable
                .Merge(saveDataSizeMaxRp.ToUnit())
                .Merge(saveDataJournalSizeMaxRp.ToUnit())
                .Subscribe(_ => HasErrors.Value = saveDataSizeMaxRp.HasErrors || saveDataJournalSizeMaxRp.HasErrors)
                .AddTo(CompositeDisposable);
        }
    }
}
