﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Reactive.Linq;
using System.Windows;
using BezelEditor.Mvvm;
using Nintendo.Authoring.AuthoringEditor.Controls;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using static Nintendo.Authoring.AuthoringEditor.Properties.Resources;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Pages
{
    public abstract class PageVmBase : ViewModelBase
    {
        public virtual string Title => ResourceManager.GetString(_titleResourceTag, Culture);

        public ParamVm[] Params { get; protected set; }

        public ReactiveProperty<bool> HasErrors { get; protected set; }

        public ReactiveProperty<Visibility> Visibility { get; }
        public ReactiveProperty<Thickness> Margin { get; }
        public ReactiveProperty<double> Height { get; }

        public PageGroups PageGroup { get; set; }

        public void UpdateVisibility(string keyword)
        {
            var isVisible = false;

            if (Params != null)
            {
                foreach (var param in Params)
                {
                    param.UpdateVisibility(keyword);

                    if (param.Visibility.Value == System.Windows.Visibility.Visible)
                        isVisible = true;
                }
            }

            Visibility.Value = isVisible ? System.Windows.Visibility.Visible : System.Windows.Visibility.Collapsed;
        }

        private readonly string _titleResourceTag;

        protected PageVmBase(string titleResourceTag)
        {
            _titleResourceTag = titleResourceTag;

            Visibility = new ReactiveProperty<Visibility>(System.Windows.Visibility.Visible).AddTo(CompositeDisposable);
            Margin =
                new ReactiveProperty<Thickness>(new Thickness(12, 16, 16, PageBottomMargin)).AddTo(CompositeDisposable);
            Height = new ReactiveProperty<double>().AddTo(CompositeDisposable);

            CultureService.Instance.ObserveProperty(x => x.Resources)
                // ReSharper disable once ExplicitCallerInfoArgument
                .Subscribe(_ => RaisePropertyChanged(nameof(Title)))
                .AddTo(CompositeDisposable);

            HasErrors = new ReactiveProperty<bool>().AddTo(CompositeDisposable);
        }

        public const double PageBottomMargin = 16;

        protected CapacityParamVm CreateCapacityParamVm<TProperty>(
            string caption,
            string comment,
            ReactiveProperty<TProperty> property,
            string isUseCaption,
            ReactiveProperty<bool> isUseProperty,
            Func<TProperty, string> warningMessage = null)
        {
            var paramVm = new CapacityParamVm(
                caption,
                comment,
                property,
                16,
                0x0000000000000000,
                0x7FFFFFFFFFFFFFFF,
                null
            ).AddTo(CompositeDisposable);

            paramVm.CaptionIsUseTag = isUseCaption;
            paramVm.IsUse = isUseProperty.AddTo(CompositeDisposable);
            paramVm.IsUse.Subscribe(_ => property.ForceNotify()).AddTo(CompositeDisposable);
            paramVm.EditingStyleCandidates = new[]
            {
                CapacityEditBox.EditingStyles.Bytes,
                CapacityEditBox.EditingStyles.KiroBytes,
                CapacityEditBox.EditingStyles.MegaBytes
            };

            if (warningMessage != null)
            {
                property.Select(warningMessage)
                    .Subscribe(m => paramVm.WarningMessage.Value = m)
                    .AddTo(CompositeDisposable);
            }

            return paramVm;
        }
    }
}
