﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Linq;
using System.Reactive.Linq;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using SimpleInjector;
using static Nintendo.Authoring.AuthoringEditor.Properties.Resources;
using Application = Nintendo.Authoring.AuthoringEditor.Core.Application;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Pages
{
    public class GeneralPageVm : PageVmBase
    {
        public GeneralPageVm(Container diContainer, Project project, ApplicationMeta appMeta)
            : base(nameof(General))
        {
            var isReadOnlyRp = appMeta.ToReactivePropertyAsSynchronized(x => x.IsReadOnly).AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////
            var logoHandling = EnumParamVm.Factory(
                nameof(LogoHandling_Caption),
                nameof(LogoHandling_Comment),
                appMeta.Application.ToReactivePropertyAsSynchronized(x => x.LogoHandling)
            ).AddTo(CompositeDisposable);
            logoHandling.IsReadOnly = isReadOnlyRp;

            /////////////////////////////////////////////////////////
            var presenceGroupId = new HexParamVm(
                nameof(PresenceGroupId_Caption),
                nameof(PresenceGroupId_Comment),
                appMeta.Application.ToReactivePropertyAsSynchronized(x => x.PresenceGroupId),
                16,
                ulong.MinValue,
                ulong.MaxValue
            ).AddTo(CompositeDisposable);
            presenceGroupId.IsReadOnly = isReadOnlyRp;
            presenceGroupId.IsUse =
                appMeta.Application.ToReactivePropertyAsSynchronized(x => x.IsUsePresenceGroupIdAppId)
                    .AddTo(CompositeDisposable);
            presenceGroupId.IsInputEnabled =
                presenceGroupId.IsUse.Inverse().ToReactiveProperty().AddTo(CompositeDisposable);
            presenceGroupId.CaptionIsUseTag = nameof(PresenceGroupId_UseApplicationId);

            /////////////////////////////////////////////////////////
            var appErrorCodeCategoryRp =
                appMeta.Application
                    .ToReactivePropertyAsSynchronized(x => x.ApplicationErrorCodeCategory)
                    .SetValidateNotifyError(s => diContainer.GetInstance<PageValidations>().ApplicationErrorCodeCategory(appMeta, s));

            var appErrorCodeCategory = new StringParamVm(
                nameof(ApplicationErrorCodeCategory_Caption),
                nameof(ApplicationErrorCodeCategory_Comment),
                appErrorCodeCategoryRp
            ).AddTo(CompositeDisposable);
            appErrorCodeCategory.IsReadOnly = isReadOnlyRp;
            appErrorCodeCategory.Width = GuiConstants.ExtraSmallWidth;

            // 使用するしない
            appErrorCodeCategory.IsUse =
                appMeta.Application.ToReactivePropertyAsSynchronized(x => x.IsUseApplicationErrorCode)
                    .AddTo(CompositeDisposable);
            appErrorCodeCategory.CaptionIsUseTag = nameof(UseApplicationErrorCode);
            appErrorCodeCategory.MaxLength = Core.Application.MaxApplicationErrorCodeCategoryLength;

            // バリデーション更新
            appErrorCodeCategory.IsUse.Subscribe(_ => appErrorCodeCategoryRp.ForceValidate())
                .AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////
            HexParamVm seedForPseudoDeviceId = null;
            if (project.AppCapability.IsSupportSeedForPseudoDeviceId)
            {
                seedForPseudoDeviceId = new HexParamVm(
                    nameof(SeedForPseudoDeviceId_Caption),
                    nameof(SeedForPseudoDeviceId_Comment),
                    appMeta.Application.ToReactivePropertyAsSynchronized(x => x.SeedForPseudoDeviceId),
                    16,
                    ulong.MinValue,
                    ulong.MaxValue
                ).AddTo(CompositeDisposable);
                seedForPseudoDeviceId.IsReadOnly = isReadOnlyRp;
                seedForPseudoDeviceId.IsUse =
                    appMeta.Application.ToReactivePropertyAsSynchronized(x => x.IsUseSeedForPseudoDeviceAppId)
                        .AddTo(CompositeDisposable);
                seedForPseudoDeviceId.IsInputEnabled =
                    seedForPseudoDeviceId.IsUse.Inverse().ToReactiveProperty().AddTo(CompositeDisposable);
                seedForPseudoDeviceId.CaptionIsUseTag = nameof(SeedForPseudoDeviceId_UseApplicationId);
            }

            // 非公開プロパティのため、値が指定されているときだけ表示する
            EnumParamVm playLogPolicy = null;
            if (appMeta.Application.PlayLogPolicySpecified &&
                appMeta.Application.PlayLogPolicy != PlayLogPolicyType.All)
            {
                playLogPolicy = EnumParamVm.Factory(
                    nameof(PlayLogPolicy_Caption),
                    nameof(PlayLogPolicy_Comment),
                    appMeta.Application.ToReactivePropertyAsSynchronized(x => x.PlayLogPolicy));
                playLogPolicy.IsReadOnly = isReadOnlyRp;
            }

            // 非公開プロパティのため、値が指定されているときだけ表示
            EnumParamVm hdcp = null;
            {
                var appProfile = diContainer.GetInstance<AppProfile>();
                var isNsp = appProfile.AppMode.IsNsp();
                // nsp の場合は値が None/Required どちらかに設定されているので、編集項目は Required のときのみ表示
                var isHdpcRequiredInNsp = isNsp && appMeta.Application.Hdcp == Application.HdcpType.Required;
                // nmeta の場合は値が null　以外の時に表示
                var isHdcpSpecifiedInMeta = isNsp == false && appMeta.Application.Hdcp.HasValue;
                if (isHdpcRequiredInNsp || isHdcpSpecifiedInMeta)
                {
                    hdcp = EnumParamVm.Factory(
                        nameof(Hdcp_Caption),
                        nameof(Hdcp_Comment),
                        appMeta.Application.ToReactivePropertyAsSynchronized(x => x.Hdcp));
                    hdcp.IsReadOnly = isReadOnlyRp;
                }
            }

            /////////////////////////////////////////////////////////
            Params = new ParamVm[]
            {
                logoHandling,
                presenceGroupId,
                appErrorCodeCategory,
                seedForPseudoDeviceId,
                playLogPolicy,
                hdcp
            }.Where(x => x != null).ToArray();

            /////////////////////////////////////////////////////////
            appMeta.Application.ObserveProperty(x => x.ValidationApplicationErrorCodeCategory)
                .Where(_ => isReadOnlyRp.Value == false)
                .Subscribe(i => HasErrors.Value = i != Core.Application.ApplicationErrorCodeCategoryValidationType.Ok)
                .AddTo(CompositeDisposable);
        }
    }
}
