﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Reactive.Linq;
using Nintendo.Authoring.AuthoringEditor.Controls;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using SimpleInjector;
using static Nintendo.Authoring.AuthoringEditor.Properties.Resources;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Pages
{
    public class DeviceSaveDataPageVm : PageVmBase
    {
        public CapacityParamVm DeviceSaveDataSizeMax { get; private set; }
        public CapacityParamVm DeviceSaveDataJouranlSizeMax { get; private set; }

        public DeviceSaveDataPageVm(Container diContainer, Project project)
            : base(nameof(DeviceSaveData))
        {
            var isReadOnlyRp = project.Meta.ToReactivePropertyAsSynchronized(x => x.IsReadOnly).AddTo(CompositeDisposable);
            var validations = diContainer.GetInstance<PageValidations>();
            var app = project.Meta.Application;

            /////////////////////////////////////////////////////////
            var summary = new DeviceSaveDataSummaryParamVm(
                nameof(DeviceSaveData_TotalSaveData_Caption),
                nameof(DeviceSaveData_TotalSaveData_Comment),
                project.Meta
            ).AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////
            var deviceSaveDataSizeRp = app.ToReactivePropertyAsSynchronized(x => x.DeviceSaveDataSize);
            var deviceSaveDataSize = CreateCapacityParamVm(
                nameof(SaveDataSize_Caption),
                nameof(DeviceSaveDataSize_Comment),
                deviceSaveDataSizeRp,
                nameof(DeviceSave_Use),
                app.ToReactivePropertyAsSynchronized(x => x.IsSpecifiedDeviceSaveDataSize),
                s => validations.SaveDataSize(app.IsSpecifiedDeviceSaveDataSize, s));

            deviceSaveDataSize.IsReadOnly = isReadOnlyRp;

            /////////////////////////////////////////////////////////
            var deviceSaveDataJournalSizeRp = app.ToReactivePropertyAsSynchronized(x => x.DeviceSaveDataJournalSize);
            var deviceSaveDataJournalSize = CreateCapacityParamVm(
                nameof(SaveDataJournalSize_Caption),
                nameof(DeviceSaveDataJournalSize_Comment),
                deviceSaveDataJournalSizeRp,
                nameof(DeviceSave_Use),
                app.ToReactivePropertyAsSynchronized(x => x.IsSpecifiedDeviceSaveDataJournalSize),
                s => validations.SaveDataJournalSize(app.IsSpecifiedDeviceSaveDataJournalSize, s));

            deviceSaveDataJournalSize.IsReadOnly = isReadOnlyRp;

            /////////////////////////////////////////////////////////
            if (project.AppCapability.IsSupportSaveDataSizeExtend)
            {
                CreateDeviceSaveDataExtendParams(app, validations, isReadOnlyRp);
            }

            /////////////////////////////////////////////////////////
            CultureService.Instance.ObserveProperty(x => x.Resources)
                .Subscribe(_ =>
                {
                    deviceSaveDataSizeRp.ForceNotify();
                    deviceSaveDataJournalSizeRp.ForceNotify();
                })
                .AddTo(CompositeDisposable);


            /////////////////////////////////////////////////////////
            Params = new ParamVm[]
            {
                summary,
                deviceSaveDataSize,
                deviceSaveDataJournalSize,
                DeviceSaveDataSizeMax,
                DeviceSaveDataJouranlSizeMax
            };
        }

        private void CreateDeviceSaveDataExtendParams(Application app, PageValidations validations,
            ReactiveProperty<bool> isReadOnlyRp)
        {
            DeviceSaveDataSizeMax = CreateCapacityParamVm(
                nameof(SaveDataSizeMax_Caption),
                nameof(DeviceSaveDataSizeMax_Comment),
                app.ToReactivePropertyAsSynchronized(x => x.DeviceSaveDataSizeMax)
                    .SetValidateNotifyError(s => validations.SaveDataSizeMax(app.IsUseDeviceSaveDataSizeMax, s)),
                nameof(IsUse),
                app.ToReactivePropertyAsSynchronized(x => x.IsUseDeviceSaveDataSizeMax));

            DeviceSaveDataSizeMax.IsReadOnly = isReadOnlyRp;

            DeviceSaveDataJouranlSizeMax = CreateCapacityParamVm(
                nameof(SaveDataJournalSizeMax_Caption),
                nameof(DeviceSaveDataJournalSizeMax_Comment),
                app.ToReactivePropertyAsSynchronized(x => x.DeviceSaveDataJournalSizeMax)
                    .SetValidateNotifyError(s => validations.SaveDataSizeMax(app.IsUseDeviceSaveDataJournalSizeMax, s)),
                nameof(IsUse),
                app.ToReactivePropertyAsSynchronized(x => x.IsUseDeviceSaveDataJournalSizeMax));

            DeviceSaveDataJouranlSizeMax.IsReadOnly = isReadOnlyRp;

            var deviceSaveDataSizeMaxRp = (ReactiveProperty<long>) DeviceSaveDataSizeMax.Property;
            var deviceSaveDataJournalSizeMapRp = (ReactiveProperty<long>) DeviceSaveDataJouranlSizeMax.Property;

            CultureService.Instance.ObserveProperty(x => x.Resources)
                .Subscribe(_ =>
                {
                    deviceSaveDataSizeMaxRp.ForceNotify();
                    deviceSaveDataJournalSizeMapRp.ForceNotify();
                })
                .AddTo(CompositeDisposable);

            Observable
                .Merge(deviceSaveDataSizeMaxRp.ToUnit())
                .Merge(deviceSaveDataJournalSizeMapRp.ToUnit())
                .Subscribe(_ =>
                    HasErrors.Value = deviceSaveDataSizeMaxRp.HasErrors || deviceSaveDataJournalSizeMapRp.HasErrors)
                .AddTo(CompositeDisposable);
        }
    }
}
