﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Linq;
using System.Reactive.Linq;
using BezelEditor.Foundation.Extentions;
using Nintendo.Authoring.AuthoringEditor.Controls;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using SimpleInjector;
using static Nintendo.Authoring.AuthoringEditor.Properties.Resources;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Pages
{
    public class CorePageVm : PageVmBase
    {
        public CapacityParamVm StackSize { get; }
        public EnumParamVm AddressSpace { get; }
        public CapacityParamVm SystemResourceSize { get; }

        public CorePageVm(Container diContainer, ApplicationMeta appMeta)
            : base(nameof(Properties.Resources.Application))
        {
            /////////////////////////////////////////////////////////
            var stackSizeRp =
                appMeta.Core.ToReactivePropertyAsSynchronized(x => x.MainThreadStackSize)
                    .SetValidateNotifyError(s => diContainer.GetInstance<PageValidations>().MainThreadStackSize(s));

            /////////////////////////////////////////////////////////
            StackSize = new CapacityParamVm(
                nameof(MainThreadStackSize_Caption),
                nameof(MainThreadStackSize_Comment),
                stackSizeRp,
                8,
                0x00000000,
                0x7FFFFFFF,
                nameof(KiB)
            ).AddTo(CompositeDisposable);
            StackSize.IsVisibleStyleSelector = false;
            StackSize.EditingStyle = CapacityEditBox.EditingStyles.KiroBytes;

            /////////////////////////////////////////////////////////
            var availableAddressSpaces = Enum.GetValues(typeof(ProcessAddressSpaceType)).Cast<object>();

            // AddressSpace32BitNoReserved は meta 側で明示的に指定されていない限り UI として表示しない
            var enabledProcessAddressSpaces =
                appMeta.Core.ProcessAddressSpace == ProcessAddressSpaceType.AddressSpace32BitNoReserved
                    ? availableAddressSpaces
                    : availableAddressSpaces.Where(
                        x => (ProcessAddressSpaceType) x !=
                             ProcessAddressSpaceType.AddressSpace32BitNoReserved);

            var addressSpaceRp = appMeta.Core.ToReactivePropertyAsSynchronized(x => x.ProcessAddressSpace);

            AddressSpace = EnumParamVm.Factory(
                nameof(ProcessAddressSpace_Caption),
                nameof(ProcessAddressSpace_Comment),
                addressSpaceRp,
                enabledProcessAddressSpaces
            ).AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////
            ReactiveProperty<uint> systemResourceSizeRp = null;
            if (diContainer.GetInstance<ApplicationCapability>().IsSupportSystemResourceSize)
            {
                systemResourceSizeRp =
                    appMeta.Core.ToReactivePropertyAsSynchronized(x => x.SystemResourceSize)
                        .SetValidateNotifyError(s =>
                            diContainer.GetInstance<PageValidations>().SystemResourceSize(s, appMeta.Core.IsProcessAddressSpace64Bit));
                SystemResourceSize = new CapacityParamVm(
                    nameof(SystemResourceSize_Caption),
                    nameof(SystemResourceSize_Comment),
                    systemResourceSizeRp,
                    8,
                    0x00000000,
                    0xFFFFFFFF,
                   null
                ).AddTo(CompositeDisposable);
                SystemResourceSize.EditingStyle = CapacityEditBox.EditingStyles.KiroBytes;
                SystemResourceSize.EditingStyleCandidates = new[]
                {
                    CapacityEditBox.EditingStyles.BytesHex,
                    CapacityEditBox.EditingStyles.KiroBytes,
                    CapacityEditBox.EditingStyles.MegaBytes
                };
                SystemResourceSize.IsUse = addressSpaceRp
                    .Select(x => x == ProcessAddressSpaceType.AddressSpace64Bit)
                    .ToReactiveProperty()
                    .AddTo(CompositeDisposable);

                addressSpaceRp.Subscribe(x => systemResourceSizeRp.ForceValidate()).AddTo(CompositeDisposable);
            }

            /////////////////////////////////////////////////////////
            Params = new ParamVm[]
            {
                StackSize,
                AddressSpace,
                SystemResourceSize
            }.Where(x => x != null).ToArray();

            /////////////////////////////////////////////////////////
            var hasErrorObservable = Observable
                .Merge(stackSizeRp.ToUnit())
                .Merge(addressSpaceRp.ToUnit());

            if (systemResourceSizeRp != null)
                hasErrorObservable = hasErrorObservable.Merge(systemResourceSizeRp.ToUnit());

            hasErrorObservable.Subscribe(_ =>
            {
                HasErrors.Value = appMeta.Core.ValidationSystemResourceSize !=
                                  Core.Core.SystemResourceSizeValidationType.Ok ||
                                  appMeta.Core.ValidationMainThreadStackSize !=
                                  Core.Core.MainThreadStackSizeValidationType.Ok;
            }).AddTo(CompositeDisposable);
        }
    }
}
