﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reactive.Linq;
using System.Threading;
using System.Threading.Tasks;
using BezelEditor.Appearance.Properties;
using BezelEditor.Foundation;
using BezelEditor.Foundation.Extentions;
using BezelEditor.Foundation.Types;
using BezelEditor.Foundation.Utilities;
using BezelEditor.Mvvm;
using BezelEditor.Mvvm.Messages;
using GongSolutions.Wpf.DragDrop;
using Livet.Messaging;
using Livet.Messaging.IO;
using Livet.Messaging.Windows;
using Microsoft.WindowsAPICodePack.Dialogs;
using Microsoft.WindowsAPICodePack.Taskbar;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.MainWindow.AppMenu;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ComparisonPanel;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel;
using Nintendo.Authoring.AuthoringEditor.MainWindow.StartupPanel;
using Nintendo.Authoring.AuthoringEditor.MainWindow.SubPanel;
using Nintendo.Authoring.AuthoringEditor.MakeNspPatchWindow;
using Nintendo.Authoring.AuthoringEditor.NspEntriesWindow;
using Nintendo.Authoring.AuthoringEditor.SelectTwoNspFilesWindow;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using SimpleInjector;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow
{
    public partial class MainWindowVm : ViewModelBase, IDropTarget
    {
        public AppMenuVm AppMenuVm { get; }

        public ReactiveProperty<ViewModelBase> SelectedPanel { get; }
        public ReadOnlyReactiveProperty<string> Title { get; }

        public ReactiveCommand<OutputTypes> SaveAsNspCommand { get; private set; }
        public ReactiveCommand ImportNspCommand { get; private set; }
        public ReactiveCommand ShowNspEntriesCommand { get; private set; }
        public ReactiveCommand OpenCommandPromptCommand { get; private set; }
        public ReactiveCommand SaveMetaCommand { get; private set; }
        public ReactiveCommand SaveAsMetaCommand { get; private set; }
        public ReactiveCommand ImportMetaCommand { get; private set; }
        public ReactiveCommand ImportPartialMetaCommand { get; private set; }
        public ReactiveCommand OpenAllCommand { get; private set; }
        public ReactiveCommand CloseAllCommand { get; private set; }
        public ReactiveCommand MakeNspPatchCommand { get; private set; }
        public ReactiveCommand<ComparisonPanelVm.Target> ExtractNspFileCommand { get; private set; }

        public ReactiveProperty<bool> CanClose { get; }

        public ReactiveProperty<bool> IsDisplayOnlyDifferences { get; }
        public ReactiveProperty<bool> IsExtractableFileType { get; }
        public ReactiveProperty<ComparisonPanelVm.Target> ExtractionTarget0 { get; }
        public ReactiveProperty<ComparisonPanelVm.Target> ExtractionTarget1 { get; }

        public ReactiveProperty<AppModeType> AppMode { get; }

        public ReadOnlyReactiveProperty<bool> CanExportProject { get; private set; }
        public ReadOnlyReactiveProperty<bool> CanImportProject { get; private set; }
        public ReadOnlyReactiveProperty<bool> CanSaveAsMeta { get; private set; }
        public ReadOnlyReactiveProperty<bool> CanSaveMeta { get; private set; }
        public ReadOnlyReactiveProperty<bool> CanImportMeta { get; private set; }
        public ReadOnlyReactiveProperty<bool> CanSaveAsNsp { get; private set; }
        public ReadOnlyReactiveProperty<bool> CanImportNsp { get; private set; }
        public ReadOnlyReactiveProperty<bool> CanCompare { get; private set; }
        public ReadOnlyReactiveProperty<bool> CanMakeNspPatch { get; private set; }
        public ReadOnlyReactiveProperty<bool> CanShowNspEntries { get; private set; }
        public ReadOnlyReactiveProperty<bool> CanImportPartialMeta { get; private set; }

        #region CurrentMode

        private PanelType _currentPanelType = PanelType.Nothing;

        public PanelType CurrentPanelType
        {
            get { return _currentPanelType; }
            set
            {
                if (_currentPanelType == value)
                    return;

                _currentPanelType = value;
                ChangePanel(value);
            }
        }

        #endregion

        public WorkingPanelVm WorkingPanel { get; }

        public enum PanelType
        {
            Nothing,
            //
            Startup,
            ProjectEdit,
            Compare
        }

        public ReactiveProperty<string> DelaySearchWord
        {
            get
            {
                if (CurrentPanelType != PanelType.ProjectEdit)
                    return null;

                var panelVm = (ProjectEditPanelVm) SelectedPanel.Value;

                return panelVm.DelaySearchWord;
            }
        }

        public ReadOnlyReactiveProperty<bool> IsWaiting { get; }

        public ReactiveProperty<PathString> TargetFilePath { get; }

        private readonly Container _diContainer;
        private readonly App _app;
        private readonly Config _config;

        private IDisposable _projectObserver;

        public MainWindowVm(Container diContainer, App app, Config config)
        {
            _diContainer = diContainer;
            _app = app;
            _config = config;

            CompositeDisposable.Add(() => _projectObserver?.Dispose());

            WorkingPanel = new WorkingPanelVm(app).AddTo(CompositeDisposable);

            IsWaiting = app.ObserveProperty(x => x.WorkingState)
                .Select(s => s == WorkingState.Waiting)
                .ToReadOnlyReactiveProperty()
                .AddTo(CompositeDisposable);

            var profile = diContainer.GetInstance<AppProfile>();
            AppMode = profile
                .ToReactivePropertyAsSynchronized(x => x.AppMode)
                .AddTo(CompositeDisposable);

            SetupCanRp(profile);

            CompositeDisposable.Add(() => SelectedPanel.Value?.Dispose());

            CanClose = new ReactiveProperty<bool>().AddTo(CompositeDisposable);

            TargetFilePath = new ReactiveProperty<PathString>(string.Empty).AddTo(CompositeDisposable);

            Title = TargetFilePath
                .Select(target =>
                {
                    return target.IsNullOrEmpty
                        ? " AuthoringEditor"
                        : $"{target.FileName} ({target.ToString().TrimEnd('\\')}) - AuthoringEditor";
                })
                .ToReadOnlyReactiveProperty()
                .AddTo(CompositeDisposable);

            UpdateProjectTargetFilePathObserver();
            Observable
                .Merge(app.ObserveProperty(x => x.Project).ToUnit())
                .Subscribe(_ => UpdateProjectTargetFilePathObserver())
                .AddTo(CompositeDisposable);

            AppMenuVm = diContainer.GetInstance<AppMenuVm>().AddTo(CompositeDisposable);
            AppMenuVm.Messenger = Messenger;

            app.ObserveProperty(x => x.WorkingProgress, false)
                .Subscribe(p =>
                {
                    TaskbarManager.Instance.SetProgressState(p == 0
                        ? TaskbarProgressBarState.Indeterminate
                        : TaskbarProgressBarState.Normal);

                    TaskbarManager.Instance.SetProgressValue(Math.Min(Math.Max(p, 0), 100), 100);
                }).AddTo(CompositeDisposable);

            SelectedPanel = new ReactiveProperty<ViewModelBase>().AddTo(CompositeDisposable);

            SetupCommands(app, config);

            IsDisplayOnlyDifferences =
                new ReactiveProperty<bool>(false, ReactivePropertyMode.DistinctUntilChanged).AddTo(CompositeDisposable);
            IsDisplayOnlyDifferences
                .Subscribe(SetDisplayOnlyDifferences)
                .AddTo(CompositeDisposable);

            IsExtractableFileType = new ReactiveProperty<bool>();
            ExtractionTarget0 = new ReactiveProperty<ComparisonPanelVm.Target>().AddTo(CompositeDisposable);
            ExtractionTarget1 = new ReactiveProperty<ComparisonPanelVm.Target>().AddTo(CompositeDisposable);

            InitializeDragAndDrop();
        }

        private void SetupCanRp(AppProfile profile)
        {
            CanExportProject = profile.ObserveProperty(x => x.CanSaveAsMeta)
                .ToReadOnlyReactiveProperty()
                .AddTo(CompositeDisposable);

            CanImportProject = profile.ObserveProperty(x => x.CanImportMeta)
                .ToReadOnlyReactiveProperty()
                .AddTo(CompositeDisposable);

            CanSaveAsMeta = profile.ObserveProperty(x => x.CanSaveAsMeta)
                .ToReadOnlyReactiveProperty()
                .AddTo(CompositeDisposable);

            CanSaveMeta = profile.ObserveProperty(x => x.CanSaveMeta)
                .ToReadOnlyReactiveProperty()
                .AddTo(CompositeDisposable);

            CanImportMeta = profile.ObserveProperty(x => x.CanImportMeta)
                .ToReadOnlyReactiveProperty()
                .AddTo(CompositeDisposable);

            CanSaveAsNsp = profile.ObserveProperty(x => x.CanSaveAsNsp)
                .ToReadOnlyReactiveProperty()
                .AddTo(CompositeDisposable);

            CanImportNsp = profile.ObserveProperty(x => x.CanImportNsp)
                .ToReadOnlyReactiveProperty()
                .AddTo(CompositeDisposable);

            CanCompare = profile.ObserveProperty(x => x.CanCompare)
                .ToReadOnlyReactiveProperty()
                .AddTo(CompositeDisposable);

            CanMakeNspPatch = profile.ObserveProperty(x => x.CanMakeNspPatch)
                .ToReadOnlyReactiveProperty()
                .AddTo(CompositeDisposable);

            CanShowNspEntries = profile.ObserveProperty(x => x.CanShowNspEntries)
                .ToReadOnlyReactiveProperty()
                .AddTo(CompositeDisposable);

            CanImportPartialMeta = profile.ObserveProperty(x => x.CanImportPartialMeta)
                .ToReadOnlyReactiveProperty()
                .AddTo(CompositeDisposable);
        }

        private void SetupCommands(App app, Config config)
        {
            //////////////////////////////////////////////////////////////////////////////////////
            SaveMetaCommand = Observable
                .Merge(app.ObserveProperty(x => x.HasErrors).ToUnit())
                .Merge(Title.ToUnit())
                .Select(_ => app.HasErrors == false && File.Exists(app.Project.TargetFilePath))
                .ToReactiveCommand()
                .AddTo(CompositeDisposable);
            SaveMetaCommand
                .Subscribe(_ => SaveMeta(app, false))
                .AddTo(CompositeDisposable);

            SaveAsMetaCommand = app.ObserveProperty(x => x.HasErrors)
                .Inverse()
                .ToReactiveCommand()
                .AddTo(CompositeDisposable);
            SaveAsMetaCommand
                .Subscribe(isSaveAs => SaveMeta(app, true))
                .AddTo(CompositeDisposable);

            //////////////////////////////////////////////////////////////////////////////////////
            ImportMetaCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            ImportMetaCommand.Subscribe(_ => ImportFile(app, config, ImportableFileType.Meta))
                .AddTo(CompositeDisposable);

            //////////////////////////////////////////////////////////////////////////////////////
            ImportPartialMetaCommand =
                Observable
                    .Merge(app.ObserveProperty(x => x.IsReadOnly).ToUnit())
                    .Merge(CanImportPartialMeta.ToUnit())
                    .Select(_ => app.IsReadOnly == false && CanImportPartialMeta.Value)
                    .ToReactiveCommand()
                    .AddTo(CompositeDisposable);
            ImportPartialMetaCommand
                .Subscribe(_ => ImportPartialMetaFile(app))
                .AddTo(CompositeDisposable);

            //////////////////////////////////////////////////////////////////////////////////////
            SaveAsNspCommand =
                Observable
                    .Merge(app.ObserveProperty(x => x.HasErrors).ToUnit())
                    .Merge(app.ObserveProperty(x => x.IsReadOnly).ToUnit())
                    .Merge(app.ObserveProperty(x => x.WorkingState).ToUnit())
                    .Select(_ =>
                        app.IsReadOnly == false &&
                        app.HasErrors == false &&
                        app.WorkingState == WorkingState.Waiting)
                    .ToReactiveCommand<OutputTypes>()
                    .AddTo(CompositeDisposable);
            SaveAsNspCommand
                .Subscribe(async t => await SaveAsNspAsync(t, app, config))
                .AddTo(CompositeDisposable);

            //////////////////////////////////////////////////////////////////////////////////////
            ShowNspEntriesCommand =
                Observable.Merge(app.ObserveProperty(x => x.IsReadOnly).ToUnit())
                    .Select(_ => app.Project.IsAvailableNspEntries)
                    .ToReactiveCommand();
            ShowNspEntriesCommand.Subscribe(_ => ShowNspEntries(app))
                .AddTo(CompositeDisposable);

            //////////////////////////////////////////////////////////////////////////////////////
            OpenCommandPromptCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            OpenCommandPromptCommand.Subscribe(_ =>
            {
                OpenCommandPrompt();
            }).AddTo(CompositeDisposable);

            //////////////////////////////////////////////////////////////////////////////////////
            ImportNspCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            ImportNspCommand.Subscribe(_ => ImportFile(app, config, ImportableFileType.Nsp))
                .AddTo(CompositeDisposable);

            //////////////////////////////////////////////////////////////////////////////////////
            OpenAllCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            OpenAllCommand.Subscribe(_ => OpenAllComparison(true))
                .AddTo(CompositeDisposable);

            CloseAllCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            CloseAllCommand.Subscribe(_ => OpenAllComparison(false))
                .AddTo(CompositeDisposable);

            ExtractNspFileCommand = new ReactiveCommand<ComparisonPanelVm.Target>().AddTo(CompositeDisposable);
            ExtractNspFileCommand.Subscribe(async target => await ExtractNspFileAsync(target.File))
                .AddTo(CompositeDisposable);

            MakeNspPatchCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            MakeNspPatchCommand.Subscribe(async _ => await MakeNspPatch())
                .AddTo(CompositeDisposable);
        }

        private async Task SaveAsNspAsync(OutputTypes outputType, App app, Config config)
        {
            var outputPath = GetNspOutputPath(OutputTypes.Nsp);
            if (string.IsNullOrEmpty(outputPath))
                return;

            WorkingPanel.IsOpenGeneratedObject.Value = true;
            app.WorkingFilePath = outputPath;

            app.WorkingKind = WorkingKind.Build;
            app.WorkingState = WorkingState.Working;
            {
                NspHandleResultType r;
                if (AppMode.Value.IsAoc())
                {
                    // AoC の場合 app.UpdateOriginalProject は呼び出されないため app.IsChangedTarget の結果に変化はない
                    r = await DoBuildingNspAsync(app, token => app.MakeAocNspAsync(outputPath, token));
                }
                else
                {
                    r = await DoBuildingNspAsync(app, token => app.SaveNspAsync(outputPath, app.Project.TargetFilePath, token));
                }
                if (r == null)
                    return;
            }
            app.WorkingState = WorkingState.Succeeded;

            config.AddToMruList(TypeHelper.ToImportable(outputType), outputPath);
        }

        private enum GetExportingNspPathResult
        {
            Ok,
            Cancel
        }

        private string GetNspOutputPath(OutputTypes outputTypes)
        {
            string outputPath;
            {
                var result = MakeExportingNspPath(out outputPath, outputTypes);
                switch (result)
                {
                    case GetExportingNspPathResult.Ok:
                        break;

                    case GetExportingNspPathResult.Cancel:
                        return null;

                    default:
                        throw new ArgumentOutOfRangeException();
                }
            }
            return outputPath;
        }

        private GetExportingNspPathResult MakeExportingNspPath(out string outputPath, OutputTypes type)
        {
            switch (type)
            {
                case OutputTypes.Nsp:
                {
                    var r = Messenger.GetResponse(new SavingFileSelectionMessage(GuiConstants.MessageKey_FileSave)
                    {
                        Filter = Properties.Resources.DialogFilter_Nsp,
                        Title = Properties.Resources.DialogTitle_OutputNspFile
                    });

                    outputPath = r?.Response?.FirstOrDefault();
                    return string.IsNullOrEmpty(outputPath)
                        ? GetExportingNspPathResult.Cancel
                        : GetExportingNspPathResult.Ok;
                }

                default:
                    throw new ArgumentOutOfRangeException(nameof(type), type, null);
            }
        }

        private void SaveMeta(App app, bool isSaveAs)
        {
            try
            {
                var exportType = AppMode.Value.IsAoc()
                    ? ExportableFileType.AocMeta
                    : ExportableFileType.AppMeta;

                var filePath = app.Project.TargetFilePath;

                if (isSaveAs)
                {
                    var message = Messenger.GetResponse(new SavingFileSelectionMessage(Resources.MessageKey_FileSave)
                    {
                        Filter = Properties.Resources.DialogFilter_Meta
                    });

                    filePath = message?.Response?.FirstOrDefault();
                    if (string.IsNullOrEmpty(filePath))
                        return;
                }

                app.Save(exportType, filePath);

                if (isSaveAs)
                {
                    var r = Messenger.GetResponse(new DialogMessage(Resources.MessageKey_Dialog)
                    {
                        Icon = TaskDialogStandardIcon.Information,
                        Caption = Properties.Resources.Information,
                        Text = Properties.Resources.MetaFileOutputSucceeded,
                        StandardButtons = DialogMessage.StandardButtonsType.Ok,
                        FooterCheckBoxChecked = true,
                        FooterCheckBoxText = Properties.Resources.OpenGeneratedObjectContainsFolder
                    });

                    if (r.Response != null)
                        if (r.Response.FooterCheckBoxChecked)
                            WindowsUtility.OpenFileByExplorer(filePath);
                }
            }
            catch (Exception e)
            {
                Messenger.Raise(new DialogMessage(GuiConstants.MessageKey_Dialog)
                {
                    Icon = TaskDialogStandardIcon.Information,
                    Caption = Properties.Resources.Information,
                    Text = string.Format(Properties.Resources.MetaFileOutputFaild, e.Message),
                    StandardButtons = DialogMessage.StandardButtonsType.Ok
                });
            }
        }

        private void ImportFile(App app, Config config, ImportableFileType type)
        {
            var helper = _diContainer.GetInstance<StringHelper>();

            if (ProjectFileHelper.CheckTargetModified(type, Messenger, app, AppMode.Value, helper.ImportFromType(type)) == false)
                return;

            ProjectFileHelper.Open(_diContainer, Messenger, type, config,
                null, // filePath
                null, // originalNspFilePath
                (p, originalNspFilePath) => app.Open(type, p, originalNspFilePath),
                p => _diContainer.GetInstance<AppProfile>().AppMode = TypeHelper.ToAppMode(app.Project));
        }

        private void ImportPartialMetaFile(App app)
        {
            var message = Messenger.GetResponse(new OpeningFileSelectionMessage(GuiConstants.MessageKey_FileOpen)
            {
                Filter = GuiConstants.ImportableFileDict[ImportableFileType.Meta].DialogFilter
            });

            var filePath = message?.Response?.FirstOrDefault();
            if (string.IsNullOrEmpty(filePath))
                return;

            try
            {
                // インポートによって上書きされる要素をユーザーに通知
                {
                    var changedElementPaths = app.ImportPartialDryRun(filePath)
                        .Select(x => $"* {x}")
                        .ToArray();

                    if (changedElementPaths.IsEmpty())
                    {
                        Messenger.Raise(new DialogMessage(GuiConstants.MessageKey_Dialog)
                        {
                            Icon = TaskDialogStandardIcon.Information,
                            Caption = Properties.Resources.Information,
                            Text = Properties.Resources.ImportPartialMetaPropertyIsEmpty,
                            StandardButtons = DialogMessage.StandardButtonsType.Ok,
                        });
                        return;
                    }

                    var r = Messenger.GetResponse(new DialogMessage(GuiConstants.MessageKey_Dialog)
                    {
                        Icon = TaskDialogStandardIcon.Information,
                        Caption = Properties.Resources.Information,
                        Text = string.Format(Properties.Resources.ImportPartialMetaConfirmation,
                            string.Join(Environment.NewLine, changedElementPaths)),
                        StandardButtons = DialogMessage.StandardButtonsType.OkCancel,
                    });
                    if (r?.Response.DialogResult != DialogMessage.DialogResultType.Ok)
                        return;
                }

                app.ImportPartial(filePath);

                Messenger.Raise(new DialogMessage(GuiConstants.MessageKey_Dialog)
                {
                    Icon = TaskDialogStandardIcon.Information,
                    Caption = Properties.Resources.Information,
                    Text = Properties.Resources.ImportPartialMetaSucceeded,
                    StandardButtons = DialogMessage.StandardButtonsType.Ok,
                });
            }
            catch (Exception e)
            {
                string text;
                if (e is ApplicationIdMismatchedException)
                {
                    var mismatch = (ApplicationIdMismatchedException)e;
                    text = string.Format(Properties.Resources.ImportPartialMetaMismatchApplicationId,
                        mismatch.ExpectedApplicationId, mismatch.ActualApplicationId);
                }
                else
                {
                    text = string.Format(Properties.Resources.ImportPartialMetaFailed, e.Message);
                }

                Messenger.Raise(new DialogMessage(GuiConstants.MessageKey_Dialog)
                {
                    Icon = TaskDialogStandardIcon.Information,
                    Caption = Properties.Resources.Information,
                    Text = text,
                    StandardButtons = DialogMessage.StandardButtonsType.Ok
                });
            }
        }

        private void ShowNspEntries(App app)
        {
            using (var vm = new NspEntriesWindowVm(app.Project))
                Messenger.Raise(new TransitionMessage(vm, "ShowNspEntriesWindow"));
        }

        private async Task<T> DoBuildingNspAsync<T>(App app,
            Func<CancellationToken, Task<T>> doBuild) where T : NspHandleResultType
        {
            app.CancelWorking();
            app.OpenGeneratedMessageTag = nameof(Properties.Resources.OpenGeneratedObjectContainsFolder);
            app.OpenGeneratedObjectAction = OpenFileByExplorer;

            var result = await doBuild(app.WorkingCancellationTokenSource.Token);
            WorkingPanel.ErrorSummary.Value =
                _diContainer.GetInstance<StringHelper>().GetNspHandleResultMessage(result.Result);
            WorkingPanel.ErrorMessages.Value = string.Join(Environment.NewLine, result.ErrorMessages);

            if (app.WorkingState == WorkingState.Canceling)
            {
                app.WorkingState = WorkingState.Canceled;
                return null;
            }
            if (result.Result != NspHandleResult.Ok)
            {
                app.WorkingState = WorkingState.Failed;
                return null;
            }

            return result;
        }

        private void OpenAllComparison(bool isOpened)
        {
            var vm = SelectedPanel.Value as ComparisonPanelVm;
            Debug.Assert(vm != null);

            vm.OpenAll(isOpened);
        }

        private async Task ExtractNspFileAsync(INspFile nspFile)
        {
            var vm = SelectedPanel.Value as ComparisonPanelVm;
            Debug.Assert(vm != null);
            await vm.ExtractFileAsync(nspFile);
        }

        private void SetDisplayOnlyDifferences(bool i)
        {
            var vm = SelectedPanel.Value as ComparisonPanelVm;
            Debug.Assert(vm != null);

            vm.IsDisplayOnlyDifferences.Value = i;
        }

        private void ChangePanel(PanelType panelType)
        {
            SelectedPanel.Value?.Dispose();
            SelectedPanel.Value = null;

            switch (panelType)
            {
                case PanelType.Startup:
                {
                    var panelVm = _diContainer.GetInstance<StartupPanelVm>();
                    panelVm.Parent = this;
                    panelVm.Messenger = Messenger;
                    panelVm.MakeNspPatchCommand = MakeNspPatchCommand;

                    SelectedPanel.Value = panelVm;
                    _diContainer.GetInstance<AppProfile>().AppMode = AppModeType.Startup;
                    break;
                }

                case PanelType.ProjectEdit:
                {
                    var panelVm = _diContainer.GetInstance<ProjectEditPanelVm>();
                    SelectedPanel.Value = panelVm;
                    break;
                }

                case PanelType.Compare:
                {
                    var panelVm = _diContainer.GetInstance<ComparisonPanelVm>();
                    panelVm.Messenger = Messenger;

                    SelectedPanel.Value = panelVm;

                    panelVm.IsDisplayOnlyDifferences.Value = IsDisplayOnlyDifferences.Value;
                    panelVm.Targets.CollectionChangedAsObservable()
                        .Subscribe(_ =>
                        {
                            IsExtractableFileType.Value = panelVm.TargetFileType.IsNsp();
                            ExtractionTarget0.Value = panelVm.Target0;
                            ExtractionTarget1.Value = panelVm.Target1;
                        }).AddTo(panelVm.CompositeDisposable);

                    break;
                }

                default:
                    throw new ArgumentOutOfRangeException(nameof(panelType), panelType, null);
            }
        }

        public void CloseCanceledCallback()
        {
            if (CurrentPanelType == PanelType.ProjectEdit)
                if (ProjectFileHelper.CheckTargetModified(
                        _diContainer.GetInstance<AppProfile>().TargetFileType,
                        Messenger,
                        _app,
                        AppMode.Value,
                        Properties.Resources.OP_CloseApp) == false)
                    return;

            CanClose.Value = true;

            System.Windows.Application.Current.Dispatcher.BeginInvoke(
                (Action) (() => Messenger.Raise(new WindowActionMessage(WindowAction.Close, "WindowAction"))));
        }

        private async Task MakeNspPatch()
        {
            using (var vm = _diContainer.GetInstance<MakeNspPatchWindowVm>())
            {
                await Messenger.RaiseAsync(new TransitionMessage(vm, "ShowMakeNspPatchWindow"));

                if (vm.IsMakePatch == false)
                    return;

                var app = _diContainer.GetInstance<App>();

                app.WorkingFilePath = app.NspPatch.OutputNspPath;

                WorkingPanel.IsOpenGeneratedObject.Value = true;
                app.WorkingKind = WorkingKind.Build;
                app.WorkingState = WorkingState.Working;
                {
                    var r = await DoBuildingNspAsync(
                        app,
                        async token => await app.MakePatchNspAsync(token).ConfigureAwait(false));
                    if (r == null)
                        return;
                }
                app.OpenGeneratedObjectAction = p => OpenNspFileByAuthoringEditor(p, app.NspPatch.OriginalNspPath);
                app.OpenGeneratedMessageTag = nameof(Properties.Resources.OpenGeneratedObjectAuthoringEditor);
                app.WorkingState = WorkingState.Succeeded;
            }
        }

        private void UpdateProjectTargetFilePathObserver()
        {
            _projectObserver?.Dispose();
            _projectObserver = Observable
                .Merge(_app.Project.ObserveProperty(x => x.TargetFilePath).ToUnit())
                .Select(_ => _app.Project.TargetFilePath)
                .Subscribe(path => TargetFilePath.Value = path);
        }

        private static void OpenFileByExplorer(string filePath)
        {
            if (Directory.Exists(filePath))
                WindowsUtility.OpenFolderByExplorer(filePath);
            else
                WindowsUtility.OpenFileByExplorer(filePath);
        }

        private void OpenNspFileByAuthoringEditor(string filePath, string originalNspFilePath)
        {
            ProjectFileHelper.Open(
                _diContainer, Messenger, ImportableFileType.Nsp, _config,
                filePath,
                originalNspFilePath,
                (p, o) => _app.Open(ImportableFileType.Nsp, p, o),
                p =>
                {
                    AppMode.Value = TypeHelper.ToAppMode(_app.Project);
                    CurrentPanelType = PanelType.ProjectEdit;
                });
        }

        private void OpenCommandPrompt()
        {
            var startInfo = new ProcessStartInfo
            {
                FileName = Environment.GetEnvironmentVariable("ComSpec") ?? "cmd.exe",
                WorkingDirectory = _app.Project.ProjectDirectory,
                UseShellExecute = false
            };
            var authoringToolRoot = Path.Combine(NintendoSdkHelper.SdkRootPath,
                @"Tools\CommandLineTools\AuthoringTool");
            startInfo.EnvironmentVariables["PATH"] += $";{authoringToolRoot}";
            Process.Start(startInfo);
        }
    }
}
