﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.IO;
using System.Linq;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.Properties;
using SimpleInjector;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ComparisonPanel.ParamTreeBuilder
{
    public abstract class ParamTreeBuilderBase : IDisposable
    {
        protected Container DiContainer { get; }

        protected Project Project => DiContainer.GetInstance<App>().Project;

        protected ParamTreeBuilderBase(Container diContainer, ObservableCollection<ComparisonPanelVm.Target> targets)
        {
            _Targets = targets;
            DiContainer = diContainer;
        }

        public abstract void Dispose();

        private readonly ObservableCollection<ComparisonPanelVm.Target> _Targets;

        public abstract ObservableCollection<Param> UpdataParams(Config config);

        protected Project Project0 => _Targets[0].Project;
        protected Project Project1 => _Targets[1].Project;
        protected ProgramInfo ProgramInfo0 => Project0.ProgramInfo;
        protected ProgramInfo ProgramInfo1 => Project1.ProgramInfo;

        protected ContentMetaProperty Prop0 => Project0.ContentMetaProperties.Properties.FirstOrDefault(
            x => x.Type == ContentMetaType.Application || x.Type == ContentMetaType.Patch);

        protected ContentMetaProperty Prop1 => Project1.ContentMetaProperties.Properties.FirstOrDefault(
            x => x.Type == ContentMetaType.Application || x.Type == ContentMetaType.Patch);

        protected ApplicationMeta Meta0 => Project0.Meta;
        protected ApplicationMeta Meta1 => Project1.Meta;
        protected Core.Core Core0 => Meta0.Core;
        protected Core.Core Core1 => Meta1.Core;
        protected Application App0 => Meta0.Application;
        protected Application App1 => Meta1.Application;
        protected AocMeta AocMeta0 => Project0.AocMeta;
        protected AocMeta AocMeta1 => Project1.AocMeta;

        protected Content Content0(ContentType type)
        {
            if (Project0.ApplicationContentMeta != null)
                return Project0.ApplicationContentMeta.Contents.FirstOrDefault(x => x.Type == type);

            if (Project0.PatchContentMeta != null)
                return Project0.PatchContentMeta.Contents.FirstOrDefault(x => x.Type == type);

            Debug.Assert(false);
            return null;
        }

        protected Content Content1(ContentType type)
        {
            if (Project1.ApplicationContentMeta != null)
                return Project1.ApplicationContentMeta.Contents.FirstOrDefault(x => x.Type == type);

            if (Project1.PatchContentMeta != null)
                return Project1.PatchContentMeta.Contents.FirstOrDefault(x => x.Type == type);

            Debug.Assert(false);
            return null;
        }

        protected static string TrimSecond(string src)
            => src?.Split(',')[0].Trim();

        protected static string TextFile(string filePath, string onNotFoundValue = null)
        {
            if (string.IsNullOrEmpty(filePath))
                return onNotFoundValue;
            if (File.Exists(filePath) == false)
                return onNotFoundValue;
            return File.ReadAllText(filePath);
        }

        protected static byte[] BinaryFile(string filePath, byte[] onNotFoundValue = null)
        {
            if (string.IsNullOrEmpty(filePath))
                return onNotFoundValue;
            if (File.Exists(filePath) == false)
                return onNotFoundValue;
            return File.ReadAllBytes(filePath);
        }

        protected static string ResString(string tag) => Resources.ResourceManager.GetString(tag, Resources.Culture);
        protected static string EnumString(object v) => ResString($"{v.GetType().Name}_{v}_Caption");

        protected static string Bool_YesNo(bool i) => ResString($"Bool_YesNo_{i}");
        protected static string Bool_Change(bool i) => ResString($"Bool_Change_{i}");
        protected static string Bool_Supported(bool i) => ResString($"Bool_Supported_{i}");
        protected static string Bool_Use(bool i) => ResString($"Bool_Use_{i}");
        protected static string Bool_Allow(bool i) => ResString($"Bool_Allow_{i}");

        protected static string Join(params string[] elements)
        {
            return string.Join(", ", elements.Where(x => string.IsNullOrEmpty(x) == false));
        }

        protected static T Element<T>(IList<T> source, int index) where T : class
            => source?.Count <= index ? null : source?[index];
    }
}
