﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reactive.Disposables;
using System.Runtime;
using System.Windows;
using System.Windows.Media;
using System.Windows.Shell;
using BezelEditor.Foundation;
using BezelEditor.Foundation.Utilities;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.MainWindow;
using Nintendo.Authoring.AuthoringEditor.MainWindow.AppMenu;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Pages;
using Nintendo.Foundation.IO;
using Reactive.Bindings.Extensions;
using SimpleInjector;
using SimpleInjector.Diagnostics;
using Constants = Nintendo.Authoring.AuthoringEditor.Core.Constants;

namespace Nintendo.Authoring.AuthoringEditor
{
    /// <summary>
    /// Entry.xaml の相互作用ロジック
    /// </summary>
    public partial class Entry
    {
        [STAThread]
        public static void Main(string[] args)
        {
            Directory.CreateDirectory(Constants.ConfigDirPath);
            ProfileOptimization.SetProfileRoot(Constants.ConfigDirPath);
            ProfileOptimization.StartProfile("Startup.Profile");

#if DEBUG
            DebugUndisposedTrackerConfiguration.Enable = true;
#endif

            {
                var e = new Entry();
                e.InitializeComponent();
                e.Run();
            }

            DebugUndisposedTracker.CheckUndisposedObject();
        }

        protected override void OnStartup(StartupEventArgs e)
        {
            base.OnStartup(e);

            SetupDiContainer();
            Reactive.Bindings.UIDispatcherScheduler.Initialize();

            _DiContainer.GetInstance<Config>().Initialize();

            CultureService.Instance.Initialize(_DiContainer.GetInstance<Config>());

            SetupThemeColor();

            var mainWindowVm = _DiContainer.GetInstance<MainWindowVm>();
            MainWindow = _DiContainer.GetInstance<MainWindow.MainWindow>();
            MainWindow.DataContext = mainWindowVm;

            // コマンドラインを解釈
            CommandLineParams @params;
            new CommandLineParser().ParseArgs(e.Args, out @params);
            ProcessCommandLine(@params, mainWindowVm);

            // モードが定まっていないようならスタートアップ画面で起動させる
            if (mainWindowVm.CurrentPanelType == MainWindowVm.PanelType.Nothing)
                mainWindowVm.CurrentPanelType = MainWindowVm.PanelType.Startup;

            MainWindow.Show();
            MainWindow.Activate();
        }

        private readonly CompositeDisposable _compositeDisposable = new CompositeDisposable();

        private static readonly IReadOnlyDictionary<AppModeType, Color> _appModeColorDict =
            new Dictionary<AppModeType, Color>
            {
                // https://material.io/guidelines/style/color.html
                {AppModeType.Startup, Color.FromRgb(0x9C, 0x27, 0xB0)},
                {AppModeType.Project, Color.FromRgb(0x9C, 0x27, 0xB0)},
                {AppModeType.ApplicationMeta, Color.FromRgb(0x5C, 0x6B, 0xC0)},
                {AppModeType.ApplicationNsp, Color.FromRgb(0x43, 0xA0, 0x47)},
                {AppModeType.AocMeta, Color.FromRgb(0x00, 0x96, 0x88)},
                {AppModeType.AocNsp, Color.FromRgb(0x8D, 0x6E, 0x63)},
                {AppModeType.PatchNsp, Color.FromRgb(0xF5, 0x7C, 0x00)},
                {AppModeType.Comparison, Color.FromRgb(0x54, 0x6E, 0x7E)},
            };

        private void SetupThemeColor()
        {
            _DiContainer.GetInstance<AppProfile>().ObserveProperty(x => x.AppMode)
                .Subscribe(mode =>
                {
                    Debug.Assert(_appModeColorDict.ContainsKey(mode));

                    var color = _appModeColorDict[mode];
                    var pressedColor = Color.FromRgb((byte) (color.R*0.7), (byte) (color.G*0.7), (byte) (color.B*0.7));

                    var primaryBrush = new SolidColorBrush(color);
                    primaryBrush.Freeze();

                    var lr = (byte) ((0xFF - color.R)*0.9 + color.R);
                    var lg = (byte) ((0xFF - color.G)*0.9 + color.G);
                    var lb = (byte) ((0xFF - color.B)*0.9 + color.B);
                    var lightPrimaryBrush = new SolidColorBrush(Color.FromRgb(lr, lg, lb));
                    lightPrimaryBrush.Freeze();

                    var pressedBrush = new SolidColorBrush(pressedColor);
                    pressedBrush.Freeze();

                    Current.Resources["PrimaryColor"] = color;
                    Current.Resources["PrimaryBrush"] = primaryBrush;
                    Current.Resources["LightPrimaryBrush"] = lightPrimaryBrush;
                    Current.Resources["Button.Pressed.Background"] = pressedBrush;
                }).AddTo(_compositeDisposable);
        }

        protected override void OnExit(ExitEventArgs e)
        {
            base.OnExit(e);

            UpdateJumpList(_DiContainer.GetInstance<Config>());

            _compositeDisposable.Dispose();
            CultureService.Instance.Dispose();
            _DiContainer.GetInstance<Config>().Destory(Constants.ConfigFilePath);
            _DiContainer.Dispose();
        }

        private readonly Container _DiContainer = new Container();

        private void SetupDiContainer()
        {
            //_DiContainer.Register(() => _DiContainer);
            _DiContainer.RegisterSingleton(() => _DiContainer);
            _DiContainer.RegisterSingleton<ApplicationCapability>();
            _DiContainer.RegisterSingleton<App>();
            _DiContainer.RegisterSingleton<AppProfile>();
            _DiContainer.RegisterSingleton<PageValidations>();
            _DiContainer.RegisterSingleton(() => YamlHelper.Load(Constants.ConfigFilePath, () => new Config()));
            _DiContainer.RegisterSingleton<MainWindowVm>();

            _DiContainer.Register(() => _DiContainer.GetInstance<App>().Project.Meta);
            _DiContainer.Register(() => _DiContainer.GetInstance<App>().Project);

            /////////////////////////////////////////////////////////////////////////////
            _DiContainer.GetRegistration(typeof(App)).Registration
                .SuppressDiagnosticWarning(DiagnosticType.DisposableTransientComponent, "suppress");

            _DiContainer.GetRegistration(typeof(AppProfile)).Registration
                .SuppressDiagnosticWarning(DiagnosticType.DisposableTransientComponent, "suppress");

            _DiContainer.GetRegistration(typeof(MainWindowVm)).Registration
                .SuppressDiagnosticWarning(DiagnosticType.DisposableTransientComponent, "suppress");

            _DiContainer.GetRegistration(typeof(Container)).Registration
                .SuppressDiagnosticWarning(DiagnosticType.DisposableTransientComponent, "suppress");

            _DiContainer.GetRegistration(typeof(ApplicationMeta)).Registration
                .SuppressDiagnosticWarning(DiagnosticType.DisposableTransientComponent, "suppress");

            _DiContainer.GetRegistration(typeof(Project)).Registration
                .SuppressDiagnosticWarning(DiagnosticType.DisposableTransientComponent, "suppress");

            _DiContainer.GetRegistration(typeof(AppMenuVm)).Registration
                .SuppressDiagnosticWarning(DiagnosticType.DisposableTransientComponent, "suppress");

#if DEBUG
            //_DiContainer.Verify();
#endif
        }

        private static void UpdateJumpList(Config config)
        {
            var jumpList = new JumpList();

            var allJumpTasks =
                MakeJumpTask(AuthoringEditor.Properties.Resources.MruProject, "project", config.MruProjects)
                    .Concat(MakeJumpTask(AuthoringEditor.Properties.Resources.MruMeta, "meta", config.MruMetas))
                    .Concat(MakeJumpTask(AuthoringEditor.Properties.Resources.MruNsp, "nsp", config.MruNsps));

            foreach (var t in allJumpTasks)
                jumpList.JumpItems.Add(t);

            jumpList.Apply();
        }

        private static IEnumerable<JumpTask> MakeJumpTask(string category, string option,
            ObservableCollection<MruItem> mruItems)
        {
            return mruItems.Select(i => new JumpTask
            {
                CustomCategory = category,
                Title = Path.GetFileName(i.FilePath),
                Arguments = $"--{option} {i.FilePath.ToPathString().SurroundDoubleQuotes}"
            });
        }

        private void ProcessCommandLine(CommandLineParams @params, MainWindowVm mainWindowVm)
        {
            if (@params == null)
                return;

            var type = ImportableFileType.Project;
            string sourceFilePath = null;
            string originalNspFilePath = null;
            {
                if (string.IsNullOrEmpty(@params.ProjectPath) == false)
                {
                    type = ImportableFileType.Project;
                    sourceFilePath = @params.ProjectPath;
                }
                else if (string.IsNullOrEmpty(@params.NintendoMetaPath) == false)
                {
                    type = ImportableFileType.Meta;
                    sourceFilePath = @params.NintendoMetaPath;
                }
                else if (string.IsNullOrEmpty(@params.NspPath) == false)
                {
                    type = ImportableFileType.Nsp;
                    sourceFilePath = @params.NspPath;
                }
                else if (string.IsNullOrEmpty(@params.InputFilePath) == false)
                {
                    var fileType = TypeHelper.FilePathToImportableFileType(@params.InputFilePath);
                    if (fileType != null)
                    {
                        type = fileType.Value;
                        sourceFilePath = @params.InputFilePath;
                    }
                }
            }

            if (string.IsNullOrEmpty(@params.InputOriginalNspFilePath) == false &&
                TypeHelper.FilePathToImportableFileType(@params.InputOriginalNspFilePath) == ImportableFileType.Nsp)
            {
                originalNspFilePath = @params.InputOriginalNspFilePath;
            }

            if (string.IsNullOrEmpty(sourceFilePath))
                return;

            ProjectFileHelper.Open(
                _DiContainer,
                mainWindowVm.Messenger,
                type,
                _DiContainer.GetInstance<Config>(),
                sourceFilePath,
                originalNspFilePath, // originalNspFilePath
                (p, origNspFilePath) =>
                {
                    _DiContainer.GetInstance<App>().Open(type, p, origNspFilePath);
                },
                p =>
                {
                    var project = _DiContainer.GetInstance<App>().Project;
                    _DiContainer.GetInstance<AppProfile>().AppMode = TypeHelper.ToAppMode(project);
                    mainWindowVm.CurrentPanelType = MainWindowVm.PanelType.ProjectEdit;
                }
            );
        }
    }
}
