﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Reactive.Disposables;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Input;

namespace Nintendo.Authoring.AuthoringEditor.Controls
{
    public class NumberEditBox : TextBox
    {
        #region Maximum

        public decimal Maximum
        {
            get { return (decimal) GetValue(MaximumProperty); }
            set { SetValue(MaximumProperty, value); }
        }

        public static readonly DependencyProperty MaximumProperty =
            DependencyProperty.Register(
                nameof(Maximum),
                typeof(decimal),
                typeof(NumberEditBox),
                new FrameworkPropertyMetadata
                {
                    DefaultValue = default(decimal),
                    BindsTwoWayByDefault = true
                }
            );

        #endregion

        #region Value

        public decimal Value
        {
            get { return (decimal) GetValue(ValueProperty); }
            set { SetValue(ValueProperty, value); }
        }

        public static readonly DependencyProperty ValueProperty =
            DependencyProperty.Register(
                nameof(Value),
                typeof(decimal),
                typeof(NumberEditBox),
                new FrameworkPropertyMetadata
                {
                    PropertyChangedCallback = OnValueChanged,
                    DefaultValue = default(decimal),
                    BindsTwoWayByDefault = true
                }
            );

        private static void OnValueChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            Debug.Assert(d is NumberEditBox);
            var self = (NumberEditBox) d;

            self.UpdateText();
        }

        #endregion

        #region Unit

        public int Unit
        {
            get { return (int) GetValue(UnitProperty); }
            set { SetValue(UnitProperty, value); }
        }

        public static readonly DependencyProperty UnitProperty =
            DependencyProperty.Register(
                nameof(Unit),
                typeof(int),
                typeof(NumberEditBox),
                new FrameworkPropertyMetadata
                {
                    DefaultValue = 1,
                    BindsTwoWayByDefault = true
                }
            );

        #endregion

        public NumberEditBox()
        {
            InputMethod.SetIsInputMethodEnabled(this, false);

            UpdateText();

            TextChanged += OnTextChanged;
            PreviewKeyDown += OnPreviewKeyDown;

            DataObject.AddPastingHandler(this, OnPaste);
        }

        private void OnPaste(object sender, DataObjectPastingEventArgs e)
        {
            var isText = e.SourceDataObject.GetDataPresent(DataFormats.UnicodeText, true);
            if (isText == false)
                return;

            var inputText = e.SourceDataObject.GetData(DataFormats.UnicodeText) as string;
            if (inputText == null)
                return;

            inputText = inputText.Trim();

            {
                var start = SelectionStart;
                var length = SelectionLength;
                var caret = CaretIndex;

                var text = Text.Substring(0, start);
                text += Text.Substring(start + length);

                var newText = text.Substring(0, CaretIndex) + inputText;
                newText += text.Substring(caret);

                if (IsNumber(newText))
                    Text = newText;
            }

            e.CancelCommand();
        }

        private void OnPreviewKeyDown(object sender, KeyEventArgs e)
        {
            if ((e.Key >= Key.D0 && e.Key <= Key.D9) ||
                (e.Key >= Key.NumPad0 && e.Key <= Key.NumPad9) ||
                (e.Key == Key.Left) ||
                (e.Key == Key.Right) ||
                (e.Key == Key.Delete) ||
                (e.Key == Key.Back) ||
                (e.Key == Key.Tab) ||
                (e.Key == Key.Home) ||
                (e.Key == Key.End))
            {
                e.Handled = false;
                return;
            }

            if (Unit != 1)
                if (e.Key == Key.OemPeriod)
                {
                    e.Handled = false;
                    return;
                }

            if ((Keyboard.Modifiers & ModifierKeys.Control) != ModifierKeys.None)
                if (e.Key == Key.C ||
                    e.Key == Key.V ||
                    e.Key == Key.X ||
                    e.Key == Key.Z ||
                    e.Key == Key.Y ||
                    e.Key == Key.A)
                {
                    e.Handled = false;
                    return;
                }

            e.Handled = true;
        }

        private bool _isChangingText;
        private bool _isUpdatingText;

        private void OnTextChanged(object sender, TextChangedEventArgs _)
        {
            if (_isUpdatingText)
                return;

            using (Disposable.Create(() => _isChangingText = false))
            {
                _isChangingText = true;

                decimal v;
                if (decimal.TryParse(Text, out v) == false)
                    return;

                v *= Unit;

                if (v > Maximum)
                {
                    v = Maximum;
                    Text = ToValueString(v);
                }

                Value = v;
            }
        }

        private void UpdateText()
        {
            if (_isChangingText)
                return;

            using (Disposable.Create(() => _isUpdatingText = false))
            {
                _isUpdatingText = true;
                Text = ToValueString(Value);
            }
        }

        private string ToValueString(decimal value)
        {
            return (value / Unit).ToString("0.###");
        }

        private bool IsNumber(string s)
        {
            try
            {
                // ReSharper disable ReturnValueOfPureMethodIsNotUsed
                if (Unit == 1)
                    ulong.Parse(s);
                else
                    decimal.Parse(s);
                // ReSharper restore ReturnValueOfPureMethodIsNotUsed

                return true;
            }
            catch
            {
                return false;
            }
        }
    }
}
