﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Threading;
using System.Windows;
using System.Windows.Interop;
using BezelEditor.Foundation.Utilities;
using Nintendo.Authoring.AuthoringEditor.Properties;

namespace Nintendo.Authoring.AuthoringEditor.Controls
{
    public class InPreparationBlock : IDisposable
    {
        private Thread _thread;
        private InPreparationWindow _window;

        private string _messagePool;

        private readonly Semaphore _windowCreateSemaphore = new Semaphore(1, 1);
        private readonly Semaphore _windowCloseSemaphore = new Semaphore(1, 1);

        private static int _depth;

        public InPreparationBlock(Window targetWindow = null, string message = null)
        {
            if (Application.Current == null)
                return;

            if (_depth == 0)
            {
                if (targetWindow == null)
                    targetWindow = Application.Current.MainWindow;

                var handle = new WindowInteropHelper(targetWindow).Handle;

                var rect = new Win32.Rect();
                Win32.GetWindowRect(handle, ref rect);

                var dpi = targetWindow.GetDpiScaleFactor();

                var left = rect.Left / dpi.X;
                var top = rect.Top / dpi.Y;
                var width = rect.Right / dpi.X - left;
                var height = rect.Bottom / dpi.Y - top;

                Start(message ?? Resources.InPreparation, left, top, width, height);
            }

            ++_depth;
        }

        public void Dispose()
        {
            if (Application.Current == null)
                return;

            --_depth;

            if (_depth == 0)
            {
                WpfUtility.DoEvents();
                Stop();
            }
        }

        public string Message
        {
            get { return _window != null ? _window.Message : string.Empty; }

            set
            {
                if (_window != null)
                    _window.Message = value;
                else
                    _messagePool = value;
            }
        }

        public void Start(string message, double left, double top, double width, double height)
        {
            Stop();

            _windowCreateSemaphore.WaitOne();
            _thread = new Thread(() =>
            {
                _window = new InPreparationWindow(message)
                {
                    Left = left,
                    Top = top,
                    Width = width,
                    Height = height
                };

                _window.ContentRendered += (_, __) => _windowCreateSemaphore.Release();

                if (_messagePool != null)
                {
                    _window.Message = _messagePool;
                    _messagePool = null;
                }

                _window.Dispatcher.Invoke(() => _window.ShowDialog());
            });

            _thread.SetApartmentState(ApartmentState.STA);
            _thread.IsBackground = true;
            _thread.Priority = ThreadPriority.Lowest;

            _thread.Start();
        }

        public void Stop()
        {
            if (_thread == null)
                return;

            _windowCreateSemaphore.WaitOne();

            if (_window != null)
            {
                _window.Dispatcher.Invoke(() =>
                {
                    _window.Hide();

                    var isHidden = false;

                    _window.HideComplited += (_, __) =>
                    {
                        _windowCloseSemaphore.WaitOne();
                        _window.Closed += (sender, args) => _windowCloseSemaphore.Release();
                        _window.Close();

                        isHidden = true;
                    };

                    while (isHidden == false)
                        WpfUtility.DoEvents();
                });

                _windowCloseSemaphore.WaitOne();
                _window = null;
                _windowCloseSemaphore.Release();
            }

            _thread.Abort();
            _thread.Join();
            _thread = null;

            _windowCreateSemaphore.Release();
        }
    }
}
