﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using System.Windows;
using System.Windows.Controls;
using BezelEditor.Foundation.Utilities;
using Livet.Commands;
using Livet.Messaging.IO;
using Microsoft.WindowsAPICodePack.Dialogs;

namespace Nintendo.Authoring.AuthoringEditor.Controls
{
    public class DirPathEditBoxSlim : Control
    {
        static DirPathEditBoxSlim()
        {
            DefaultStyleKeyProperty.OverrideMetadata(typeof(DirPathEditBoxSlim),
                new FrameworkPropertyMetadata(typeof(DirPathEditBoxSlim)));
        }

        #region Path

        public string Path
        {
            get { return (string) GetValue(PathProperty); }
            set { SetValue(PathProperty, value); }
        }

        public static readonly DependencyProperty PathProperty =
            DependencyProperty.Register(
                "Path",
                typeof(string),
                typeof(DirPathEditBoxSlim),
                new FrameworkPropertyMetadata()
                {
                    BindsTwoWayByDefault = true,
                    DefaultValue = default(string)
                }
            );

        #endregion

        public string Title
        {
            get { return (string) GetValue(TitleProperty); }
            set { SetValue(TitleProperty, value); }
        }

        public static readonly DependencyProperty TitleProperty =
            DependencyProperty.Register(
                "Title",
                typeof(string),
                typeof(DirPathEditBoxSlim),
                new FrameworkPropertyMetadata
                {
                    BindsTwoWayByDefault = true,
                    DefaultValue = default(string)
                }
            );

        public string Filter
        {
            get { return (string) GetValue(FilterProperty); }
            set { SetValue(FilterProperty, value); }
        }

        public static readonly DependencyProperty FilterProperty =
            DependencyProperty.Register(
                "Filter",
                typeof(string),
                typeof(DirPathEditBoxSlim),
                new FrameworkPropertyMetadata
                {
                    BindsTwoWayByDefault = true,
                    DefaultValue = default(string)
                }
            );

        public string InitialDirectory
        {
            get { return (string) GetValue(InitialDirectoryProperty); }
            set { SetValue(InitialDirectoryProperty, value); }
        }

        public static readonly DependencyProperty InitialDirectoryProperty =
            DependencyProperty.Register(
                "InitialDirectory",
                typeof(string),
                typeof(DirPathEditBoxSlim),
                new FrameworkPropertyMetadata
                {
                    BindsTwoWayByDefault = true,
                    DefaultValue = default(string)
                }
            );

        #region IsReadOnly

        public bool IsReadOnly
        {
            get { return (bool) GetValue(IsReadOnlyProperty); }
            set { SetValue(IsReadOnlyProperty, value); }
        }

        public static readonly DependencyProperty IsReadOnlyProperty =
            DependencyProperty.Register(
                nameof(IsReadOnly),
                typeof(bool),
                typeof(DirPathEditBoxSlim),
                new FrameworkPropertyMetadata
                {
                    BindsTwoWayByDefault = true,
                    DefaultValue = default(bool)
                }
            );

        #endregion

        private ListenerCommand<FileSelectionMessage> _OpenFileCommand;

        public ListenerCommand<FileSelectionMessage> OpenFileCommand
            => _OpenFileCommand ?? (_OpenFileCommand = new ListenerCommand<FileSelectionMessage>(OpenFile, CanOpenFile))
            ;

        public void OpenFile(FileSelectionMessage m)
        {
            var initDir = string.IsNullOrEmpty(Path)
                ? InitialDirectory
                : Environment.ExpandEnvironmentVariables(Path);
            using (var dialog = new CommonOpenFileDialog
            {
                Title = Title,
                IsFolderPicker = true,
                InitialDirectory = initDir
            })
            {
                var result = dialog.ShowDialog();
                if (result == CommonFileDialogResult.Ok)
                    Path = dialog.FileName;
            }
        }

        public static bool CanOpenFile()
        {
            return true;
        }

        private void UserControl_Drop(object sender, DragEventArgs e)
        {
            var files = e.Data.GetData(DataFormats.FileDrop) as string[];

            var path = files?.FirstOrDefault(Directory.Exists);
            if (path != null)
                Path = path;
        }

        private static void UserControl_PreviewDragOver(object sender, DragEventArgs e)
        {
            e.Effects = DragDropEffects.None;

            if (e.Data.GetDataPresent(DataFormats.FileDrop, true))
            {
                var files = e.Data.GetData(DataFormats.FileDrop) as string[];
                if (files != null)
                {
                    if (files.Any(Directory.Exists))
                        e.Effects = DragDropEffects.Copy;
                }
            }

            e.Handled = true;
        }

        public override void OnApplyTemplate()
        {
            base.OnApplyTemplate();

            var basePanel = this.FindChild<Grid>(null);

            basePanel.Drop += UserControl_Drop;
            basePanel.PreviewDragOver += UserControl_PreviewDragOver;
        }
    }
}
