﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Linq;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Interactivity;
using System.Windows.Media;
using System.Windows.Shapes;
using BezelEditor.Control;
using BezelEditor.Foundation.Extentions;
using Fluent;
using Nintendo.Authoring.AuthoringEditor.Core;

namespace Nintendo.Authoring.AuthoringEditor.Behaviors
{
    public class ApplicationModeSpecificTitleBarForegroundBehavior : Behavior<RibbonWindow>
    {
        public AppModeType ApplicationMode
        {
            get { return (AppModeType)GetValue(ApplicationModeProperty); }
            set { SetValue(ApplicationModeProperty, value); }
        }

        public static readonly DependencyProperty ApplicationModeProperty =
            DependencyProperty.Register(
                nameof(ApplicationMode),
                typeof(AppModeType),
                typeof(ApplicationModeSpecificTitleBarForegroundBehavior),
                new FrameworkPropertyMetadata
                {
                    PropertyChangedCallback = PropertyChangedCallback,
                    BindsTwoWayByDefault = false,
                });

        protected override void OnAttached()
        {
            base.OnAttached();
            if (AssociatedObject == null)
                return;
            AssociatedObject.Loaded += AssociatedObjectOnLoaded;
        }

        protected override void OnDetaching()
        {
            if (AssociatedObject != null)
            {
                AssociatedObject.Loaded -= AssociatedObjectOnLoaded;
            }
            base.OnDetaching();
        }

        private void AssociatedObjectOnLoaded(object sender, RoutedEventArgs e)
        {
            ForceChangeTitleBarForeground(ApplicationMode, AssociatedObject);
        }

        private static void PropertyChangedCallback(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            var behavior = d as ApplicationModeSpecificTitleBarForegroundBehavior;
            var element = behavior?.AssociatedObject;
            if (element == null)
                return;
            if (e.NewValue == null)
                return;
            ForceChangeTitleBarForeground((AppModeType)e.NewValue, element);
        }

        private static void ForceChangeTitleBarForeground(AppModeType mode, RibbonWindow element)
        {
            var brush = MakeTitleBrush(mode);

            var title = element.FindVisualChild("PART_Title") as TextBlock;
            if (title != null)
                title.Foreground = brush;

            var winCmds = element.FindVisualChild("PART_WindowCommands");
            if (winCmds != null)
                foreach (var p in winCmds.Descendants<Path>())
                    p.Fill = brush;
        }

        private static Brush MakeTitleBrush(AppModeType mode)
        {
            var attrs = mode.GetType()
                .GetField(mode.ToString())
                .GetCustomAttributes(false)
                .OfType<SpecificTitleBarAttribute>();

            var attr = attrs.FirstOrDefault();
            if (attr == null)
                return Brushes.White;

            return attr.IsDark ? Brushes.Black : Brushes.White;
        }
    }
}
