﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using System.Security.Cryptography.X509Certificates;
using System.Text;
using System.Threading.Tasks;
using Nintendo.Foundation.IO;

namespace Nintendo.Authoring.AuthoringEditor.Helper
{
    public class Program
    {
        static int Main(string[] args)
        {
            AppDomain.CurrentDomain.UnhandledException += UnhandledException;

            CommandLineParams @params = null;
            try
            {
                if (new CommandLineParser().ParseArgs(args, out @params) == false)
                    return -1;
            }
            catch (ArgumentException)
            {
                return -1;
            }

            if (@params.CompareOption != null)
                return (int)IsIdenticalNca(@params.CompareOption);
            if (@params.ExtractOption != null)
                return Extract(@params.ExtractOption);

            return -1;
        }

        private static NcaCompareResult IsIdenticalNca(CompareCommandParams option)
        {
            Func<string, bool> fileFilter;
            if (string.IsNullOrEmpty(option.NcaPrefixPath))
                fileFilter = x => true;
            else
                fileFilter = x => x.StartsWith(option.NcaPrefixPath);

            using (var source = new NspReader(option.SourceNspPath, option.OriginalNspPath))
            using (var target = new NspReader(option.TargetNspPath, option.OriginalNspPath))
            {
                var sourceNcaReader = string.IsNullOrEmpty(option.SourceNcaFileName)
                    ? source.GetNcaReader(option.ContentType)
                    : source.GetNcaReader(option.SourceNcaFileName);

                var targetNcaReader = string.IsNullOrEmpty(option.TargetNcaFileName)
                    ? target.GetNcaReader(option.ContentType)
                    : target.GetNcaReader(option.TargetNcaFileName);

                // 両方とも存在しない
                if (sourceNcaReader == null && targetNcaReader == null)
                    return NcaCompareResult.Identical;

                // 一方だけ存在する
                {
                    var flags = NcaCompareResult.ContainsDifference;

                    if (sourceNcaReader != null)
                    {
                        flags |= NcaCompareResult.SourceNcaExists;
                        if (sourceNcaReader.IsContainsFile(fileFilter))
                            flags |= NcaCompareResult.SourceFileExists;
                    }
                    if (targetNcaReader != null)
                    {
                        flags |= NcaCompareResult.TargetNcaExists;
                        if (targetNcaReader.IsContainsFile(fileFilter))
                            flags |= NcaCompareResult.TargetFileExists;
                    }

                    if (sourceNcaReader == null || targetNcaReader == null)
                        return flags;
                }

                // コンソールへの進捗表示
                Task.Run(async () =>
                {
                    while (true)
                    {
                        var totalBytes = sourceNcaReader.CompareTotalBytes;
                        if (totalBytes > 0)
                            Console.WriteLine($"{sourceNcaReader.CompareReadBytes}/{totalBytes}");
                        await Task.Delay(TimeSpan.FromSeconds(1.0)).ConfigureAwait(false);
                    }
                });

                // 両方が存在する
                return sourceNcaReader.Compare(targetNcaReader, fileFilter);
            }
        }

        private static int Extract(ExtractCommandParams option)
        {
            using (var reader = new NspReader(option.NspPath, option.OriginalNspPath))
            {
                if (string.IsNullOrEmpty(option.ExtractTargetFile) == false)
                {
                    if (option.IsStandardOutput)
                        ExtractSingleFileToStandardOutput(reader, option);
                    else
                        ExtractSingleFile(reader, option);
                }
                else if (string.IsNullOrEmpty(option.ExtractFileListPath) == false)
                    ExtractFiles(reader, option);
                else if (option.ExtractContentType != ContentType.None)
                    ExtractSpecificContentTypeNca(reader, option);
                else
                    ExtractAllFiles(reader, option);
            }
            return 0;
        }

        private static void ExtractSingleFileToStandardOutput(NspReader reader, ExtractCommandParams option)
        {
            var encodingName = option.StandardOutputEncoding ?? Encoding.Default.HeaderName;
            Console.OutputEncoding = Encoding.GetEncoding(encodingName);

            var extractFile = reader.GetExtractFile(x => x == option.ExtractTargetFile ? x : null);
            if (extractFile == null)
                throw new FileNotFoundException();

            Console.Out.Write(extractFile.ReadAllText(Console.OutputEncoding));
        }

        private static void ExtractSingleFile(NspReader reader, ExtractCommandParams option)
        {
            var extractFile = reader.GetExtractFile(x =>
            {
                if (x != option.ExtractTargetFile)
                    return null;
                if (string.IsNullOrEmpty(option.OutputPath) == false)
                    return option.OutputPath;
                return x;
            });

            if (extractFile == null)
                throw new FileNotFoundException();

            var outputDirectory = option.OutputDirectoryPath ?? Directory.GetCurrentDirectory();
            var outputPath = extractFile.WriteFile(outputDirectory);
            Console.WriteLine($"1/1\t{extractFile.FullName}\t{outputPath}");
        }

        private static void ExtractFiles(NspReader reader, ExtractCommandParams option)
        {
            var outputDirectory = option.OutputDirectoryPath ?? Directory.GetCurrentDirectory();
            var extractFileInfos = reader.GetExtractFilesFromListFile(option.ExtractFileListPath).ToArray();
            ExtractInternal(extractFileInfos, outputDirectory);
        }

        private static void ExtractSpecificContentTypeNca(NspReader reader, ExtractCommandParams option)
        {
            var outputDirectory = option.OutputDirectoryPath ?? Directory.GetCurrentDirectory();
            var extractFileInfos = reader.GetExtractFiles(x => x, option.ExtractContentType).ToArray();
            ExtractInternal(extractFileInfos, outputDirectory);
        }

        private static void ExtractAllFiles(NspReader reader, ExtractCommandParams option)
        {
            var outputDirectory = option.OutputDirectoryPath ?? Directory.GetCurrentDirectory();
            var extractFileInfos = reader.GetExtractFiles(x => x).ToArray();
            ExtractInternal(extractFileInfos, outputDirectory);
        }

        private static void ExtractInternal(ExtractFileInfo[] extractFileInfos, string outputDirectory)
        {
            var total = extractFileInfos.Length;
            var written = 0;
            foreach (var e in extractFileInfos)
            {
                var outputPath = e.WriteFile(outputDirectory);
                written++;
                Console.WriteLine($"{written}/{total}\t{e.FullName}\t{outputPath}");
            }
        }

        private static void UnhandledException(object sender, UnhandledExceptionEventArgs e)
        {
            var ex = e.ExceptionObject as Exception;
            if (ex == null)
            {
                Console.Error.WriteLine(e.ToString());
                Environment.Exit(-1);
            }

            var aex = ex as AggregateException;
            if (aex != null)
            {
                foreach (var ee in aex.InnerExceptions)
                {
                    Console.Error.WriteLine(ee.ToString());
                    Console.Error.WriteLine(ee.StackTrace);
                    Console.Error.WriteLine("----");
                }
                Environment.Exit(-1);
            }

            Console.Error.WriteLine(ex.ToString());
            Console.Error.WriteLine(ex.StackTrace);
            Environment.Exit(-1);
        }
    }
}
