﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Xml.Linq;
using Nintendo.Authoring.AuthoringLibrary;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringEditor.Helper
{
    public class NspReader : IDisposable
    {
        public NintendoSubmissionPackageReader Reader { get; }
        public AuthoringConfiguration Configuration { get; } = new AuthoringConfiguration();

        public NspReader(string nspPath, string originalPath = null)
        {
            Reader = originalPath == null
                ? new NintendoSubmissionPackageReader(nspPath)
                : new PatchedNintendoSubmissionPackageReader(nspPath, new NintendoSubmissionPackageReader(originalPath));
        }

        public NcaReader GetNcaReader(string ncaFileName)
        {
            var keyConfig = Configuration.GetKeyConfiguration();
            var key = new NcaKeyGenerator(keyConfig);

            NintendoContentArchiveReader ncaReader;
            NintendoContentArchiveReader originalNcaReader = null;

            var patchedReader = Reader as PatchedNintendoSubmissionPackageReader;
            if (patchedReader != null)
            {
                var readers = patchedReader.OpenPatchableNintendoContentArchiveReader(ncaFileName, key);
                ncaReader = readers.Item1;
                originalNcaReader = readers.Item2;
            }
            else
            {
                ncaReader = Reader.OpenNintendoContentArchiveReader(ncaFileName, key);
            }
            TicketUtility.SetExternalKey(ref ncaReader, Reader);

            return new NcaReader(ncaFileName, ncaReader, originalNcaReader);
        }

        // TORIAEZU: 実際には 同一 ContentType を持つ nca が複数存在するケースがあるので、後ほど IEnuemrable<NcaReader> に修正する
        public NcaReader GetNcaReader(ContentType contentType)
        {
            var xmlText = ExtractAllTextFromRoot(x => x.EndsWith(".cnmt.xml"));
            var contentMetaXml = XDocument.Parse(xmlText);
            var contentElement = contentMetaXml.Root?
                .Elements("Content")
                .FirstOrDefault(x => x.Element("Type")?.Value == contentType.ToString());
            var contentId = contentElement?.Element("Id")?.Value;
            if (string.IsNullOrEmpty(contentId))
                return null;
            return GetNcaReader($"{contentId}.nca");
        }

        // 全ファイルを extract する可能性があるのでルートの .nca は単独で抽出しない
        public IEnumerable<ExtractFileInfo> GetExtractFiles(Func<string, string> matcher) =>
            GetExtractFilesInternal(matcher, x => Path.GetExtension(x.Item1) != ".nca");

        public IEnumerable<ExtractFileInfo> GetExtractFiles(Func<string, string> matcher, ContentType contentType)
        {
            using (var reader = GetNcaReader(contentType))
            {
                var extractFileInfos = GetExtractNcaFiles(reader, matcher);
                foreach (var e in extractFileInfos)
                    yield return e;
            }
        }

        public IEnumerable<ExtractFileInfo> GetExtractFiles(IDictionary<string, string> targets)
            => GetExtractFiles(x => GetDictionaryValue(targets, x));

        // ルートの nca も単独で抽出が可能
        public ExtractFileInfo GetExtractFile(Func<string, string> matcher)
            => GetExtractFilesInternal(matcher, x => true).FirstOrDefault();

        public IEnumerable<ExtractFileInfo> GetExtractFilesFromListFile(string listFilePath)
        {
            var targets = File.ReadAllLines(listFilePath)
                .Where(x => string.IsNullOrEmpty(x) == false)
                .Select(x => x.Split(new[] { '\t' }, 2, StringSplitOptions.None))
                .Where(x => x.Length == 2 && x.All(y => string.IsNullOrEmpty(y) == false))
                .ToDictionary(x => x[0], y => y[1]);

            return GetExtractFiles(targets);
        }

        public string ExtractAllTextFromRoot(Func<string, bool> matcher, Encoding encoding)
        {
            var extractFile = Reader.ListFileInfo().FirstOrDefault(x => matcher(x.Item1));
            if (extractFile == null)
                return null;
            using (var writer = new MemoryStream())
            {
                FsUtil.WriteFile(writer, Reader, extractFile.Item1, extractFile.Item2);
                writer.Flush();
                writer.Position = 0;
                return new StreamReader(writer, encoding).ReadToEnd();
            }
        }

        public string ExtractAllTextFromRoot(Func<string, bool> matcher) => ExtractAllTextFromRoot(matcher, Encoding.UTF8);

        private IEnumerable<ExtractFileInfo> GetExtractFilesInternal(Func<string, string> matcher, Func<Tuple<string, long>, bool> rootFileSelector)
        {
            var fileInfos = Reader.ListFileInfo();

            foreach (var file in fileInfos.Where(rootFileSelector))
            {
                var extractedFile = GetExtractRootFile(file, matcher);
                if (extractedFile != null)
                    yield return extractedFile;
            }

            var ncaFileInfos = fileInfos.Where(x => Path.GetExtension(x.Item1) == ".nca");
            foreach (var file in ncaFileInfos)
            {
                var fileName = file.Item1;
                var reader = GetNcaReaderNoThrows(fileName);
                if (reader == null)
                    continue;
                using (reader)
                {
                    var extactFileInfos = GetExtractNcaFiles(reader, matcher);
                    foreach (var e in extactFileInfos)
                        yield return e;
                }
            }
        }

        private NcaReader GetNcaReaderNoThrows(string fileName)
        {
            try
            {
                return GetNcaReader(fileName);
            }
            catch
            {
                return null;
            }
        }

        private ExtractFileInfo GetExtractRootFile(Tuple<string, long> file, Func<string, string> matcher)
        {
            var fileName = file.Item1;

            var outputPath = matcher(fileName);
            if (outputPath == null)
                return null;

            return new ExtractFileInfo
            {
                Reader = Reader,
                FileSize = file.Item2,
                Name = fileName,
                FullName = fileName,
                OutputPath = outputPath
            };
        }

        private static IEnumerable<ExtractFileInfo> GetExtractNcaFiles(NcaReader reader, Func<string, string> matcher)
        {
            foreach (var fileInfo in reader.GetFileInfoList())
            {
                var filePrefix = $"{reader.FileName}/fs{fileInfo.Index}/";
                var targetFiles = fileInfo.Files
                    .Select(x =>
                    {
                        var fullName = $"{filePrefix}{x.Item1}";
                        var outputPath = matcher(fullName);
                        if (outputPath == null)
                            return null;
                        return new ExtractFileInfo
                        {
                            Name = x.Item1,
                            FileSize = x.Item2,
                            FullName = fullName,
                            OutputPath = outputPath,
                            Reader = fileInfo.Reader
                        };
                    })
                    .Where(x => x != null);

                foreach (var e in targetFiles)
                    yield return e;
            }
        }

        private static TValue GetDictionaryValue<TKey, TValue>(IDictionary<TKey, TValue> dict, TKey key, TValue defaultValue = default(TValue))
        {
            TValue value;
            if (dict.TryGetValue(key, out value))
                return value;
            return defaultValue;
        }

        public void Dispose()
        {
            Reader?.Dispose();
        }
    }
}
