﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Threading;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringEditor.Helper
{
    public class NcaReader : IEquatable<NcaReader>, IDisposable
    {
        public string FileName { get; set; }
        public NintendoContentArchiveReader Reader { get; }
        public NintendoContentArchiveReader OriginalReader { get; }

        private long _compareReadBytes;
        public long CompareReadBytes => _compareReadBytes;

        private long _compareTotalBytes;
        public long CompareTotalBytes => _compareTotalBytes;

        public NcaReader(string fileName, NintendoContentArchiveReader ncaReader, NintendoContentArchiveReader originalNcaReader = null)
        {
            FileName = fileName;
            OriginalReader = originalNcaReader;
            Reader = ncaReader;
        }

        public bool IsContainsFile(Func<string, bool> fileFilter)
        {
            var fileInfo = GetFileInfoList().ToList();
            foreach (var nca in fileInfo)
            {
                var fsIndex = nca.Index;
                var isConfainsFileInNca = nca.Files.Any(x => CompareFileSelector(fileFilter, fsIndex, x.Item1));
                if (isConfainsFileInNca)
                {
                    return true;
                }
            }
            return false;
        }

        public NcaCompareResult Compare(NcaReader other, Func<string, bool> compareFileFilter)
        {
            Debug.Assert(other != null);

            var sourceNcaFileInfo = GetFileInfoList().ToList();
            var targetNcaFileInfo = other.GetFileInfoList().ToList();

            // ContentType が Program なら fs2 (ロゴ) の領域を比較対象外にする
            if (Reader.GetContentType() == NintendoContentFileSystemMetaConstant.ContentTypeProgram)
            {
                const int logoFsIndex = 2;
                sourceNcaFileInfo = sourceNcaFileInfo.Where(x => x.Index != logoFsIndex).ToList();
                targetNcaFileInfo = targetNcaFileInfo.Where(x => x.Index != logoFsIndex).ToList();
            }

            var compareResult = NcaCompareResult.SourceNcaExists | NcaCompareResult.TargetNcaExists;

            if (sourceNcaFileInfo.Count != targetNcaFileInfo.Count)
                return NcaCompareResult.ContainsDifference | compareResult;

            Interlocked.Exchange(ref _compareReadBytes, 0);
            // TORIAEZU: 比較対象のファイルがフィルタされると値が正確でなくなる
            Interlocked.Exchange(ref _compareTotalBytes, sourceNcaFileInfo.Sum(x => x.Size));

            foreach (var nca in sourceNcaFileInfo.Zip(targetNcaFileInfo, Tuple.Create))
            {
                if (nca.Item1.Index != nca.Item2.Index)
                    return NcaCompareResult.ContainsDifference | compareResult;

                var fsIndex = nca.Item1.Index;

                var sourceNcaFiles = GetOrderedNcaFiles(nca.Item1.Files)
                    .Where(x => CompareFileSelector(compareFileFilter, fsIndex, x.Item1))
                    .ToArray();
                var targetNcaFiles = GetOrderedNcaFiles(nca.Item2.Files)
                    .Where(x => CompareFileSelector(compareFileFilter, fsIndex, x.Item1))
                    .ToArray();

                if (sourceNcaFiles.Length > 0)
                    compareResult |= NcaCompareResult.SourceFileExists;
                if (targetNcaFiles.Length > 0)
                    compareResult |= NcaCompareResult.TargetFileExists;

                if (sourceNcaFiles.Length != targetNcaFiles.Length)
                    return NcaCompareResult.ContainsDifference | compareResult;

                foreach (var file in sourceNcaFiles.Zip(targetNcaFiles, Tuple.Create))
                {
                    var sourceName = file.Item1.Item1;
                    var targetName = file.Item2.Item1;

                    if (sourceName != targetName)
                        return NcaCompareResult.ContainsDifference | compareResult;

                    var sourceSize = file.Item1.Item2;
                    var targetSize = file.Item2.Item2;

                    if (sourceSize != targetSize)
                        return NcaCompareResult.ContainsDifference | compareResult;

                    if (FsUtil.CompareFile(nca.Item1.Reader, nca.Item2.Reader, sourceName, sourceSize,
                            x => Interlocked.Add(ref _compareReadBytes, x)) == false)
                    {
                        return NcaCompareResult.ContainsDifference | compareResult;
                    }
                }
            }

            return NcaCompareResult.Identical | compareResult;
        }

        public static bool CompareFileSelector(Func<string, bool> compareFileFilter, int fsIndex, string fileName)
        {
            if (compareFileFilter == null)
                return true;
            return compareFileFilter($"fs{fsIndex}/{fileName}");
        }

        public NcaCompareResult Compare(NcaReader other) => Compare(other, null);

        public bool Equals(NcaReader other) => Compare(other).HasFlag(NcaCompareResult.Identical);

        private Tuple<string, long>[] GetOrderedNcaFiles(List<Tuple<string, long>> ncaFiles)
        {
            return ncaFiles.OrderBy(x => x.Item1, StringComparer.InvariantCulture).ToArray();
        }

        public IEnumerable<NcaFileInfo> GetFileInfoList()
        {
            foreach (var ncaInfo in Reader.ListFsInfo(OriginalReader).OrderBy(x => x.Item1))
            {
                var fsIndex = ncaInfo.Item1;
                var fsReader = Reader.OpenFileSystemArchiveReader(fsIndex, OriginalReader);
                yield return new NcaFileInfo
                {
                    Index = fsIndex,
                    Size = ncaInfo.Item2,
                    Reader = fsReader,
                    Files = fsReader.ListFileInfo()
                };
            }
        }

        public void Dispose()
        {
            Reader?.Dispose();
            OriginalReader?.Dispose();
        }
    }
}
