﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reactive.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Threading.Tasks;
using System.Xml.Serialization;
using BezelEditor.Foundation;
using BezelEditor.Foundation.Utilities;

namespace Nintendo.Authoring.AuthoringEditor.Foundation
{
    public class NspFileEnumerable : INspFileEnumerable
    {
        public NspFileEnumeration EnumerationMode { get; }
        public PathString NspPath { get; }
        public INspFile OriginalNspFile { get; }
        public PathString DiffPathNspPath { get; }

        private volatile bool _IsCancellationRequested;

        public NspFileEnumerable(INspFile nspFile, NspFileEnumerateParameter parameter)
        {
            NspPath = nspFile.NspPath;
            OriginalNspFile = nspFile.OriginalNspFile;
            EnumerationMode = parameter.EnumerationType;
            DiffPathNspPath = parameter.DiffPatchNspFilePath;
        }

        public void CancelEnumeration()
        {
            _IsCancellationRequested = true;
        }

        public IEnumerator<NspFileEntry> GetEnumerator()
        {
            var args = new List<string>();
            var isDiffPatch = EnumerationMode == NspFileEnumeration.DiffPatch;

            if (isDiffPatch)
            {
                SetDiffPatchCommandParameter(args);
            }
            else
            {
                SetListComandParameter(args);
            }

            using (var job = AuthoringToolWrapper.Create(string.Join(" ", args)))
            {
                job.IsRedirectStandardError = true;

                var factory = isDiffPatch ? (Func<string, NspFileEntry>)MakeDiffPatchEntry : MakeFileEntry;
                var nspFileEntries = GetNspFileEntriesObservable(job, factory);
                foreach (var e in nspFileEntries.ToEnumerable())
                    yield return e;

                if (job.ExitCode != 0)
                    throw new NspFileReadException(NspPath, string.Join(Environment.NewLine, job.StandardError));
            }
        }

        private void SetListComandParameter(List<string> args)
        {
            args.Add("list");
            args.Add(NspPath.SurroundDoubleQuotes);
            if (OriginalNspFile?.IsExists == true)
            {
                args.Add($"--original {OriginalNspFile.NspPath.SurroundDoubleQuotes}");

                if (EnumerationMode == NspFileEnumeration.PatchedOnly)
                    args.Add("--patched-only");
            }
        }

        private void SetDiffPatchCommandParameter(List<string> args)
        {
            args.Add("diffpatch");
            args.Add(OriginalNspFile.NspPath.SurroundDoubleQuotes);
            args.Add(NspPath.SurroundDoubleQuotes);
            args.Add(DiffPathNspPath.SurroundDoubleQuotes);
        }

        private const string NspRootTerminated = "---------------------------------------------";

        private IObservable<NspFileEntry> GetNspFileEntriesObservable(AuthoringToolJob job, Func<string, NspFileEntry> factory)
        {
            var nspFileEntries = Observable.Create<NspFileEntry>(o =>
                {
                    var isRootOnly = EnumerationMode == NspFileEnumeration.RootOnly;
                    var d = Observable.FromEvent<DataReceivedEventHandler, DataReceivedEventArgs>(
                            h => (_, e) => h(e),
                            h => job.Process.OutputDataReceived += h,
                            h => job.Process.OutputDataReceived -= h
                        )
                        .Select(e => e.Data)
                        .Where(x => string.IsNullOrEmpty(x) == false)
                        .Subscribe(x =>
                        {
                            if (_IsCancellationRequested || (isRootOnly && x == NspRootTerminated))
                            {
                                job.TerminateAsSuccessful();
                                return;
                            }
                            var e = factory(x);
                            if (e != null)
                                o.OnNext(e);
                        });

                    job.Start();
                    job.WaitForExit();
                    o.OnCompleted();

                    return d;
                });

            return nspFileEntries;
        }

        IEnumerator IEnumerable.GetEnumerator()
        {
            return GetEnumerator();
        }

        private static readonly Regex NspFileEntryRegex = new Regex(@"^(.+)\t\((\d+) byte\)$");
        private static readonly Regex NspDiffPatchEntryRegex = new Regex(@"^(?<type>[!\+\-]) (?<path>.*?)\s+\((?:(?<size>\d+)|(?<previousSize>\d+) -> (?<currentSize>\d+)) byte\)$");

        private static NspFileEntry MakeFileEntry(string output)
        {
            if (string.IsNullOrEmpty(output))
                return null;

            var match = NspFileEntryRegex.Match(output);
            if (match.Success == false)
                return null;

            return new NspFileEntry
            {
                FilePath = match.Groups[1].Value.TrimEnd(),
                FileSize = ulong.Parse(match.Groups[2].Value),
            };
        }

        private static NspFileEntry MakeDiffPatchEntry(string output)
        {
            if (string.IsNullOrEmpty(output))
                return null;

            var match = NspDiffPatchEntryRegex.Match(output);
            if (match.Success == false)
                return null;

            var fileSizeStr = match.Groups["size"].Value;
            var hasFileSize = string.IsNullOrEmpty(fileSizeStr) == false;
            var fileSize = ulong.Parse(hasFileSize ? fileSizeStr : match.Groups["previousSize"].Value);

            return new NspFileEntry
            {
                FilePath = match.Groups["path"].Value,
                FileSize = fileSize,
                ModifiedFileSize = hasFileSize ? fileSize : ulong.Parse(match.Groups["currentSize"].Value),
                ModifiedType = ToModifiedType(match.Groups["type"].Value)
            };
        }

        private static NspFileModifiedType ToModifiedType(string type)
        {
            switch (type)
            {
                case "!":
                    return NspFileModifiedType.Changed;
                case "+":
                    return NspFileModifiedType.Added;
                case "-":
                    return NspFileModifiedType.Removed;
                default:
                    return NspFileModifiedType.Unknown;
            }
        }
    }

    public class NspFile : INspFile
    {
        public NspFileEnumeration EnumerationMode { get; set; }

        public PathString NspPath { get; set; }

        public INspFile OriginalNspFile { get; set; }

        public bool IsExists => File.Exists(NspPath);
        public bool IsAuthoringToolOperational => true;

        public NspFile(string nspFilePath)
        {
            NspPath = nspFilePath.ToPathString();
        }

        public INspFileEnumerable Files(NspFileEnumerateParameter parameter) => new NspFileEnumerable(this, parameter);

        public string ReadAllText(string pathInNsp, Encoding encoding) =>
            ReadAllTextAsync(pathInNsp, encoding).Result;

        public string ReadAllText(string filePath) => ReadAllText(filePath, Encoding.UTF8);

        public Task<string> ReadAllTextAsync(string pathInNsp, Encoding encoding) =>
            ReadAllTextInternalAsync(NspPath, pathInNsp, encoding);

        public Task<string> ReadAllTextAsync(string filePath) => ReadAllTextAsync(filePath, Encoding.UTF8);

        public async Task<T> ReadXmlAsync<T>(string filePath, Encoding encoding)
        {
            var xmlText = await ReadAllTextAsync(filePath, encoding).ConfigureAwait(false);
            var serializer = new XmlSerializer(typeof(T));
            using (var reader = new StringReader(xmlText))
            {
                return (T) serializer.Deserialize(reader);
            }
        }

        public Task<T> ReadXmlAsync<T>(string filePath) => ReadXmlAsync<T>(filePath, Encoding.UTF8);

        public Task ExtractAsync(string pathInNsp, string path) =>
            ExtractInternalAsync(NspPath, $"--target \"{pathInNsp}\" --output {path.ToPathString().SurroundDoubleQuotes}", null);

        public Task ExtractContentAsync(ContentType contentType, string path) =>
            ExtractInternalAsync(NspPath, $"{GetOutputDirectory(path)} --content-type {contentType}", null);

        public Task ExtractAllAsync(PathString outputDir, Action<int> progressChanged, CancellationToken cancellationToken) =>
            ExtractInternalAsync(NspPath, GetOutputDirectory(outputDir), progressChanged, cancellationToken);

        public async Task ExtractFilesAsync(IDictionary<string, string> targets)
        {
            var tempListPath = Path.GetTempFileName().ToPathString();
            File.WriteAllLines(tempListPath, targets.Select(x => $"{x.Key}\t{x.Value}"));

            await ExtractInternalAsync(NspPath, $"--target-list {tempListPath.SurroundDoubleQuotes}", null).ConfigureAwait(false);

            try
            {
                File.Delete(tempListPath);
            }
            catch { }
        }

        public NcaCompareResult GetNcaIdenticalResult(INspFile target, string contentType, string prefixPath = null, Action<float> onProgressChange = null, CancellationToken token = default(CancellationToken))
        {
            var targetNsp = target as NspFile;
            if (targetNsp == null)
                throw new ArgumentException(nameof(target));

            var argsBuilder = new StringBuilder();

            argsBuilder.Append(
                "compare" +
                $" --content-type {contentType}" +
                $" --source {NspPath.SurroundDoubleQuotes}" +
                $" --target {targetNsp.NspPath.SurroundDoubleQuotes}");

            if (OriginalNspFile?.NspPath != null)
                argsBuilder.Append($" --original {OriginalNspFile.NspPath.SurroundDoubleQuotes}");
            else if (targetNsp.OriginalNspFile?.NspPath != null)
                argsBuilder.Append($" --original {targetNsp.OriginalNspFile.NspPath.SurroundDoubleQuotes}");

            if (string.IsNullOrEmpty(prefixPath) == false)
                argsBuilder.Append($" --nca-prefix-path {prefixPath}");

            var args = argsBuilder.ToString();
            using (var process = ProcessUtility.CreateProcess(AuthoringToolHelper.AuthoringEditorHelperExe, args))
            {
                var outputSync = new ManualResetEventSlim(false);
                var errorSync = new ManualResetEventSlim(false);
                var errorOutput = new StringBuilder();

                process.OutputDataReceived += (_, e) =>
                {
                    if (e.Data == null)
                    {
                        outputSync.Set();
                        return;
                    }
                    var t = e.Data.Split(new[] {'/'}, 2, StringSplitOptions.None);
                    if (t.Length != 2)
                        return;
                    long readBytes;
                    long totalBytes;
                    if (long.TryParse(t[0], out readBytes) && long.TryParse(t[1], out totalBytes))
                    {
                        onProgressChange?.Invoke((float) (readBytes / (double) totalBytes * 100));
                    }
                };
                process.ErrorDataReceived += (_, e) =>
                {
                    if (e.Data == null)
                    {
                        errorSync.Set();
                        return;
                    }
                    errorOutput.AppendLine(e.Data);
                };
                process.Start();
                process.BeginOutputReadLine();
                process.BeginErrorReadLine();

                token.Register(() =>
                {
                    try
                    {
                        if (process.HasExited == false)
                            process.Kill();
                    }
                    catch
                    {
                        // プロセス終了中の例外は無視 (終了は成功したという前提)
                    }
                });

                process.WaitForExit();

                outputSync.Wait(token);
                errorSync.Wait(token);

                onProgressChange?.Invoke(100);

                if (process.ExitCode == -1)
                    throw new InvalidOperationException(errorOutput.ToString());

                return (NcaCompareResult) process.ExitCode;
            }
        }

        private async Task ExtractInternalAsync(PathString nspPath,
            string additionalArgs,
            Action<int> progressChanged,
            CancellationToken cancellationToken = default(CancellationToken))
        {
            var args = $"extract {nspPath.SurroundDoubleQuotes} {additionalArgs}";
            if (OriginalNspFile?.IsExists == true)
            {
                args += $" --original {OriginalNspFile.NspPath.SurroundDoubleQuotes}";
            }

            using (var runner = new AuthoringEditorHelperRunner(this, args, progressChanged, cancellationToken))
            {
                runner.Start();
                await runner.WaitForExit().ConfigureAwait(false);
            }
        }

        private async Task<string> ReadAllTextInternalAsync(PathString nspPath, string fileInNsp, Encoding encoding)
        {
            var args = $"extract {nspPath.SurroundDoubleQuotes} --target \"{fileInNsp}\" --standard-output --standard-output-encoding {encoding.HeaderName}";
            if (OriginalNspFile?.IsExists == true)
            {
                args += $" --original {OriginalNspFile.NspPath.SurroundDoubleQuotes}";
            }

            using (var outputSync = new ManualResetEventSlim(false))
            using (var runner = new AuthoringEditorHelperRunner(this, args, null, default(CancellationToken)))
            {
                var captureOutput = new StringBuilder();
                runner.RunnerProcess.StartInfo.StandardOutputEncoding = encoding;
                runner.RunnerProcess.StartInfo.RedirectStandardOutput = true;
                runner.RunnerProcess.OutputDataReceived += (_, e) =>
                {
                    if (e.Data == null)
                    {
                        outputSync.Set();
                        return;
                    }
                    captureOutput.AppendLine(e.Data);
                };
                runner.Start();
                await runner.WaitForExit().ConfigureAwait(false);
                outputSync.Wait();
                return captureOutput.ToString();
            }
        }

        private static string GetOutputDirectory(PathString outputDir)
        {
            var finalOutputDir = outputDir.DirectoryWithoutLastSeparator.ToPathString();
            return $" --output-dir {finalOutputDir.SurroundDoubleQuotes}";
        }

        private class AuthoringEditorHelperRunner : IDisposable
        {
            public Process RunnerProcess { get; }

            private readonly INspFile _NspFile;
            private readonly CancellationToken _CancellationToken;
            private readonly Action<int> _ProgressChanged;
            private readonly StringBuilder _ErrorOutput = new StringBuilder();

            public void Dispose()
            {
                RunnerProcess.Dispose();
            }

            public AuthoringEditorHelperRunner(INspFile nspFile, string args, Action<int> progressChanged,
                CancellationToken cancellationToken)
            {
                _NspFile = nspFile;
                _CancellationToken = cancellationToken;
                _ProgressChanged = progressChanged;
                RunnerProcess = ProcessUtility.CreateProcess(AuthoringToolHelper.AuthoringEditorHelperExe, args, false);

                if (_ProgressChanged != null)
                {
                    RunnerProcess.StartInfo.RedirectStandardOutput = true;
                    RunnerProcess.OutputDataReceived += (_, e) =>
                    {
                        if (e.Data == null)
                            return;
                        var progressOutput = e.Data.Split(new[] { '/', '\t' }, 3);
                        if (progressOutput.Length != 3)
                            return;
                        var total = 0;
                        var written = 0;
                        if (int.TryParse(progressOutput[0], out written) && int.TryParse(progressOutput[1], out total))
                            _ProgressChanged((int)(written / (double)total * 100));
                    };
                }

                RunnerProcess.StartInfo.RedirectStandardError = true;
                RunnerProcess.ErrorDataReceived += (_, e) =>
                {
                    if (e.Data == null)
                        return;
                    _ErrorOutput.AppendLine(e.Data);
                };

            }

            public void Start()
            {
                RunnerProcess.Start();

                if (_CancellationToken != default(CancellationToken))
                {
                    _CancellationToken.Register(() =>
                    {
                        try
                        {
                            RunnerProcess.Kill();
                        }
                        catch { } // プロセス強制終了に伴う例外は無視
                    });
                }

                if (RunnerProcess.StartInfo.RedirectStandardOutput)
                    RunnerProcess.BeginOutputReadLine();
                RunnerProcess.BeginErrorReadLine();

                _ProgressChanged?.Invoke(100);
            }

            public async Task WaitForExit()
            {
                await Task.Run(() =>
                {
                    try
                    {
                        RunnerProcess.WaitForExit();
                    }
                    catch { } // プロセス終了の待機に伴う例外は無視
                }, _CancellationToken)
                    .ContinueWith(x => { }, _CancellationToken, TaskContinuationOptions.None, TaskScheduler.Default)
                    .ConfigureAwait(false);
                if (RunnerProcess.ExitCode != 0)
                    throw new NspFileReadException(_NspFile.NspPath, _ErrorOutput.ToString());
            }
        }

    }
}
