﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;

namespace Nintendo.Authoring.AuthoringEditor.Foundation
{
    public class DisposableDirectory : IDisposable
    {
        public string RootPath { get; }

        public DisposableDirectory()
        {
            RootPath = Path.Combine(
                Path.GetTempPath(),
                @"Nintendo\AuthoringEditor",
                Path.GetRandomFileName());
            Directory.CreateDirectory(RootPath);
        }

        public void Dispose()
        {
            try
            {
                Directory.Delete(RootPath, true);
            }
            catch
            {
                // ignore
            }
        }

        public string CreateFile(string path, string text = "")
        {
            var tempFile = Path.Combine(RootPath, path);
            File.WriteAllText(tempFile, text);
            return tempFile;
        }

        public string CreateFile(string path, byte[] data)
        {
            var tempFile = Path.Combine(RootPath, path);
            File.WriteAllBytes(tempFile, data);
            return tempFile;
        }

        public string CreateFolder(string path)
        {
            var tempDir = Path.Combine(RootPath, path);
            Directory.CreateDirectory(tempDir);
            return tempDir;
        }

        public DisposableDirectoryBuilder Build() => new DisposableDirectoryBuilder(this);
    }

    public class DisposableDirectoryBuilder
    {
        private readonly DisposableDirectory _Directory;

        public DisposableDirectoryBuilder(DisposableDirectory tempDir)
        {
            _Directory = tempDir;
        }

        public DisposableDirectoryBuilder CreateFile(string path)
        {
            _Directory.CreateFile(path);
            return this;
        }

        public DisposableDirectoryBuilder CreateFile(string path, byte[] data)
        {
            _Directory.CreateFile(path, data);
            return this;
        }

        public DisposableDirectoryBuilder CreateFolder(string path)
        {
            _Directory.CreateFolder(path);
            return this;
        }
    }
}
