﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using BezelEditor.Foundation.Extentions;
using Nintendo.Authoring.AuthoringEditor.Foundation;

namespace Nintendo.Authoring.AuthoringEditor.Core
{
    public class SaveDataSizeCalculator
    {
        private readonly List<int> _files = new List<int>();
        private int _dirCount;

        #region セーブデータ仕様

        // 1 ブロック当たりのバイト数
        private const int BlockByteSize = 16*1024;
        // ファイルエントリのマージンとして確保するブロック数
        private const int FileEntryMarginBlockCount = 1;
        // ディレクトリエントリのマージンとして確保するブロック数
        private const int DirectoryEntryMarginBlockCount = 1;

        #endregion

        public int SaveDataSize => CulculateSaveDataSize();

        static SaveDataSizeCalculator()
        {
            Debug.Assert(MathHelper.IsPowerOfTwo(BlockByteSize));
        }

        public void AddFileEntry(int byteSize)
        {
            if (byteSize < 0)
                throw new ArgumentOutOfRangeException();

            _files.Add(byteSize);
        }

        public void AddFileEntries(IEnumerable<int> byteSizes)
        {
            byteSizes.AddRangeTo(_files);
        }

        public void AddDirectoryEntry()
        {
            AddDirectoryEntries(1);
        }

        public void AddDirectoryEntries(int count)
        {
            _dirCount += count;
        }

        private int CulculateSaveDataSize()
        {
            // ファイルイメージ
            var allFileByteSize = _files.Sum(AlignBlockByteSize);
            Debug.Assert(IsAlignedBlockByteSize(allFileByteSize));

            // ファイルアローケーションテーブル
            var fileAllocTableCount = AlignEntryCountPerBlock(_files.Count);
            var allFileAllocTableByteSize = fileAllocTableCount*BlockByteSize;
            Debug.Assert(IsAlignedBlockByteSize(allFileAllocTableByteSize));

            // ディレクトリアローケーションテーブル
            var dirAllocTableCount = AlignEntryCountPerBlock(_dirCount);
            var allDirAllocTableByteSize = dirAllocTableCount*BlockByteSize;
            Debug.Assert(IsAlignedBlockByteSize(allDirAllocTableByteSize));

            // ファイル、ディレクトリエントリ用として余分に確保しておくブロック
            var directoryEntryMarginByteSize = DirectoryEntryMarginBlockCount*BlockByteSize;
            var fileEntryMarginByteSize = FileEntryMarginBlockCount*BlockByteSize;
            var marginByteSize = directoryEntryMarginByteSize + fileEntryMarginByteSize;
            Debug.Assert(IsAlignedBlockByteSize(marginByteSize));

            // 合計サイズ
            var allSize = 0
                          + allFileByteSize
                          + allFileAllocTableByteSize
                          + allDirAllocTableByteSize
                          + marginByteSize;
            Debug.Assert(IsAlignedBlockByteSize(allSize));
            return allSize;
        }

        private static bool IsAlignedBlockByteSize(int v)
        {
            return (v & (BlockByteSize - 1)) == 0;
        }

        private static int AlignBlockByteSize(int v)
        {
            return (v + (BlockByteSize - 1)) & ~(BlockByteSize - 1);
        }

        private static int AlignEntryCountPerBlock(int v)
        {
            int managementEntry = 2; // 管理領域として 2 ブロックが必要
            int entryByteSize = 96; // 管理領域も含めて 1 エントリあたり 96 バイト必要
            return ((managementEntry + v)*entryByteSize + BlockByteSize - 1)/BlockByteSize;
        }
    }
}
