﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq.Expressions;
using System.Reflection;
using System.Runtime.CompilerServices;
using BezelEditor.Mvvm;

namespace Nintendo.Authoring.AuthoringEditor.Core
{
    public abstract class PropertyTrackingDispoableModel<T> : DisposableModelBase, IPropertyMerge<T>
    {
        private static readonly Hashtable _PropertyInfoCache = new Hashtable();
        private readonly Hashtable _ChangeProperties = new Hashtable();

        public bool IsPropertyChanged(string propertyName)
        {
            return _ChangeProperties.ContainsKey(propertyName);
        }

        protected bool TryGetChangedProperty<TProperty>(Expression<Func<T, TProperty>> propertySelector, out TProperty value)
        {
            var propertyInfo = GetCachedPropertyInfo(propertySelector);
            if (_ChangeProperties.ContainsKey(propertyInfo.Name))
            {
                value = (TProperty)propertyInfo.GetValue(this);
                return true;
            }
            value = default(TProperty);
            return false;
        }

        protected bool TrySetChangedProperty<TProperty>(Expression<Func<T, TProperty>> propertySelector, T targetObject)
        {
            var propertyInfo = GetCachedPropertyInfo(propertySelector);
            if (_ChangeProperties.ContainsKey(propertyInfo.Name))
            {
                var newValue = propertyInfo.GetValue(this);
                propertyInfo.SetValue(targetObject, newValue);
                return true;
            }
            return false;
        }

        public new bool SetProperty<TProperty>(ref TProperty backingField, TProperty newValue, [CallerMemberName] string propertyName = null)
        {
            var changed = base.SetProperty(ref backingField, newValue, propertyName);
            if (changed)
            {
                lock (_ChangeProperties)
                {
                    _ChangeProperties[propertyName] = true;
                }
            }
            return changed;
        }

        protected void ClearChangePropertyNames()
        {
            lock (_ChangeProperties)
            {
                _ChangeProperties.Clear();
            }
        }

        public abstract IEnumerable<string> GetChangedPropertyNames();

        public abstract void MergePropertyValues(T value);

        private static PropertyInfo GetCachedPropertyInfo<TProperty>(Expression<Func<T, TProperty>> propertySelector)
        {
            var propertyInfo = (PropertyInfo)_PropertyInfoCache[propertySelector];
            if (propertyInfo == null)
            {
                var memberExpression = (MemberExpression)propertySelector.Body;
                propertyInfo = (PropertyInfo)memberExpression.Member;
                lock (_PropertyInfoCache)
                {
                    _PropertyInfoCache[propertySelector] = propertyInfo;
                }
            }
            return propertyInfo;
        }

    }
}
