﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Globalization;
using System.IO;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using System.Linq;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using System.Xml.Linq;
using System.Xml.Serialization;
using BezelEditor.Foundation;
using SimpleInjector;

namespace Nintendo.Authoring.AuthoringEditor.Core
{
    public class NspImporter
    {
        public static readonly string ContentMetaFileFileExtension = ".cnmt.xml";
        public static readonly string ApplicationControlPropertyFileExtension = ".nacp.xml";
        public static readonly string ProgramInfoFileExtension = ".programinfo.xml";
        public static readonly string CardSpecFile = "cardspec.xml";
        public static readonly string LegalInfoFileExtension = ".legalinfo.xml";

        public class Icon
        {
            public LanguageType Language { get; set; }
            public string RawFilePath { get; set; }
            public string NxFilePath { get; set; }
        }

        public class Option
        {
            public Container DiContainer { get; set; }
            public INspFile InputNspFile { get; set; }
            public string TempNspExtractFilePath { get; set; }
        }

        public class ResultType : NspHandleResultType
        {
            public ApplicationMeta ApplicationMeta { get; set; }
            public ApplicationContentMeta ApplicationContentMeta { get; set; }
            public AocMeta AocMeta { get; set; }
            public PatchContentMeta PatchContentMeta { get; set; }
            public AocContentMeta[] AocContentMetas { get; set; }
            public ProgramInfo ProgramInfo { get; set; }
            public ContentMetaProperties ContentMetaProperties { get; set; }
            public NspExtraResourceImporter ExtraResourceImporter { get; set; }
        }

        public static async Task<ContentMetaType> DetectContentMetaTypeAsync(INspFile nspFile)
        {
            try
            {
                var contentMetas = await ReadAllContentMeta<ContentMeta>(nspFile).WhenAll().ConfigureAwait(false);
                return contentMetas.First().Type;
            }
            catch
            {
                return ContentMetaType.Unknown;
            }
        }

        public async Task<ResultType> ToMetaAsync(Option option)
        {
            if (option.InputNspFile == null || option.InputNspFile.IsExists == false)
                return new ResultType {Result = NspHandleResult.NotFoundNspFile};

            if (Directory.Exists(option.TempNspExtractFilePath) == false)
                return new ResultType {Result = NspHandleResult.NotFoundTemporaryNspExtractDir};

            if (File.Exists(AuthoringToolHelper.AuthoringToolExe) == false)
                return new ResultType {Result = NspHandleResult.NotFoundAuthoringToolExe};

            return await ToMetaInternalAsync(option).ConfigureAwait(false);
        }

        public static IEnumerable<Task<T>> ReadAllContentMeta<T>(INspFile nspFile) where T : ContentMeta
        {
            return
                nspFile.Files(new NspFileEnumerateParameter{ EnumerationType = NspFileEnumeration.RootOnly })
                    .Where(x => ContentMetaXmlRegex.IsMatch(x.FilePath))
                    .Select(file => nspFile.ReadXmlAsync<T>(file.FilePath));
        }

        public static T ReadContentMeta<T>(INspFile nspFile) where T : ContentMeta
        {
            return ReadAllContentMeta<T>(nspFile).WhenAll().Result.FirstOrDefault();
        }

        public static IEnumerable<Icon> FindNspIcons(INspFile nspFile)
        {
            var icons = new Dictionary<LanguageType, Icon>();
            foreach (var entry in nspFile.Files(new NspFileEnumerateParameter{ EnumerationType = NspFileEnumeration.RootOnly }))
            {
                var match = IconFileNameRegex.Match(entry.FilePath);
                if (match.Success == false)
                    continue;

                LanguageType languageType;
                if (Enum.TryParse(match.Groups[2].Value, out languageType) == false)
                    continue;

                Icon icon;
                if (icons.TryGetValue(languageType, out icon) == false)
                {
                    icons[languageType] = new Icon {Language = languageType};
                }

                var iconType = match.Groups[1].Value.ToLowerInvariant();
                switch (iconType)
                {
                    case "nx":
                        icons[languageType].NxFilePath = entry.FilePath;
                        break;
                    case "raw":
                        icons[languageType].RawFilePath = entry.FilePath;
                        break;
                    default:
                        throw new ArgumentException(nameof(iconType));
                }
            }
            return icons.Values;
        }

        private static async Task<ResultType> ToMetaInternalAsync(Option option)
        {
            // Nsp から読み込んだ Meta 等の破棄は呼び出し元で行う
            var result = new ResultType
            {
                Result = NspHandleResult.Error
            };

            try
            {
                ResultType error;

                switch (await DetectContentMetaTypeAsync(option.InputNspFile).ConfigureAwait(false))
                {
                    case ContentMetaType.AddOnContent:
                        error = await ReadAddOnContentMetaAsync(option, result).ConfigureAwait(false);
                        break;

                    case ContentMetaType.Application:
                        error = await ReadApplicationMetaAsync(option, result).ConfigureAwait(false);
                        break;

                    case ContentMetaType.Patch:
                        if (option.DiContainer.GetInstance<ApplicationCapability>().IsSupportMakingPatch == false)
                            return new ResultType { Result = NspHandleResult.PatchIsNotSupported };
                        error = await ReadPatchMetaAsync(option, result).ConfigureAwait(false);
                        break;

                    case ContentMetaType.Unknown:
                        error = new ResultType { Result = NspHandleResult.Error };
                        break;

                    default:
                        throw new ArgumentOutOfRangeException();
                }

                if (error != null)
                    return error;

                if (result.AocMeta == null)
                    result.AocMeta = new AocMeta();

                result.Result = NspHandleResult.Ok;
                return result;
            }
            catch (Exception e)
            {
                return new ResultType {Result = NspHandleResult.Error, ErrorMessages = new[] {e.Message}};
            }
            finally
            {
                if (result.Result != NspHandleResult.Ok)
                {
                    result.ApplicationMeta?.Dispose();
                    result.ApplicationContentMeta?.Dispose();
                    result.AocMeta?.Dispose();
                    result.PatchContentMeta?.Dispose();
                    if (result.AocContentMetas != null)
                    {
                        foreach (var e in result.AocContentMetas)
                            e.Dispose();
                    }
                    result.ProgramInfo?.Dispose();
                    result.ContentMetaProperties?.Dispose();
                }
            }
        }

        private static async Task<ResultType> ReadPatchMetaAsync(Option option, ResultType resultType)
        {
            var nspFile = option.InputNspFile;
            var patchContentMetas = await ReadAllContentMeta<PatchContentMeta>(nspFile).WhenAll().ConfigureAwait(false);
            resultType.PatchContentMeta =  patchContentMetas.FirstOrDefault();
            if (resultType.PatchContentMeta == null)
                return new ResultType
                {
                    Result = NspHandleResult.ExtractError,
                    ErrorMessages = new[] { $"File not found: *{ContentMetaFileFileExtension}" }
                };

            var contentMeta = resultType.PatchContentMeta;
            var r = await ReadApplicationMetaInternalAsync(resultType, contentMeta, option.InputNspFile).ConfigureAwait(false);

            // 製品化処理されたパッチの場合 OR 展開済みパッチ nsp の場合
            // => v0 のアプリは不要
            //    製品化されたパッチの場合: 読み込んだとしても nca の中身が読めないため
            //    展開済みパッチ nsp の場合: ディレクトリ構造に必要なファイルは存在するため
            if (contentMeta.IsProduction == false &&
                nspFile.IsAuthoringToolOperational &&
                (nspFile.OriginalNspFile == null || nspFile.OriginalNspFile.IsExists == false))
            {
                return new ResultType
                {
                    Result = NspHandleResult.NeedOriginalApplicationNspError,
                    ErrorMessages = new [] {"Need original application NSP file."}
                };
            }

            resultType.ContentMetaProperties = await ReadPropertiesAsync(option, resultType).ConfigureAwait(false);
            await ReadIconFilesAsync(option, resultType).ConfigureAwait(false);

            resultType.ExtraResourceImporter = new NspExtraResourceImporter(
                resultType.ApplicationMeta.Application,
                option.TempNspExtractFilePath,
                option.InputNspFile, contentMeta);

            return r;
        }

        private static async Task<ResultType> ReadApplicationMetaAsync(Option option, ResultType resultType)
        {
            var appContentMetas = await ReadAllContentMeta<ApplicationContentMeta>(option.InputNspFile).WhenAll().ConfigureAwait(false);
            var contentMeta = appContentMetas.FirstOrDefault();
            resultType.ApplicationContentMeta = contentMeta;
            if (resultType.ApplicationContentMeta == null)
                return new ResultType
                {
                    Result = NspHandleResult.ExtractError,
                    ErrorMessages = new[] {"File not found: *.cnmt.xml"}
                };

            var r = await ReadApplicationMetaInternalAsync(resultType, contentMeta, option.InputNspFile).ConfigureAwait(false);

            resultType.ContentMetaProperties = await ReadPropertiesAsync(option, resultType).ConfigureAwait(false);
            await ReadIconFilesAsync(option, resultType).ConfigureAwait(false);
            await ReadUnpublishableErrorAsync(option, resultType).ConfigureAwait(false);

            resultType.ExtraResourceImporter = new NspExtraResourceImporter(
                resultType.ApplicationMeta.Application,
                option.TempNspExtractFilePath,
                option.InputNspFile,
                contentMeta);

            return r;
        }

        private static async Task<ResultType> ReadAddOnContentMetaAsync(Option option, ResultType resultType)
        {
            var aocContentMetas = await ReadAllContentMeta<AocContentMeta>(option.InputNspFile).WhenAll().ConfigureAwait(false);
            if (aocContentMetas == null || aocContentMetas.Length == 0)
            {
                return new ResultType
                {
                    Result = NspHandleResult.ExtractError,
                    ErrorMessages = new[] { "File not found: *.cnmt.xml" }
                };
            }

            resultType.AocContentMetas = aocContentMetas;
            resultType.AocMeta = ReadAocMeta(option.DiContainer, aocContentMetas);
            // TODO: Aoc時でもMetaの参照を作っているが、作らなくてもいいようにする
            resultType.ApplicationMeta = new ApplicationMeta();

            return null;
        }

        #region Read AoC nsp Utilties

        private static AocMeta ReadAocMeta(Container diContainer, AocContentMeta[] aocContentMetas)
        {
            var aocMeta = new AocMeta
            {
                ApplicationId = aocContentMetas[0].ApplicationId
            };

            // Contents を作る
            foreach (var src in aocContentMetas)
            {
                var dst = new AocContent(diContainer, aocMeta)
                {
                    Index = src.Index,
                    ReleaseVersion = src.Version >> 16,
                    RequiredApplicationReleaseVersion = src.RequiredApplicationVersion >> 16,
                    Digest = src.Digest,
                    Tag = src.Tag,
                    ContentDataSize = src.Contents[0].Size
                };

                aocMeta.Contents.Add(dst);
            }

            return aocMeta;
        }

        #endregion

        #region Read Application nsp Utiltiles

        private static async Task<ResultType> ReadApplicationMetaInternalAsync(ResultType resultType, ContentMeta contentMeta, INspFile nspFile)
        {
            var controlId = contentMeta.GetContentId(ContentType.Control);
            if (controlId == null)
            {
                return new ResultType
                {
                    Result = NspHandleResult.ExtractError,
                    ErrorMessages = new[] { $"File not found: *{ApplicationControlPropertyFileExtension}" }
                };
            }

            var appXml = await nspFile.ReadAllTextAsync($"{controlId}{ApplicationControlPropertyFileExtension}").ConfigureAwait(false);
            Application app;
            {
                var serializer = new XmlSerializer(typeof(Application));
                using (var reader = new StringReader(appXml))
                {
                    app = (Application)serializer.Deserialize(reader);
                }
                app.SourceXmlText = appXml;
            }

            var programId = contentMeta.GetContentId(ContentType.Program);
            if (programId == null)
            {
                return new ResultType
                {
                    Result = NspHandleResult.ExtractError,
                    ErrorMessages = new[] { $"File not found: *{ProgramInfoFileExtension}" }
                };
            }
            resultType.ProgramInfo = await nspFile.ReadXmlAsync<ProgramInfo>($"{programId}{ProgramInfoFileExtension}").ConfigureAwait(false);

            var cardspecXml = await nspFile.ReadAllTextAsync(CardSpecFile).ConfigureAwait(false);
            if (string.IsNullOrEmpty(cardspecXml))
            {
                return new ResultType
                {
                    Result = NspHandleResult.ExtractError,
                    ErrorMessages = new[] { $"File not found: {CardSpecFile}" }
                };
            }

            var meta = new ApplicationMeta();
            meta.Application = app;
            FixupNspContentMeta(cardspecXml, meta, contentMeta);
            resultType.ApplicationMeta = meta;

            return null;
        }

        private static async Task ReadIconFilesAsync(Option option, ResultType resultType)
        {
            var nspFile = option.InputNspFile;
            var titles = resultType.ApplicationMeta.Application.Titles.ToArray();

            var icons = FindNspIcons(nspFile).ToArray();
            var outputIcons = new Dictionary<string, string>();

            foreach (var icon in icons)
            {
                var title = titles.FirstOrDefault(x => x.Language == icon.Language);
                if (title == null)
                    continue;

                // nx
                {
                    var rawExtractPath = Path.Combine(option.TempNspExtractFilePath, $"{icon.Language}.raw.jpg");
                    outputIcons.Add(icon.RawFilePath, rawExtractPath);
                    title.OriginalIconFilePath = new ExpandablePath(rawExtractPath);
                    title.NspRawIconFilePath = icon.RawFilePath;
                }

                // raw
                {
                    var nxExtractPath = Path.Combine(option.TempNspExtractFilePath, $"{icon.Language}.nx.jpg");
                    outputIcons.Add(icon.NxFilePath, nxExtractPath);
                    title.OriginalNxIconFilePath = new ExpandablePath(nxExtractPath);
                    title.NspNxIconFilePath = icon.NxFilePath;
                }
            }
            await nspFile.ExtractFilesAsync(outputIcons).ConfigureAwait(false);
        }

        private static async Task<ContentMetaProperties> ReadPropertiesAsync(Option option, ResultType resultType)
        {
            if (!option.InputNspFile.IsAuthoringToolOperational)
            {
                return new ContentMetaProperties();
            }
            return await GetNspContentMetaPropertiesAsync(option.InputNspFile.NspPath).ConfigureAwait(false) ??
                   new ContentMetaProperties();
        }

        private static async Task ReadUnpublishableErrorAsync(Option option, ResultType resultType)
        {
            if (!option.InputNspFile.IsAuthoringToolOperational)
            {
                return;
            }
            var checker = new NspPublishableChecker();
            var checkError = await checker.CheckUnpublishableErrorAsync(option.InputNspFile.NspPath)
                .ConfigureAwait(false);
            if (checkError == null)
                return;
            var unpublishableErrors = resultType.ApplicationMeta.UnpublishableErrors;
            var ja = new CultureInfo("ja-jp").TwoLetterISOLanguageName;
            foreach (var error in checkError.Content.Errors.OrderBy(x => x.Message.Id))
            {
                var title = new CulturizedMessage();
                {
                    title[ja] = error.Message.Title.Japanese;
                    title.DefaultMessage = error.Message.Title.English;
                }
                var description = new CulturizedMessage();
                {
                    description[ja] = error.Message.Description.Japanese;
                    description.DefaultMessage = error.Message.Description.English;
                }

                unpublishableErrors.Add(new ApplicationMetaError
                {
                    Id = error.Message.Id,
                    ErrorSeverity = ApplicationMetaErrorSeverity.Error,
                    CulturizedTitle = title,
                    CulturizedDescription = description
                });
            }
        }

        private static async Task<ContentMetaProperties> GetNspContentMetaPropertiesAsync(PathString nspPath)
        {
            var args = $"getnspproperty --xml {nspPath.SurroundDoubleQuotes}";
            using (var job = AuthoringToolWrapper.Create(args))
            {
                job.IsRedirectStandardError = true;
                job.IsRedirectStandardOutput = true;
                job.Start();

                await job.WaitForExitAsync().ConfigureAwait(false);
                if (job.ExitCode != 0)
                    return null; // getnspproperty オプションが存在しない古いバージョン

                var xmlText = string.Join(Environment.NewLine, job.StandardOutput);
                if (string.IsNullOrEmpty(xmlText))
                    return new ContentMetaProperties();

                var contentMetaProps =
                    (ContentMetaProperties)
                    new XmlSerializer(typeof(ContentMetaProperties)).Deserialize(new StringReader(xmlText));
                return contentMetaProps;
            }
        }

        private static void FixupNspContentMeta(string cardspecXmlText, ApplicationMeta appMeta, ContentMeta contentMeta)
        {
            appMeta.Core.Name = contentMeta.Type.ToString();
            // パッチならアプリケーション ID の値は ContentMeta に直接格納されているのでそれを使う
            appMeta.Core.ApplicationIdHex = contentMeta.Type == ContentMetaType.Patch
                ? (contentMeta as PatchContentMeta)?.ApplicationIdHex
                : contentMeta.Id.ToHex();
            // ContentMeta の Version = Meta/Application/ReleaseVersion * 65536 + Meta/Application/PrivateVersion
            appMeta.Application.ReleaseVersion = (ushort)(contentMeta.Version >> 16);

            {
                var cardSpec = XDocument.Parse(cardspecXmlText);

                var size = cardSpec.Descendants("Size").FirstOrDefault()?.Value;
                var clockRate = cardSpec.Descendants("ClockRate").FirstOrDefault()?.Value;

                var isAutoSetSize = cardSpec.Descendants("AutoSetSize").FirstOrDefault()?.Value ?? string.Empty;
                var isAutoSetClockRate = cardSpec.Descendants("AutoSetClockRate").FirstOrDefault()?.Value ?? string.Empty;

                appMeta.CardSpec.IsAutomaticSettingSize =
                    isAutoSetSize.Equals("true", StringComparison.InvariantCultureIgnoreCase) ||
                    size == null;
                appMeta.CardSpec.Size = size == null ? 1 : int.Parse(size);

                appMeta.CardSpec.IsAutomaticSettingClockRate =
                    isAutoSetClockRate.Equals("true", StringComparison.InvariantCultureIgnoreCase) ||
                    clockRate == null;
                appMeta.CardSpec.ClockRate = clockRate == null ? 25 : int.Parse(clockRate);
            }
        }

        private static readonly Regex IconFileNameRegex = new Regex(@"^[a-f0-9]+\.(raw|nx)\.([^\.]+)\.jpg$");
        private static readonly Regex ContentMetaXmlRegex = new Regex(@"^[a-f0-9]+" + Regex.Escape(ContentMetaFileFileExtension) + "$");

        #endregion
    }
}
