﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Reflection;

namespace Nintendo.Authoring.AuthoringEditor.Core
{
    [AttributeUsage(AttributeTargets.Field, AllowMultiple = true)]
    public class FileExtensionAttribute : Attribute
    {
        public string FileExtension { get; }

        public FileExtensionAttribute(string extension)
        {
            FileExtension = extension;
        }
    }

    public enum ImportableFileType
    {
        [FileExtension(".narp")]
        [FileExtension(".nrp")]
        Project,

        [FileExtension(".nmeta")]
        [FileExtension(".meta")]
        Meta,

        [FileExtension(".nsp")]
        [FileExtension(".nspu")]
        [FileExtension(".xml")]
        Nsp,
    }

    public enum ExportableFileType
    {
        [FileExtension(".narp")]
        [FileExtension(".nrp")]
        Project,

        [FileExtension(".nmeta")]
        [FileExtension(".meta")]
        AppMeta,

        [FileExtension(".nmeta")]
        [FileExtension(".meta")]
        AocMeta
    }

    public static class FileTypeHelper
    {
        public static IEnumerable<string> GetFileExtensions(this ImportableFileType fileType)
        {
            var extensions = fileType.GetType()
                .GetField(fileType.ToString())
                .GetCustomAttributes<FileExtensionAttribute>(false);

            Debug.Assert(extensions != null);

            return extensions.Select(x => x.FileExtension);
        }

        public static ImportableFileType? ToImportableFileType(string fileExtension)
        {
            if (string.IsNullOrEmpty(fileExtension))
                return null;
            var extension = fileExtension.ToLowerInvariant();
            foreach (var fileType in Enum
                .GetValues(typeof(ImportableFileType))
                .Cast<ImportableFileType>()
                .Where(x => x.GetFileExtensions().Contains(extension)))
            {
                return fileType;
            }
            return null;
        }

        public static ImportableFileType ToImportableFileType(this ExportableFileType src)
        {
            switch (src)
            {
                case ExportableFileType.Project:
                    return ImportableFileType.Project;

                case ExportableFileType.AppMeta:
                case ExportableFileType.AocMeta:
                    return ImportableFileType.Meta;

                default:
                    throw new ArgumentOutOfRangeException(nameof(src), src, null);
            }
        }

        public static ExportableFileType ToExportableFileType(this ImportableFileType src, AppModeType appMode)
        {
            switch (src)
            {
                case ImportableFileType.Project:
                    return ExportableFileType.Project;

                case ImportableFileType.Meta:
                    return appMode.IsAoc()
                        ? ExportableFileType.AocMeta
                        : ExportableFileType.AppMeta;

                case ImportableFileType.Nsp:
                    throw new NotSupportedException();

                default:
                    throw new ArgumentOutOfRangeException(nameof(src), src, null);
            }
        }
    }
}
