﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Configuration;
using System.IO;
using System.Linq;
using System.Xml.Linq;
using System.Xml.Serialization;
using System.Xml.XPath;
using BezelEditor.Foundation.Collections;
using BezelEditor.Mvvm;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using YamlDotNet.Serialization;

namespace Nintendo.Authoring.AuthoringEditor.Core
{
    public enum AccessibilityType
    {
        Read,
        Write,
        ReadWrite
    }

    public class SaveDataOwnerId : DisposableModelBase
    {
        // タイトルがアクセス可能な、他タイトルのセーブデータへのアクセス権設定
        #region Accessibility

        private AccessibilityType _Accessibility = AccessibilityType.Read;

        public AccessibilityType Accessibility
        {
            get { return _Accessibility; }
            set { SetProperty(ref _Accessibility, value); }
        }

        #endregion

        // タイトルがアクセス可能な、他タイトルのセーブデータに設定された ApplicationId
        #region Id

        private ulong _applicationId;

        [XmlIgnore]
        public ulong ApplicationId
        {
            get { return _applicationId; }
            set { SetProperty(ref _applicationId, value); }
        }

        [XmlElement("Id")]
        public string ApplicationIdHex
        {
            get { return ApplicationId.ToHex(); }
            set { ApplicationId = value.ToUlong(); }
        }

        #endregion
    }

    public class FsAccessControlData : PropertyTrackingDispoableModel<FsAccessControlData>
    {
        #region IsUseFlagPresets

        private bool _IsUseFlagPresets;

        [XmlIgnore]
        public bool IsUseFlagPresets
        {
            get { return _IsUseFlagPresets; }
            set { SetProperty(ref _IsUseFlagPresets, value); }
        }

        #endregion

        #region FlagPresets

        private string _FlagPresets;

        [YamlIgnore]
        [XmlElement("FlagPresets")]
        public string FlagPresetsForXml
        {
            get { return IsUseFlagPresets && string.IsNullOrEmpty(FlagPresets) == false ? FlagPresets : null; }
            set { FlagPresets = value; }
        }

        [XmlIgnore]
        public string FlagPresets
        {
            get { return _FlagPresets; }
            set { SetProperty(ref _FlagPresets, value); }
        }

        #endregion

        #region SaveDataOwnerIds

        private ObservableCollection<SaveDataOwnerId> _SaveDataOwnerIds = new DisposableObservableCollection<SaveDataOwnerId>();

        [XmlElement("SaveDataOwnerIds")]
        public ObservableCollection<SaveDataOwnerId> SaveDataOwnerIds
        {
            get { return _SaveDataOwnerIds; }
            set { SetProperty(ref _SaveDataOwnerIds, value); }
        }

        #endregion

        private static readonly string _InternalDefaultFlagPresets = "Debug";

        public static string GetDefaultFlagPresets()
        {
            if (File.Exists(NintendoSdkHelper.ApplicationDescFilePath) == false)
                return _InternalDefaultFlagPresets;
            var doc = XDocument.Load(NintendoSdkHelper.ApplicationDescFilePath);
            var defaultFlagPresets = doc.XPathSelectElement("/Desc/Default/FsAccessControlData/FlagPresets")?.Value;
            return defaultFlagPresets ?? _InternalDefaultFlagPresets;
        }

        public override IEnumerable<string> GetChangedPropertyNames()
        {
            if (IsPropertyChanged(nameof(FlagPresets)))
                yield return nameof(FlagPresets);
            if (!SaveDataOwnerIds.IsEmpty())
                yield return nameof(SaveDataOwnerIds);
        }

        public override void MergePropertyValues(FsAccessControlData value)
        {
            value.TrySetChangedProperty(x => x.FlagPresets, this);
            value.TrySetChangedProperty(x => x.IsUseFlagPresets, this);

            if (!value.SaveDataOwnerIds.IsEmpty())
            {
                SaveDataOwnerIds.Replace(value.SaveDataOwnerIds,
                    (x, y) => x.ApplicationId == y.ApplicationId,
                    (src, newValue) => src.Accessibility = newValue.Accessibility);
            }
        }
    }
}
