﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Collections.ObjectModel;
using System.Linq;
using System.Xml.Serialization;
using BezelEditor.Mvvm;
using Nintendo.Authoring.AuthoringEditor.Foundation;

namespace Nintendo.Authoring.AuthoringEditor.Core
{
    public class Content : DisposableModelBase
    {
        #region Type

        private ContentType _Type;

        public ContentType Type
        {
            get { return _Type; }
            set { SetProperty(ref _Type, value); }
        }

        #endregion

        #region Id

        private string _Id;

        public string Id
        {
            get { return _Id; }
            set { SetProperty(ref _Id, value); }
        }

        #endregion

        #region Size

        private ulong _Size;

        public ulong Size
        {
            get { return _Size; }
            set { SetProperty(ref _Size, value); }
        }

        #endregion

        #region Hash

        private string _Hash;

        public string Hash
        {
            get { return _Hash; }
            set { SetProperty(ref _Hash, value); }
        }
        #endregion
    }

    public class ContentMeta : DisposableModelBase
    {
        #region Type

        private ContentMetaType _Type;

        public ContentMetaType Type
        {
            get { return _Type; }
            set { SetProperty(ref _Type, value); }
        }

        #endregion

        #region Id

        private ulong _Id;

        [XmlIgnore]
        public ulong Id
        {
            get { return _Id; }
            set { SetProperty(ref _Id, value); }
        }

        [XmlElement("Id")]
        public string IdHex
        {
            get { return Id.ToHex(); }
            set { Id = value.ToUlong(); }
        }

        #endregion

        #region Version

        private uint _Version;

        public uint Version
        {
            get { return _Version; }
            set { SetProperty(ref _Version, value); }
        }

        #endregion

        #region Digest

        private string _Digest;

        public string Digest
        {
            get { return _Digest; }
            set { SetProperty(ref _Digest, value); }
        }
        #endregion

        #region Contents

        private ObservableCollection<Content> _Contents = new ObservableCollection<Content>();

        [XmlElement("Content")]
        public ObservableCollection<Content> Contents
        {
            get { return _Contents; }
            set { SetProperty(ref _Contents, value); }
        }

        #endregion

        public string GetContentId(ContentType type) => GetContent(type)?.Id;

        public Content GetContent(ContentType type) => Contents.FirstOrDefault(x => x.Type == type);
    }

    [XmlRoot("ContentMeta")]
    public class ApplicationContentMeta : ContentMeta
    {
        #region RequiredSystemVersion

        private uint _RequiredSystemVersion;

        public uint RequiredSystemVersion
        {
            get { return _RequiredSystemVersion; }
            set { SetProperty(ref _RequiredSystemVersion, value); }
        }

        #endregion

        #region RequiredDownloadSystemVersion

        private uint _RequiredDownloadSystemVersion;

        public uint RequiredDownloadSystemVersion
        {
            get { return _RequiredDownloadSystemVersion; }
            set { SetProperty(ref _RequiredDownloadSystemVersion, value); }
        }

        #endregion

        #region PatchId

        private ulong _PatchId;

        [XmlIgnore]
        public ulong PatchId
        {
            get { return _PatchId; }
            set { SetProperty(ref _PatchId, value); }
        }

        [XmlElement("PatchId")]
        public string PatchIdHex
        {
            get { return PatchId.ToHex(); }
            set { PatchId = value.ToUlong(); }
        }

        #endregion

        #region ReleaseVersion

        [XmlIgnore]
        public ushort ReleaseVersion => (ushort) (Version >> 16);

        #endregion
    }

    public class ContentMetaSizeProperty : DisposableModelBase
    {
        #region DownLoad

        private ulong _Download;

        public ulong DownLoad
        {
            get { return _Download; }
            set { SetProperty(ref _Download, value); }
        }

        #endregion

        #region MinimumSaveData

        private ulong _MinimumSaveData;

        public ulong MinimumSaveData
        {
            get { return _MinimumSaveData; }
            set { SetProperty(ref _MinimumSaveData, value); }
        }

        #endregion

        #region EachUserAccountSaveData

        private ulong _EachUserAccountSaveData;

        public ulong EachUserAccountSaveData
        {
            get { return _EachUserAccountSaveData; }
            set { SetProperty(ref _EachUserAccountSaveData, value); }
        }

        #endregion

        #region ApplicationAreaUsedOnCard

        private ulong _ApplicationAreaUsedOnCard;

        public ulong ApplicationAreaUsedOnCard
        {
            get { return _ApplicationAreaUsedOnCard; }
            set { SetProperty(ref _ApplicationAreaUsedOnCard, value); }
        }

        #endregion

        #region ApplicationAreaAvailableOnCard

        private ulong _ApplicationAreaAvailableOnCard;

        public ulong ApplicationAreaAvailableOnCard
        {
            get { return _ApplicationAreaAvailableOnCard; }
            set { SetProperty(ref _ApplicationAreaAvailableOnCard, value); }
        }

        #endregion
    }

    public class ContentMetaProperty : DisposableModelBase
    {
        #region Type

        private ContentMetaType _Type;

        public ContentMetaType Type
        {
            get { return _Type; }
            set { SetProperty(ref _Type, value); }
        }

        #endregion

        #region Id

        private ulong _Id;

        [XmlIgnore]
        public ulong Id
        {
            get { return _Id; }
            set { SetProperty(ref _Id, value); }
        }

        [XmlElement("Id")]
        public string IdHex
        {
            get { return Id.ToHex(); }
            set { Id = value.ToUlong(); }
        }

        #endregion

        #region Size

        private ContentMetaSizeProperty _Size;

        public ContentMetaSizeProperty Size
        {
            get { return _Size; }
            set { SetProperty(ref _Size, value); }
        }

        #endregion
    }

    public class ContentMetaProperties : DisposableModelBase
    {
        #region Properties

        private ObservableCollection<ContentMetaProperty> _Properties = new ObservableCollection<ContentMetaProperty>();

        [XmlElement("Property")]
        public ObservableCollection<ContentMetaProperty> Properties
        {
            get { return _Properties; }
            set { SetProperty(ref _Properties, value); }
        }

        #endregion
    }

    [XmlRoot("ContentMeta")]
    public class AocContentMeta : ContentMeta
    {
        #region RequiredDownloadSystemVersion

        private uint _RequiredDownloadSystemVersion;

        public uint RequiredDownloadSystemVersion
        {
            get { return _RequiredDownloadSystemVersion; }
            set { SetProperty(ref _RequiredDownloadSystemVersion, value); }
        }

        #endregion

        #region RequiredApplicationVersion

        private ulong _RequiredApplicationVersion;

        public ulong RequiredApplicationVersion
        {
            get { return _RequiredApplicationVersion; }
            set { SetProperty(ref _RequiredApplicationVersion, value); }
        }

        #endregion

        #region ApplicationId

        private ulong _ApplicationId;

        [XmlIgnore]
        public ulong ApplicationId
        {
            get { return _ApplicationId; }
            set { SetProperty(ref _ApplicationId, value); }
        }

        [XmlElement("ApplicationId")]
        public string ApplicationIdHex
        {
            get { return ApplicationId.ToHex(); }
            set { ApplicationId = value.ToUlong(); }
        }

        #endregion

        #region Tag

        private string _Tag;

        public string Tag
        {
            get { return _Tag; }
            set { SetProperty(ref _Tag, value); }
        }

        #endregion

        #region Index

        private ulong _Index;

        public ulong Index
        {
            get { return _Index; }
            set { SetProperty(ref _Index, value); }
        }

        #endregion
    }

    [XmlRoot("ContentMeta")]
    public class PatchContentMeta : ContentMeta
    {
        #region RequiredSystemVersion

        private uint _RequiredSystemVersion;

        public uint RequiredSystemVersion
        {
            get { return _RequiredSystemVersion; }
            set { SetProperty(ref _RequiredSystemVersion, value); }
        }

        #endregion

        #region ApplicationId

        private ulong _ApplicationId;

        [XmlIgnore]
        public ulong ApplicationId
        {
            get { return _ApplicationId; }
            set { SetProperty(ref _ApplicationId, value); }
        }

        [XmlElement("ApplicationId")]
        public string ApplicationIdHex
        {
            get { return ApplicationId.ToHex(); }
            set { ApplicationId = value.ToUlong(); }
        }

        #endregion

        #region History

        private ObservableCollection<ContentMeta> _History = new ObservableCollection<ContentMeta>();

        [XmlElement("History")]
        public ObservableCollection<ContentMeta> History
        {
            get { return _History; }
            set { SetProperty(ref _History, value); }
        }

        #endregion

        #region DeltaHistory

        private ObservableCollection<DeltaHistory> _DeltaHistory = new ObservableCollection<DeltaHistory>();

        [XmlElement("DeltaHistory")]
        public ObservableCollection<DeltaHistory> DeltaHistory
        {
            get { return _DeltaHistory; }
            set { SetProperty(ref _DeltaHistory, value); }
        }

        #endregion

        #region ReleaseVersion

        [XmlIgnore]
        public ushort ReleaseVersion => (ushort)(Version >> 16);

        #endregion

        #region OriginalApplication

        [XmlIgnore]
        public ContentMeta OriginalApplication => History
            .Where(x => x.Type == ContentMetaType.Application)
            .FirstOrDefault(x => (x.Version>>16) == 0);

        #endregion

        [XmlIgnore]
        public bool IsUsePatchDelta
        {
            get
            {
                var deltaDownloadSize = DeltaHistory.FirstOrDefault(x => x.Destination.Version == Version)?.DownloadSize;
                var directUpdateDownloadSize = 0ul;
                foreach (var content in Contents.Where(x => x.Type != ContentType.DeltaFragment))
                    directUpdateDownloadSize += content.Size;
                // パッチ間差分を使用する == 直接更新のダウンロードサイズの方が大きい
                return deltaDownloadSize < directUpdateDownloadSize;
            }
        }

        // 2.x.x NUP までのパッチ間更新使用の有無ロジック
        //
        // * 直接更新サイズ
        //   パッチに含まれる全 Content のサイズを合算したもの
        //
        // * パッチ間更新サイズ
        //   DeltaHistory の中で Destination がパッチバージョンと等しい DownloadSize + DeltaHistory の 0 番目(v1 -> v2) の DownloadSize を合算したもの
        [XmlIgnore]
        public bool IsUsePatchDeltaBefore3NUP
        {
            get
            {
                var deltaDownloadSize =
                    DeltaHistory.FirstOrDefault(x => x.Destination.Version == Version)?.DownloadSize +
                    DeltaHistory.First()?.DownloadSize;
                var directUpdateDownloadSize = 0ul;
                foreach (var content in Contents)
                    directUpdateDownloadSize += content.Size;
                // パッチ間差分を使用する == 直接更新のダウンロードサイズの方が大きい
                return deltaDownloadSize < directUpdateDownloadSize;
            }
        }

        // パッチ差分情報が nsp に含まれている時点で製品化処理済みとみなす
        [XmlIgnore]
        public bool IsProduction => DeltaHistory.IsEmpty() == false;
    }

    public class DeltaHistory : ViewModelBase
    {
        #region Source

        private PatchLocation _Source;

        public PatchLocation Source
        {
            get { return _Source; }
            set { SetProperty(ref _Source, value); }
        }

        #endregion

        #region Destination

        private PatchLocation _Destination;

        public PatchLocation Destination
        {
            get { return _Destination; }
            set { SetProperty(ref _Destination, value); }
        }

        #endregion

        #region DownloadSize

        private ulong _DownloadSize;

        public ulong DownloadSize
        {
            get { return _DownloadSize; }
            set { SetProperty(ref _DownloadSize, value); }
        }

        #endregion
    }

    public class PatchLocation : ViewModelBase
    {
        #region PatchId

        private ulong _PatchId;

        [XmlIgnore]
        public ulong PatchId
        {
            get { return _PatchId; }
            set { SetProperty(ref _PatchId, value); }
        }

        [XmlElement("PatchId")]
        public string PatchIdIdHex
        {
            get { return PatchId.ToHex(); }
            set { PatchId = value.ToUlong(); }
        }

        #endregion

        #region Version

        private uint _Version;

        public uint Version
        {
            get { return _Version; }
            set { SetProperty(ref _Version, value); }
        }

        #endregion

        [XmlIgnore]
        public ushort ReleaseVersion => (ushort) (Version >> 16);
    }
}
