﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Globalization;
using System.Linq;
using BezelEditor.Foundation;
using BezelEditor.Mvvm;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Reactive.Bindings.Extensions;

namespace Nintendo.Authoring.AuthoringEditor.Core
{
    public class Config : DisposableModelBase
    {
        #region MruProjects

        private ObservableCollection<MruItem> _mruProjects = new ObservableCollection<MruItem>();

        public ObservableCollection<MruItem> MruProjects
        {
            get { return _mruProjects; }
            set { SetProperty(ref _mruProjects, value); }
        }

        #endregion

        #region MruMetas

        private ObservableCollection<MruItem> _mruMetas = new ObservableCollection<MruItem>();

        public ObservableCollection<MruItem> MruMetas
        {
            get { return _mruMetas; }
            set { SetProperty(ref _mruMetas, value); }
        }

        #endregion

        #region MruNsps

        private ObservableCollection<MruItem> _mruNsps = new ObservableCollection<MruItem>();

        public ObservableCollection<MruItem> MruNsps
        {
            get { return _mruNsps; }
            set { SetProperty(ref _mruNsps, value); }
        }

        #endregion

        #region Culture

        private string _Culture;

        public string Culture
        {
            get
            {
                if (_Culture != null)
                    return _Culture;

                var name = CultureInfo.CurrentUICulture.Name;
                var isoName = CultureInfo.CurrentUICulture.TwoLetterISOLanguageName;

                if (Constants.SupportedCultures.Any(c => c.CultureName == name))
                    _Culture = name;
                else if (Constants.SupportedCultures.Any(c => c.CultureName == isoName))
                    _Culture = isoName;
                else
                    _Culture = "en";

                return _Culture;
            }
            set { SetProperty(ref _Culture, value); }
        }

        #endregion

        private const int MaxMruFileCount = 5;

        private IDisposable _mruProjectsObserver;
        private IDisposable _mruMetasObserver;
        private IDisposable _mruNspsObserver;

        private IEnumerable<IDisposable> AllMruObservers
        {
            get
            {
                yield return _mruProjectsObserver;
                yield return _mruMetasObserver;
                yield return _mruNspsObserver;
            }
        }

        public string FirstOrDefault(ImportableFileType targetType, PathString filePath)
        {
            var target = TargetMruList(targetType);

            return target.FirstOrDefault(x => x.FilePath == filePath)?.FilePath;
        }

        public void AddToMruList(ImportableFileType targetType, PathString filePath)
        {
            var target = TargetMruList(targetType);

            AddToMruInternal(target, filePath);
            NormalizeMruList(target);
        }

        public void RemoveFromMruList(ImportableFileType targetType, PathString filePath)
        {
            var target = TargetMruList(targetType);

            var f = target.FirstOrDefault(x => x.FilePath == filePath);

            target.Remove(f);
        }

        public ObservableCollection<MruItem> TargetMruList(ImportableFileType targetType)
        {
            switch (targetType)
            {
                case ImportableFileType.Project:
                    return MruProjects;

                case ImportableFileType.Meta:
                    return MruMetas;

                case ImportableFileType.Nsp:
                    return MruNsps;

                default:
                    throw new ArgumentOutOfRangeException(nameof(targetType), targetType, null);
            }
        }

        private static void AddToMruInternal(ObservableCollection<MruItem> mruList, PathString filePath)
        {
            var found = mruList.FirstOrDefault(x => x.FilePath.ToPathString() == filePath);
            var isFoundPinned = found?.IsPinned ?? false;

            var removeItems = mruList.Where(x => x.FilePath.ToPathString() == filePath).ToArray();

            foreach (var i in removeItems)
                mruList.Remove(i);

            mruList.Insert(0, new MruItem {FilePath = filePath, IsPinned = isFoundPinned});
        }

        private static void NormalizeMruList(ObservableCollection<MruItem> mruList)
        {
            var items = mruList.ToArray();

            mruList.Clear();

            items
                .Where(i => i.IsPinned)
                .Concat(items.Where(i => i.IsPinned == false))
                .Take(MaxMruFileCount)
                .ForEach(mruList.Add);
        }

        public Config()
        {
            CompositeDisposable.Add(() => AllMruObservers.ForEach(x => x?.Dispose()));

            this.ObserveProperty(x => x.MruProjects)
                .Subscribe(_ => SetupMruObserver(ref _mruProjectsObserver, MruProjects))
                .AddTo(CompositeDisposable);

            this.ObserveProperty(x => x.MruMetas)
                .Subscribe(_ => SetupMruObserver(ref _mruMetasObserver, MruMetas))
                .AddTo(CompositeDisposable);

            this.ObserveProperty(x => x.MruNsps)
                .Subscribe(_ => SetupMruObserver(ref _mruNspsObserver, MruNsps))
                .AddTo(CompositeDisposable);
        }

        private void SetupMruObserver(ref IDisposable observer, ObservableCollection<MruItem> mru)
        {
            observer?.Dispose();

            if (mru != null)
                observer = mru.ObserveElementPropertyChanged().Subscribe(_ => NormalizeMruList(mru));
        }

        public void Initialize()
        {
            // 何もしない
        }

        public void Destory(string configFilePath)
        {
            YamlHelper.Save(configFilePath, this);
        }
    }
}
