﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Drawing.Imaging;
using System.IO;
using System.Linq;
using System.Reactive;
using System.Reactive.Linq;
using System.Text.RegularExpressions;
using System.Xml.Serialization;
using BezelEditor.Foundation.Extentions;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Reactive.Bindings.Extensions;
using YamlDotNet.Serialization;

namespace Nintendo.Authoring.AuthoringEditor.Core
{
    public partial class ApplicationMeta
    {
        #region HasErrors

        private bool _hasErrors;

        [YamlIgnore]
        [XmlIgnore]
        public bool HasErrors
        {
            get { return _hasErrors; }
            set { SetProperty(ref _hasErrors, value); }
        }

        #endregion

        [YamlIgnore]
        [XmlIgnore]
        public bool HasErrorNonPublicProperties => Application.HasErrorNonPublicProperties ||
                                                   Core.HasErrorNonPublicProperties;

        private void InitializeValidation()
        {
            CompositeDisposable.Add(() => _hasErrorObserver?.Dispose());
            this.ObserveProperty(x => x.Application)
                .Subscribe(_ => UpdateHasErrorObserver())
                .AddTo(CompositeDisposable);
        }

        private IDisposable _hasErrorObserver;

        private void UpdateHasErrorObserver()
        {
            _hasErrorObserver?.Dispose();

            _hasErrorObserver = Observable
                .Merge(this.ObserveProperty(x => x.DiContainer).ToUnit())
                .Merge(Application.ObserveProperty(x => x.IsErrorRatings).ToUnit())
                .Merge(Application.ObserveProperty(x => x.IsErrorTitles).ToUnit())
                .Merge(Application.ObserveProperty(x => x.IsErrorSupportedLanguages).ToUnit())
                .Merge(Application.ObserveProperty(x => x.IsErrorUserSaveDataOperation).ToUnit())
                .Merge(Application.ObserveProperty(x => x.ValidationApplicationErrorCodeCategory).ToUnit())
                .Merge(Application.ObserveProperty(x => x.ValidationDisplayVersion).ToUnit())
                .Merge(Application.ObserveProperty(x => x.ValidationSupportedLanguages).ToUnit())
                .Merge(Application.ObserveProperty(x => x.ValidationHtmlDocumentPath).ToUnit())
                .Merge(Application.ObserveProperty(x => x.ValidationAccessibleUrlsFilePath).ToUnit())
                .Merge(Application.ObserveProperty(x => x.ValidationLegalInformationFilePath).ToUnit())
                .Merge(Application.ObserveProperty(x => x.ValidationBcatDeliveryCacheStorageSize).ToUnit())
                .Merge(Application.ObserveProperty(x => x.ValidationBcatPassphrase).ToUnit())
                .Merge(Application.ObserveProperty(x => x.ValidationSaveDataSizeMax).ToUnit())
                .Merge(Application.ObserveProperty(x => x.ValidationSaveDataJournalSizeMax).ToUnit())
                .Merge(Application.ObserveProperty(x => x.ValidationDeviceSaveDataSizeMax).ToUnit())
                .Merge(Application.ObserveProperty(x => x.ValidationDeviceSaveDataJournalSizeMax).ToUnit())
                .Merge(Application.ObserveProperty(x => x.ValidationFilterDescriptionFilePath).ToUnit())
                .Merge(Core.ObserveProperty(x => x.ValidationApplicationId).ToUnit())
                .Merge(Core.ObserveProperty(x => x.ValidationMainThreadStackSize).ToUnit())
                .Merge(Core.ObserveProperty(x => x.ValidationSystemResourceSize).ToUnit())
                .Merge(CardSpec.ObserveProperty(x => x.ValidationSize).ToUnit())
                .Merge(CardSpec.ObserveProperty(x => x.ValidationClockRate).ToUnit())
                .Subscribe(_ =>
                {
                    if (DiContainer == null)
                    {
                        HasErrors = true;
                        return;
                    }

                    HasErrors =
                        Application.IsErrorRatings ||
                        Application.IsErrorTitles ||
                        Application.IsErrorSupportedLanguages ||
                        Application.IsErrorUserSaveDataOperation ||
                        Application.ValidationApplicationErrorCodeCategory != Application.ApplicationErrorCodeCategoryValidationType.Ok ||
                        Application.ValidationDisplayVersion != Application.DisplayVersionValidationType.Ok ||
                        //
                        Application.ValidationHtmlDocumentPath != Application.HtmlDocumentPathValidationType.Ok ||
                        Application.ValidationAccessibleUrlsFilePath != Application.AccessibleUrlsFilePathValidationType.Ok ||
                        Application.ValidationLegalInformationFilePath != Application.LegalInformationFilePathValidationType.Ok ||
                        Application.ValidationFilterDescriptionFilePath != Application.FilterDescriptionFileValidationType.Ok ||
                        //
                        Application.ValidationBcatDeliveryCacheStorageSize != Application.BcatDeliveryCacheStorageSizeValidationType.Ok ||
                        Application.ValidationBcatPassphrase != Application.BcatPassphraseValidationType.Ok ||
                        //
                        Application.ValidationSaveDataSizeMax != Application.SaveDataSizeMaxValidationType.Ok ||
                        Application.ValidationSaveDataJournalSizeMax != Application.SaveDataSizeMaxValidationType.Ok ||
                        Application.ValidationDeviceSaveDataSizeMax != Application.SaveDataSizeMaxValidationType.Ok ||
                        Application.ValidationDeviceSaveDataJournalSizeMax != Application.SaveDataSizeMaxValidationType.Ok ||
                        //
                        Application.ValidationSupportedLanguages != Application.SupportedLanguagesValidationType.Ok ||
                        //
                        Core.ValidationApplicationId != Core.ApplicationIdValidationType.Ok ||
                        Core.ValidationMainThreadStackSize != Core.MainThreadStackSizeValidationType.Ok ||
                        Core.ValidationSystemResourceSize != Core.SystemResourceSizeValidationType.Ok ||
                        //
                        CardSpec.ValidationSize != CardSpec.SizeValidationType.Ok ||
                        CardSpec.ValidationClockRate != CardSpec.ClockRateValidationType.Ok;
                });
        }
    }

    public partial class Core
    {
        private void InitializeValidation()
        {
            // ReSharper disable ExplicitCallerInfoArgument
            this.ObserveProperty(x => x.MainThreadStackSize)
                .Subscribe(_ => RaisePropertyChanged(nameof(ValidationMainThreadStackSize)))
                .AddTo(CompositeDisposable);
            Observable
                .Merge(this.ObserveProperty(x => x.SystemResourceSize).ToUnit())
                .Merge(this.ObserveProperty(x => x.ProcessAddressSpace).ToUnit())
                .Subscribe(_ => RaisePropertyChanged(nameof(ValidationSystemResourceSize)))
                .AddTo(CompositeDisposable);
            // ReSharper restore ExplicitCallerInfoArgument
        }

        #region ApplicationId

        [YamlIgnore]
        [XmlIgnore]
        public ApplicationIdValidationType ValidationApplicationId
            => ValidateApplicationId(ApplicationId);

        public static ApplicationIdValidationType ValidateApplicationId(ulong size)
        {
            return
                size >= Constants.ProgramIdMinimum && size <= Constants.ProgramIdMaximum
                    ? ApplicationIdValidationType.Ok
                    : ApplicationIdValidationType.RangeOver;
        }

        public enum ApplicationIdValidationType
        {
            Ok,
            RangeOver
        }

        #endregion

        #region MainThreadStackSize

        [YamlIgnore]
        [XmlIgnore]
        public MainThreadStackSizeValidationType ValidationMainThreadStackSize
            => ValidateMainThreadStackSize(MainThreadStackSize);

        public static MainThreadStackSizeValidationType ValidateMainThreadStackSize(ulong size)
        {
            if (size == 0)
                return MainThreadStackSizeValidationType.Empty;

            if ((size & (4 * 1024 - 1)) != 0)
                return MainThreadStackSizeValidationType.AlignError;

            return MainThreadStackSizeValidationType.Ok;
        }

        public enum MainThreadStackSizeValidationType
        {
            Ok,
            AlignError,
            Empty
        }

        #endregion

        #region SystemResourceSize

        [YamlIgnore]
        [XmlIgnore]
        public SystemResourceSizeValidationType ValidationSystemResourceSize
            => ValidateSystemResourceSize(SystemResourceSize, IsProcessAddressSpace64Bit);

        public static SystemResourceSizeValidationType ValidateSystemResourceSize(uint size, bool isProcessSpace64Bit)
        {
            if (isProcessSpace64Bit == false)
                return SystemResourceSizeValidationType.Ok;
            if ((size & (0x200000 - 1)) != 0)
                return SystemResourceSizeValidationType.AlignError;
            return SystemResourceSizeValidationType.Ok;
        }

        public enum SystemResourceSizeValidationType
        {
            Ok,
            AlignError
        }

        #endregion

        [YamlIgnore]
        [XmlIgnore]
        public bool HasErrorNonPublicProperties => false;
    }

    public partial class Application
    {
        #region DisplayVersion

        [YamlIgnore]
        [XmlIgnore]
        public DisplayVersionValidationType ValidationDisplayVersion => ValidateDisplayVersion(DisplayVersion);

        public static DisplayVersionValidationType ValidateDisplayVersion(string displayVersion)
        {
            if (string.IsNullOrEmpty(displayVersion))
                return DisplayVersionValidationType.Empty;

            if (displayVersion.Length > MaxDisplayVersionLength)
                return DisplayVersionValidationType.MaxLengthError;

            if (Regex.IsMatch(displayVersion, "^[a-zA-Z0-9!-/:-@¥[-`{-~]+$") == false)
                return DisplayVersionValidationType.CharaError;

            return DisplayVersionValidationType.Ok;
        }

        public const int MaxDisplayVersionLength = 15;

        public enum DisplayVersionValidationType
        {
            Ok,
            Empty,
            MaxLengthError,
            CharaError,
        }

        #endregion

        #region SupportedLanguages

        [YamlIgnore]
        [XmlIgnore]
        public SupportedLanguagesValidationType ValidationSupportedLanguages
            => ValidateSupportedLanguages(SupportedLanguages);

        public static SupportedLanguagesValidationType ValidateSupportedLanguages(
            ObservableCollection<SupportedLanguage> supportedLanguages)
        {
            if (supportedLanguages.Where(x => x.IsSupported).IsEmpty())
                return SupportedLanguagesValidationType.EmptyError;

            return SupportedLanguagesValidationType.Ok;
        }

        public enum SupportedLanguagesValidationType
        {
            Ok,
            EmptyError
        }

        #endregion

        #region ApplicationErrorCodeCategory

        [YamlIgnore]
        [XmlIgnore]
        public ApplicationErrorCodeCategoryValidationType ValidationApplicationErrorCodeCategory =>
            ValidateApplicationErrorCodeCategory(IsUseApplicationErrorCode, ApplicationErrorCodeCategory);

        public static ApplicationErrorCodeCategoryValidationType ValidateApplicationErrorCodeCategory(
            bool isUseApplicationErrorCode,
            string applicationErrorCodeCategory)
        {
            if (isUseApplicationErrorCode == false)
                return ApplicationErrorCodeCategoryValidationType.Ok;

            if (string.IsNullOrEmpty(applicationErrorCodeCategory))
                return ApplicationErrorCodeCategoryValidationType.EmptyError;

            return applicationErrorCodeCategory.Length == MaxApplicationErrorCodeCategoryLength
                ? ApplicationErrorCodeCategoryValidationType.Ok
                : ApplicationErrorCodeCategoryValidationType.LengthError;
        }

        public const int MaxApplicationErrorCodeCategoryLength = 5;

        public enum ApplicationErrorCodeCategoryValidationType
        {
            Ok,
            EmptyError,
            LengthError
        }

        #endregion

        #region Title

        [YamlIgnore]
        [XmlIgnore]
        public TitleValidationType ValidationTitle => ValidateTitle(Titles);

        public static TitleValidationType ValidateTitle(ObservableCollection<Title> titles)
        {
            if (titles.IsEmpty())
                return TitleValidationType.EmptyError;

            return TitleValidationType.Ok;
        }

        public enum TitleValidationType
        {
            Ok,
            EmptyError,
        }

        #endregion

        #region SaveDataSize

        [YamlIgnore]
        [XmlIgnore]
        public SaveDataSizeValidationType ValidationSaveDataSize =>
            ValidateSaveDataSize(IsUseSaveData, SaveDataSize);

        public static SaveDataSizeValidationType ValidateSaveDataSize(bool isUse, long size)
        {
            if (isUse == false)
                return SaveDataSizeValidationType.Ok;

            if (size < 16 * 1024)
                return SaveDataSizeValidationType.SizeError;

            if ((size & (16 * 1024 - 1)) != 0)
                return SaveDataSizeValidationType.AlignError;

            return SaveDataSizeValidationType.Ok;
        }

        public enum SaveDataSizeValidationType
        {
            Ok,
            SizeError,
            AlignError
        }

        #endregion

        #region SaveDataJournalSize

        [YamlIgnore]
        [XmlIgnore]
        public SaveDataJournalSizeValidationType ValidationSaveDataJournalSize =>
            ValidateSaveDataJournalSize(IsSpecifiedSaveDataJournal, SaveDataJournalSize);

        public static SaveDataJournalSizeValidationType ValidateSaveDataJournalSize(bool isUse, long size)
        {
            if (isUse == false)
                return SaveDataJournalSizeValidationType.Ok;

            if (size < 16 * 1024)
                return SaveDataJournalSizeValidationType.SizeError;

            if ((size & (16 * 1024 - 1)) != 0)
                return SaveDataJournalSizeValidationType.AlignError;

            return SaveDataJournalSizeValidationType.Ok;
        }

        public enum SaveDataJournalSizeValidationType
        {
            Ok,
            AlignError,
            SizeError
        }

        #endregion

        #region SaveDataSizeMax

        [YamlIgnore]
        [XmlIgnore]
        public SaveDataSizeMaxValidationType ValidationSaveDataSizeMax =>
            ValidateSaveDataSizeMax(IsUseUserAccountSaveDataSizeMax, UserAccountSaveDataSizeMax);

        [YamlIgnore]
        [XmlIgnore]
        public SaveDataSizeMaxValidationType ValidationSaveDataJournalSizeMax =>
            ValidateSaveDataSizeMax(IsUseUserAccountSaveDataJournalSizeMax, UserAccountSaveDataJournalSizeMax);

        [YamlIgnore]
        [XmlIgnore]
        public SaveDataSizeMaxValidationType ValidationDeviceSaveDataSizeMax =>
            ValidateSaveDataSizeMax(IsUseDeviceSaveDataSizeMax, DeviceSaveDataSizeMax);

        [YamlIgnore]
        [XmlIgnore]
        public SaveDataSizeMaxValidationType ValidationDeviceSaveDataJournalSizeMax =>
            ValidateSaveDataSizeMax(IsUseDeviceSaveDataJournalSizeMax, DeviceSaveDataJournalSizeMax);

        public static SaveDataSizeMaxValidationType ValidateSaveDataSizeMax(bool isUse, long size)
        {
            if (isUse == false)
                return SaveDataSizeMaxValidationType.Ok;

            const long MiB = 1024 * 1024;

            if (size < 1 * MiB)
                return SaveDataSizeMaxValidationType.SizeError;

            if ((size & (1 * MiB - 1)) != 0)
                return SaveDataSizeMaxValidationType.AlignError;

            return SaveDataSizeMaxValidationType.Ok;
        }

        public enum SaveDataSizeMaxValidationType
        {
            Ok,
            AlignError,
            SizeError
        }

        #endregion

        #region TemporaryStorageSize

        public static TemporaryStorageSizeValidationType ValidateTemporaryStorageSize(bool isUse, long size)
        {
            if (isUse == false)
                return TemporaryStorageSizeValidationType.Ok;

            if (size < 1024 * 16)
                return TemporaryStorageSizeValidationType.MinSizeError;

            if ((size & (1024 * 16 - 1)) != 0)
                return TemporaryStorageSizeValidationType.AlignError;

            const long MiB = 1024 * 1024;

            if (size > 4000 * MiB)
                return TemporaryStorageSizeValidationType.MaxSizeError;

            return TemporaryStorageSizeValidationType.Ok;
        }

        public enum TemporaryStorageSizeValidationType
        {
            Ok,
            AlignError,
            MinSizeError,
            MaxSizeError
        }

        #endregion

        #region CacheStorageSize

        public static CacheStorageSizeValidationType ValidateCacheStorageSize(bool isUse, long size)
        {
            if (isUse == false)
                return CacheStorageSizeValidationType.Ok;

            if (size < 1024 * 16)
                return CacheStorageSizeValidationType.SizeError;

            if ((size & (1024 * 16 - 1)) != 0)
                return CacheStorageSizeValidationType.AlignError;

            return CacheStorageSizeValidationType.Ok;
        }

        public enum CacheStorageSizeValidationType
        {
            Ok,
            AlignError,
            SizeError
        }

        #endregion

        #region IsErrorTitles

        private bool _IsErrorTitles;

        [YamlIgnore]
        [XmlIgnore]
        public bool IsErrorTitles
        {
            get { return _IsErrorTitles; }
            set { SetProperty(ref _IsErrorTitles, value); }
        }

        #endregion

        #region IsErrorRatings

        private bool _IsErrorRatings;

        [YamlIgnore]
        [XmlIgnore]
        public bool IsErrorRatings
        {
            get { return _IsErrorRatings; }
            set { SetProperty(ref _IsErrorRatings, value); }
        }

        #endregion

        #region IsErrorSupportedLanguages

        private bool _IsErrorSupportedLanguages;

        [YamlIgnore]
        [XmlIgnore]
        public bool IsErrorSupportedLanguages
        {
            get { return _IsErrorSupportedLanguages; }
            set { SetProperty(ref _IsErrorSupportedLanguages, value); }
        }

        #endregion

        #region DeviceSaveDataSize

        [YamlIgnore]
        [XmlIgnore]
        public SaveDataSizeValidationType ValidationDeviceSaveDataSize =>
            ValidateSaveDataSize(IsSpecifiedDeviceSaveDataSize, DeviceSaveDataSize);

        #endregion

        #region DeviceSaveDataJournalSize

        [YamlIgnore]
        [XmlIgnore]
        public SaveDataJournalSizeValidationType ValidationDeviceSaveDataJournalSize =>
            ValidateSaveDataJournalSize(IsSpecifiedDeviceSaveDataJournalSize, DeviceSaveDataJournalSize);

        #endregion

        #region HtmlDocumentDirectoryPath

        [YamlIgnore]
        [XmlIgnore]
        public HtmlDocumentPathValidationType ValidationHtmlDocumentPath
            => ValidateHtmlDocumentPath(
                IsReplaceHtmlDocumentPath,
                DiContainer?.GetInstance<Project>().ToAbsolutePath(HtmlDocumentPath.Path),
                HtmlDocumentPath.IsExpandEnvironmentVariable);

        public HtmlDocumentPathValidationType ValidateHtmlDocumentPath(bool isReplace, string path, bool isExpandEnvironmentVariable)
        {
            if (isReplace == false)
                return HtmlDocumentPathValidationType.Ok;

            if (isExpandEnvironmentVariable)
                return HtmlDocumentPathValidationType.Ok;

            if (string.IsNullOrEmpty(path))
                return HtmlDocumentPathValidationType.PathIsEmpty;

            if (Directory.Exists(path) == false)
                return HtmlDocumentPathValidationType.DirectoryNotFound;

            return HtmlDocumentPathValidationType.Ok;
        }

        public enum HtmlDocumentPathValidationType
        {
            Ok,
            DirectoryNotFound,
            PathIsEmpty
        }

        #endregion

        #region AccessibleUrlsFilePath

        [YamlIgnore]
        [XmlIgnore]
        public AccessibleUrlsFilePathValidationType ValidationAccessibleUrlsFilePath
            => ValidateAccessibleUrlsFilePath(
                IsReplaceAccessibleUrlsFilePath,
                DiContainer?.GetInstance<Project>().ToAbsolutePath(AccessibleUrlsFilePath.Path),
                AccessibleUrlsFilePath.IsExpandEnvironmentVariable);

        public AccessibleUrlsFilePathValidationType ValidateAccessibleUrlsFilePath(bool isReplace, string path, bool isExpandEnvironmentVaraible)
        {
            if (isReplace == false)
                return AccessibleUrlsFilePathValidationType.Ok;

            if (isExpandEnvironmentVaraible)
                return AccessibleUrlsFilePathValidationType.Ok;

            if (string.IsNullOrEmpty(path))
                return AccessibleUrlsFilePathValidationType.PathIsEmpty;

            if (File.Exists(path) == false)
                return AccessibleUrlsFilePathValidationType.FileNotFound;

            return AccessibleUrlsFilePathValidationType.Ok;
        }

        public enum AccessibleUrlsFilePathValidationType
        {
            Ok,
            FileNotFound,
            PathIsEmpty
        }

        #endregion

        #region LegalInformationFilePath

        [YamlIgnore]
        [XmlIgnore]
        public LegalInformationFilePathValidationType ValidationLegalInformationFilePath
            => ValidateLegalInformationFilePath(
                IsReplaceLegalInformationFilePath,
                DiContainer?.GetInstance<Project>().ToAbsolutePath(LegalInformationFilePath.Path),
                LegalInformationFilePath.IsExpandEnvironmentVariable);

        public LegalInformationFilePathValidationType ValidateLegalInformationFilePath(
            bool isReplace, string path, bool isExpandEnvironmentVaraible)
        {
            if (isReplace == false)
                return LegalInformationFilePathValidationType.Ok;

            if (isExpandEnvironmentVaraible)
                return LegalInformationFilePathValidationType.Ok;

            if (string.IsNullOrEmpty(path))
                return LegalInformationFilePathValidationType.EmptyError;

            if (File.Exists(path) == false)
                return LegalInformationFilePathValidationType.FileNotFound;

            return LegalInformationFilePathValidationType.Ok;
        }

        public enum LegalInformationFilePathValidationType
        {
            Ok,
            EmptyError,
            FileNotFound,
        }

        #endregion

        #region BcatDeliveryCacheStorageSize

        [YamlIgnore]
        [XmlIgnore]
        public BcatDeliveryCacheStorageSizeValidationType ValidationBcatDeliveryCacheStorageSize =>
            ValidateBcatDeliveryCacheStorageSize(BcatDeliveryCacheStorageSize, IsUseBcat);

        public static BcatDeliveryCacheStorageSizeValidationType ValidateBcatDeliveryCacheStorageSize(long size, bool isUseBcat)
        {
            if (isUseBcat == false)
                return BcatDeliveryCacheStorageSizeValidationType.Ok;
            if ((size & (1024*1024 - 1)) != 0)
                return BcatDeliveryCacheStorageSizeValidationType.AlignError;
            if ((size >= 1024*1024*5 && size <= 1024*1024*64) == false)
                return BcatDeliveryCacheStorageSizeValidationType.SizeError;
            return BcatDeliveryCacheStorageSizeValidationType.Ok;
        }

        public enum BcatDeliveryCacheStorageSizeValidationType
        {
            Ok,
            AlignError,
            SizeError
        }

        #endregion

        #region BcatPassphrase

        [YamlIgnore]
        [XmlIgnore]
        public BcatPassphraseValidationType ValidationBcatPassphrase =>
            ValidateBcatPassphrase(BcatPassphrase, IsUseBcat);

        public const int BcatPassphraseLength = 64;

        public static BcatPassphraseValidationType ValidateBcatPassphrase(string passPhrase, bool isUseBcat)
        {
            if (isUseBcat == false)
                return BcatPassphraseValidationType.Ok;
            if (string.IsNullOrEmpty(passPhrase))
                return BcatPassphraseValidationType.FormatError;
            if (Regex.IsMatch(passPhrase, "^[a-f0-9]{" + BcatPassphraseLength + "}$") == false)
                return BcatPassphraseValidationType.FormatError;
            return BcatPassphraseValidationType.Ok;
        }

        public enum BcatPassphraseValidationType
        {
            Ok,
            FormatError,
        }

        #endregion

        #region IsErrorUserSaveDataOperation

        private bool _IsErrorUserSaveDataOperation;

        [YamlIgnore]
        [XmlIgnore]
        public bool IsErrorUserSaveDataOperation
        {
            get { return _IsErrorUserSaveDataOperation; }
            set { SetProperty(ref _IsErrorUserSaveDataOperation, value); }
        }

        #endregion

        #region FilterDescriptionFilePath

        [YamlIgnore]
        [XmlIgnore]
        public FilterDescriptionFileValidationType ValidationFilterDescriptionFilePath
            => ValidateFilterDescriptionFilePath(
                IsUseFilterDescriptionFilePath,
                DiContainer?.GetInstance<Project>().ToAbsolutePath(FilterDescriptionFilePath));

        public FilterDescriptionFileValidationType ValidateFilterDescriptionFilePath(bool isUse, string path)
        {
            if (isUse == false)
                return FilterDescriptionFileValidationType.Ok;

            if (string.IsNullOrEmpty(path))
                return FilterDescriptionFileValidationType.PathIsEmpty;

            if (File.Exists(path) == false)
                return FilterDescriptionFileValidationType.FileNotFound;

            return FilterDescriptionFileValidationType.Ok;
        }

        public enum FilterDescriptionFileValidationType
        {
            Ok,
            FileNotFound,
            PathIsEmpty
        }

        #endregion

        [YamlIgnore]
        [XmlIgnore]
        public bool HasErrorNonPublicProperties
        {
            get
            {
                if (Isbn?.Length > 36)
                    return true;

                if (SaveDataOwnerId?.IsUlong() == false)
                    return true;

                if (_IsInvalidDeviceDataSize)
                    return true;

                if (_IsInvalidDeviceDataJournalSize)
                    return true;

                return false;
            }
        }

        private void InitializeValidation()
        {
            // ReSharper disable ExplicitCallerInfoArgument
            Observable
                .Merge(this.ObserveProperty(x => x.IsUseApplicationErrorCode).ToUnit())
                .Merge(this.ObserveProperty(x => x.ApplicationErrorCodeCategory).ToUnit())
                .Subscribe(_ => RaisePropertyChanged(nameof(ValidationApplicationErrorCodeCategory)))
                .AddTo(CompositeDisposable);

            //////////////////

            Observable
                .Merge(this.ObserveProperty(x => x.HtmlDocumentPath).ToUnit())
                .Merge(this.ObserveProperty(x => x.IsReplaceHtmlDocumentPath).ToUnit())
                .Merge(HtmlDocumentPath.PropertyChangedAsObservable().ToUnit())
                .Subscribe(_ => RaisePropertyChanged(nameof(ValidationHtmlDocumentPath)))
                .AddTo(CompositeDisposable);

            Observable
                .Merge(this.ObserveProperty(x => x.AccessibleUrlsFilePath).ToUnit())
                .Merge(this.ObserveProperty(x => x.IsReplaceAccessibleUrlsFilePath).ToUnit())
                .Merge(AccessibleUrlsFilePath.PropertyChangedAsObservable().ToUnit())
                .Subscribe(_ => RaisePropertyChanged(nameof(ValidationAccessibleUrlsFilePath)))
                .AddTo(CompositeDisposable);

            Observable
                .Merge(this.ObserveProperty(x => x.LegalInformationFilePath).ToUnit())
                .Merge(this.ObserveProperty(x => x.IsReplaceLegalInformationFilePath).ToUnit())
                .Merge(LegalInformationFilePath.PropertyChangedAsObservable().ToUnit())
                .Subscribe(_ => RaisePropertyChanged(nameof(ValidationLegalInformationFilePath)))
                .AddTo(CompositeDisposable);

            Observable
                .Merge(this.ObserveProperty(x => x.FilterDescriptionFilePath).ToUnit())
                .Merge(this.ObserveProperty(x => x.IsUseFilterDescriptionFilePath).ToUnit())
                .Subscribe(_ => RaisePropertyChanged(nameof(ValidationFilterDescriptionFilePath)))
                .AddTo(CompositeDisposable);

            //////////////////
            this.ObserveProperty(x => x.DisplayVersion)
                .Subscribe(_ => RaisePropertyChanged(nameof(ValidationDisplayVersion)))
                .AddTo(CompositeDisposable);

            //////////////////
            this.ObserveProperty(x => x.IsUseBcat)
                .Subscribe(_ =>
                {
                    RaisePropertyChanged(nameof(ValidationBcatDeliveryCacheStorageSize));
                    RaisePropertyChanged(nameof(ValidationBcatPassphrase));
                })
                .AddTo(CompositeDisposable);

            this.ObserveProperty(x => x.BcatPassphrase)
                .Subscribe(_ => RaisePropertyChanged(nameof(ValidationBcatPassphrase)))
                .AddTo(CompositeDisposable);

            this.ObserveProperty(x => x.BcatDeliveryCacheStorageSize)
                .Subscribe(_ => RaisePropertyChanged(nameof(ValidationBcatDeliveryCacheStorageSize)))
                .AddTo(CompositeDisposable);
            // ReSharper restore ExplicitCallerInfoArgument

            CompositeDisposable.Add(() => _IsErrorTitlesObserver?.Dispose());
            CompositeDisposable.Add(() => _SupportedHasTitlesObserver?.Dispose());

            if (DiContainer != null)
                UpdateTitlesObservers();

            Observable
                .Merge(this.ObserveProperty(x => x.Titles).ToUnit())
                .Merge(this.ObserveProperty(x => x.SupportedLanguages).ToUnit())
                .Merge(this.ObserveProperty(x => x.DiContainer).ToUnit())
                .Subscribe(_ =>
                {
                    if (DiContainer != null)
                        UpdateTitlesObservers();
                })
                .AddTo(CompositeDisposable);

            CompositeDisposable.Add(() => _IsErrorRatingsObserver?.Dispose());
            UpdateIsErrorRatingsObserver();
            this.ObserveProperty(x => x.Ratings)
                .Subscribe(_ => UpdateIsErrorRatingsObserver())
                .AddTo(CompositeDisposable);

            CompositeDisposable.Add(() => _IsErrorSupportedLanguagesObserver?.Dispose());
            UpdateIsErrorSupportedLanguagesObserver();
            this.ObserveProperty(x => x.SupportedLanguages)
                .Subscribe(_ => UpdateIsErrorSupportedLanguagesObserver())
                .AddTo(CompositeDisposable);

            Observable
                .Merge(this.ObserveProperty(x => x.UserAccountSaveDataSizeMax).ToUnit())
                .Merge(this.ObserveProperty(x => x.IsUseUserAccountSaveDataSizeMax).ToUnit())
                .Subscribe(_ => RaisePropertyChanged(nameof(ValidationSaveDataSizeMax)))
                .AddTo(CompositeDisposable);

            Observable
                .Merge(this.ObserveProperty(x => x.UserAccountSaveDataJournalSizeMax).ToUnit())
                .Merge(this.ObserveProperty(x => x.IsUseUserAccountSaveDataJournalSizeMax).ToUnit())
                .Subscribe(_ => RaisePropertyChanged(nameof(ValidationSaveDataJournalSizeMax)))
                .AddTo(CompositeDisposable);

            Observable
                .Merge(this.ObserveProperty(x => x.DeviceSaveDataSizeMax).ToUnit())
                .Merge(this.ObserveProperty(x => x.IsUseDeviceSaveDataSizeMax).ToUnit())
                .Subscribe(_ => RaisePropertyChanged(nameof(ValidationDeviceSaveDataSizeMax)))
                .AddTo(CompositeDisposable);

            Observable
                .Merge(this.ObserveProperty(x => x.DeviceSaveDataJournalSizeMax).ToUnit())
                .Merge(this.ObserveProperty(x => x.IsUseDeviceSaveDataJournalSizeMax).ToUnit())
                .Subscribe(_ => RaisePropertyChanged(nameof(ValidationDeviceSaveDataJournalSizeMax)))
                .AddTo(CompositeDisposable);
        }

        private IDisposable _IsErrorTitlesObserver;
        private IDisposable _IsErrorRatingsObserver;
        private IDisposable _IsErrorSupportedLanguagesObserver;
        private IDisposable _SupportedHasTitlesObserver;

        private void UpdateTitlesObservers()
        {
            _IsErrorTitlesObserver?.Dispose();
            _SupportedHasTitlesObserver?.Dispose();

            _SupportedHasTitlesObserver = Observable
                .Merge(Titles.ObserveElementProperty(x => x.Language).ToUnit())
                .Merge(SupportedLanguages.ObserveElementProperty(x => x.IsSupported).ToUnit())
                .Subscribe(_ =>
                {
                    var supportedLanguages = SupportedLanguages.Where(x => x.IsSupported);
                    foreach (var s in supportedLanguages)
                    {
                        s.HasTitle = Titles.Any(x => x.Language == s.Language);
                    }
                });

            _IsErrorTitlesObserver = Observable
                .Merge(Titles.CollectionChangedAsObservable().ToUnit())
                .Merge(Titles.ObserveElementPropertyChanged().ToUnit())
                .Merge(Titles.ObserveElementProperty(x => x.IsForceValidation).ToUnit())
                .Merge(Observable.Return(Unit.Default))
                .Subscribe(_ =>
                {
                    {
                        var langs = new HashSet<LanguageType>();

                        foreach (var t in Titles)
                        {
                            if (langs.Contains(t.Language))
                            {
                                IsErrorTitles = true;
                                return;
                            }
                            langs.Add(t.Language);

                            if (t.HasErrors)
                            {
                                IsErrorTitles = true;
                                return;
                            }
                        }
                    }

                    if (ValidationTitle != TitleValidationType.Ok)
                    {
                        IsErrorTitles = true;
                        return;
                    }

                    IsErrorTitles = false;
                });
        }

        private void UpdateIsErrorRatingsObserver()
        {
            _IsErrorRatingsObserver?.Dispose();

            _IsErrorRatingsObserver = Observable
                .Merge(Ratings.CollectionChangedAsObservable().ToUnit())
                .Merge(Ratings.ObserveElementProperty(x => x.Organization).ToUnit())
                .Merge(Observable.Return(Unit.Default))
                .Subscribe(_ =>
                {
                    var organizations = new HashSet<string>();

                    foreach (var r in Ratings)
                    {
                        if (organizations.Contains(r.Organization))
                        {
                            IsErrorRatings = true;
                            return;
                        }
                        organizations.Add(r.Organization);
                    }

                    IsErrorRatings = false;
                });
        }

        private void UpdateIsErrorSupportedLanguagesObserver()
        {
            _IsErrorSupportedLanguagesObserver?.Dispose();

            _IsErrorSupportedLanguagesObserver = Observable
                .Merge(SupportedLanguages.CollectionChangedAsObservable().ToUnit())
                .Merge(SupportedLanguages.ObserveElementProperty(x => x.IsSupported).ToUnit())
                .Merge(Observable.Return(Unit.Default))
                .Subscribe(_ => IsErrorSupportedLanguages = SupportedLanguages.Where(x => x.IsSupported).IsEmpty());
        }
    }

    public partial class CardSpec
    {
        public static readonly int[] SizeRange = {1, 2, 4, 8, 16, 32};
        public static readonly int[] ClockRateRange = {25, 50};

        #region Size

        [YamlIgnore]
        [XmlIgnore]
        public SizeValidationType ValidationSize => ValidateSize(IsAutomaticSettingSize, Size);

        public static SizeValidationType ValidateSize(bool isAutomaticSetting, int size)
        {
            if (isAutomaticSetting)
                return SizeValidationType.Ok;

            if (size == 1)
            {
                return SizeValidationType.UnsupportedSizeError;
            }

            return SizeRange.Any(x => x == size)
                ? SizeValidationType.Ok
                : SizeValidationType.SizeError;
        }

        public enum SizeValidationType
        {
            Ok,
            SizeError,
            UnsupportedSizeError
        }

        #endregion

        #region ClockRate

        [YamlIgnore]
        [XmlIgnore]
        public ClockRateValidationType ValidationClockRate
            => ValidateClockRate(IsAutomaticSettingClockRate, ClockRate, Size);

        public static ClockRateValidationType ValidateClockRate(bool isAutomaticSetting, int rate, int size)
        {
            if (isAutomaticSetting)
                return ClockRateValidationType.Ok;

            if (size < 8)
            {
                if (rate == 25)
                    return ClockRateValidationType.Ok;

                if (rate == 50)
                    return ClockRateValidationType.Under8GB;

                return ClockRateValidationType.ClockRateError;
            }

            return ClockRateRange.Any(x => x == rate)
                ? ClockRateValidationType.Ok
                : ClockRateValidationType.ClockRateError;
        }

        public enum ClockRateValidationType
        {
            Ok,
            ClockRateError,
            Under8GB
        }

        #endregion

        private void InitializeValidation()
        {
            // ReSharper disable ExplicitCallerInfoArgument
            Observable
                .Merge(this.ObserveProperty(x => x.IsAutomaticSettingSize).ToUnit())
                .Merge(this.ObserveProperty(x => x.Size).ToUnit())
                .Subscribe(_ => RaisePropertyChanged(nameof(ValidationSize)))
                .AddTo(CompositeDisposable);

            Observable
                .Merge(this.ObserveProperty(x => x.IsAutomaticSettingClockRate).ToUnit())
                .Merge(this.ObserveProperty(x => x.ClockRate).ToUnit())
                .Subscribe(_ => RaisePropertyChanged(nameof(ValidationClockRate)))
                .AddTo(CompositeDisposable);
            // ReSharper restore ExplicitCallerInfoArgument
        }
    }

    public partial class Title
    {
        #region IconSize Constraints

        public static readonly int OffDeviceIconWidth = 1024;
        public static readonly int OffDeviceIconHeight = 1024;

        public static readonly int NxIconWidth = 256;
        public static readonly int NxIconHeight = 256;

        #endregion

        #region Name

        [YamlIgnore]
        [XmlIgnore]
        public NameValidationType ValidationName => ValidateName(Name);

        public static NameValidationType ValidateName(string name)
        {
            if (string.IsNullOrEmpty(name))
                return NameValidationType.EmptyError;

            return name.Length <= MaxNameLength
                ? NameValidationType.Ok
                : NameValidationType.LengthError;
        }

        public const int MaxNameLength = 127;

        public enum NameValidationType
        {
            Ok,
            EmptyError,
            LengthError
        }

        #endregion

        #region Publisher

        [YamlIgnore]
        [XmlIgnore]
        public PublisherValidationType ValidationPublisher => ValidatePublisher(Publisher);

        public static PublisherValidationType ValidatePublisher(string publisher)
        {
            if (string.IsNullOrEmpty(publisher))
                return PublisherValidationType.EmptyError;

            return publisher.Length <= MaxPublisherLength
                ? PublisherValidationType.Ok
                : PublisherValidationType.LengthError;
        }

        public const int MaxPublisherLength = 63;

        public enum PublisherValidationType
        {
            Ok,
            EmptyError,
            LengthError
        }

        #endregion

        #region IconFilePath

        [YamlIgnore]
        [XmlIgnore]
        public IconFilePathValidationType ValidationIconFilePath => ValidateIconFilePath(
            IsReplaceIcon,
            true,
            DiContainer?.GetInstance<Project>().ToAbsolutePath(IconFilePath.Path),
            IconFilePath.IsExpandEnvironmentVariable,
            OffDeviceIconWidth, OffDeviceIconHeight);

        [YamlIgnore]
        [XmlIgnore]
        public IconFilePathValidationType ValidationOriginalIconFilePath => ValidateIconFilePath(
            IsReplaceIcon == false,
            false,
            OriginalIconFilePath.Path,
            OriginalIconFilePath.IsExpandEnvironmentVariable,
            OffDeviceIconWidth, OffDeviceIconHeight);

        public IconFilePathValidationType ValidateIconFilePath(
            bool isReplace, bool isRawFormat,
            string iconFilePath,
            bool isExpandEnvironmentVariable,
            int width, int height)
        {
            if (isReplace == false)
                return IconFilePathValidationType.Ok;

            if (isExpandEnvironmentVariable)
                return IconFilePathValidationType.Ok;

            if (string.IsNullOrEmpty(iconFilePath))
                return IconFilePathValidationType.EmptyError;

            if (File.Exists(iconFilePath) == false)
                return IconFilePathValidationType.FileNotFound;

            try
            {
                using (var img = System.Drawing.Image.FromFile(iconFilePath))
                {
                    if (isRawFormat)
                    {
                        if (img.RawFormat.Equals(ImageFormat.Bmp) == false)
                            return IconFilePathValidationType.FormatError;

                        if (img.PixelFormat != PixelFormat.Format24bppRgb)
                            return IconFilePathValidationType.FormatError;
                    }
                    else
                    {
                        if (img.RawFormat.Equals(ImageFormat.Jpeg) == false)
                            return IconFilePathValidationType.FormatError;
                    }

                    if (img.Width != width || img.Height != height)
                        return IconFilePathValidationType.SizeError;
                }
            }
            catch
            {
                return IconFilePathValidationType.FormatError;
            }

            return IconFilePathValidationType.Ok;
        }

        public enum IconFilePathValidationType
        {
            Ok,
            EmptyError,
            FileNotFound,
            SizeError,
            FormatError
        }

        #endregion

        #region NxIconFilePath

        [YamlIgnore]
        [XmlIgnore]
        public NxIconFilePathValidationType ValidationNxIconFilePath => ValidateNxIconFilePath(
            IsReplaceNxIcon,
            NxIconFilePath.Path,
            NxIconFilePath.IsExpandEnvironmentVariable,
            NxIconWidth, NxIconHeight);

        public NxIconFilePathValidationType ValidateNxIconFilePath(
            bool isReplace,
            string nxIconFilePath,
            bool isExpandEnvironmentVariable,
            int nxIconWidth, int nxIconHeight)
        {
            if (isReplace == false)
                return NxIconFilePathValidationType.Ok;

            if (isExpandEnvironmentVariable)
                return NxIconFilePathValidationType.Ok;

            if (string.IsNullOrEmpty(nxIconFilePath))
                return NxIconFilePathValidationType.EmptyError;

            if (File.Exists(nxIconFilePath) == false)
                return NxIconFilePathValidationType.FileNotFound;

            try
            {
                using (var img = System.Drawing.Image.FromFile(nxIconFilePath))
                {
                    if (img.RawFormat.Equals(ImageFormat.Jpeg) == false)
                        return NxIconFilePathValidationType.FormatError;

                    if (img.Width != nxIconWidth || img.Height != nxIconHeight)
                        return NxIconFilePathValidationType.SizeError;

                    if (JpegHelper.IsProgressive(nxIconFilePath))
                        return NxIconFilePathValidationType.FormatError;

                    if (new FileInfo(nxIconFilePath).Length >= 100 * 1024)
                        return NxIconFilePathValidationType.FileSizeError;
                }
            }
            catch
            {
                return NxIconFilePathValidationType.FormatError;
            }

            return NxIconFilePathValidationType.Ok;
        }

        public enum NxIconFilePathValidationType
        {
            Ok,
            EmptyError,
            FileNotFound,
            SizeError,
            FileSizeError,
            FormatError
        }

        #endregion

        [YamlIgnore]
        [XmlIgnore]
        public bool HasErrors
        {
            get
            {
                if (ValidationName != NameValidationType.Ok)
                    return true;

                if (ValidationPublisher != PublisherValidationType.Ok)
                    return true;

                if (ValidationIconFilePath != IconFilePathValidationType.Ok)
                    return true;

                if (ValidationNxIconFilePath != NxIconFilePathValidationType.Ok)
                    return true;

                return false;
            }
        }
    }

    public partial class SaveDataOperation
    {
        public enum CopyValidationType
        {
            Ok,
            UnspecifiedNotAllowError
        }

        [YamlIgnore]
        [XmlIgnore]
        public CopyValidationType ValidationCopy => ValidateCopy(Copy);

        public static CopyValidationType ValidateCopy(CopyType copy)
        {
            if (copy == CopyType.Unspecified)
                return CopyValidationType.UnspecifiedNotAllowError;
            return CopyValidationType.Ok;
        }

        public enum RollbackValidationType
        {
            Ok,
            // Unspecified の状態を許容しない
            UnspecifiedNotAllowError,
            // RollbackTerm が 0 でも 24 でもない
            InvalidRollbackTermError,
            // (Rollback, RollbackTerm) = (Allow, 0 以外)
            CopyAllowedTermError
        }

        [YamlIgnore]
        [XmlIgnore]
        public RollbackValidationType ValidationRollback
            => ValidateRollback(Copy, Rollback, RollbackTerm);

        public static RollbackValidationType ValidateRollback(CopyType copy, RollbackType rollback, int rollbackTerm)
        {
            if (rollback == RollbackType.Deny)
                return RollbackValidationType.Ok; // 任天堂の窓口に要相談
            if (rollback == RollbackType.Unspecified)
                return RollbackValidationType.UnspecifiedNotAllowError;

            if (rollbackTerm != 0 && rollbackTerm != 24)
                return RollbackValidationType.InvalidRollbackTermError;
            if (copy == CopyType.Allow && rollbackTerm != 0)
                return RollbackValidationType.CopyAllowedTermError;

            return RollbackValidationType.Ok;
        }

        private void InitializeValidation()
        {
            Observable
                .Merge(this.ObserveProperty(x => x.Rollback).ToUnit())
                .Merge(this.ObserveProperty(x => x.RollbackTerm).ToUnit())
                // ReSharper disable once ExplicitCallerInfoArgument
                .Subscribe(_ => RaisePropertyChanged(nameof(ValidationRollback)))
                .AddTo(CompositeDisposable);
        }
    }
}
