﻿using System;
using BezelEditor.Mvvm;
using Reactive.Bindings.Extensions;

namespace Nintendo.Authoring.AuthoringEditor.Core
{
    [AttributeUsage(AttributeTargets.Field)]
    public class SpecificTitleBarAttribute : Attribute
    {
        public bool IsDark { get; }

        public SpecificTitleBarAttribute(bool isDark)
        {
            IsDark = isDark;
        }
    }

    public enum AppModeType
    {
        // ※メンバーを編集したら AppModeTypeExtension を確認してください

        [SpecificTitleBar(true)]
        Startup,

        // 編集  todo: Edit*** に名前を変更
        Project,
        ApplicationMeta,
        ApplicationNsp,
        AocMeta,
        AocNsp,
        PatchNsp,

        // 比較
        Comparison,
    }

    public static class AppModeTypeExtension
    {
        public static bool IsAoc(this AppModeType type)
        {
            return
                type == AppModeType.AocMeta ||
                type == AppModeType.AocNsp;
        }

        public static bool IsMeta(this AppModeType type)
        {
            return
                type == AppModeType.ApplicationMeta ||
                type == AppModeType.AocMeta;
        }

        public static bool IsNsp(this AppModeType type)
        {
            return
                type == AppModeType.ApplicationNsp ||
                type == AppModeType.PatchNsp;
        }
    }

    public class AppProfile : DisposableModelBase
    {
        ////////////////////////////////////////////////////////////////////////////////

        #region CanEditCardSpec

        private bool _CanEditCardSpec;

        public bool CanEditCardSpec
        {
            get { return _CanEditCardSpec; }
            set { SetProperty(ref _CanEditCardSpec, value); }
        }

        #endregion

        ////////////////////////////////////////////////////////////////////////////////

        #region IsEnableCorePageGroup

        private bool _IsEnableCorePageGroup;

        public bool IsEnableCorePageGroup
        {
            get { return _IsEnableCorePageGroup; }
            set { SetProperty(ref _IsEnableCorePageGroup, value); }
        }

        #endregion

        ////////////////////////////////////////////////////////////////////////////////

        #region CanImportProject

        private bool _CanImportProject;

        public bool CanImportProject
        {
            get { return _CanImportProject; }
            set { SetProperty(ref _CanImportProject, value); }
        }

        #endregion

        #region CanExportProject

        private bool _CanExportProject;

        public bool CanExportProject
        {
            get { return _CanExportProject; }
            set { SetProperty(ref _CanExportProject, value); }
        }

        #endregion

        #region CanImportMeta

        private bool _CanImportMeta;

        public bool CanImportMeta
        {
            get { return _CanImportMeta; }
            set { SetProperty(ref _CanImportMeta, value); }
        }

        #endregion

        #region CanSaveAsMeta

        private bool _CanSaveAsMeta;

        public bool CanSaveAsMeta
        {
            get { return _CanSaveAsMeta; }
            set { SetProperty(ref _CanSaveAsMeta, value); }
        }

        #endregion

        #region CanSaveMeta

        private bool _CanSaveMeta;

        public bool CanSaveMeta
        {
            get { return _CanSaveMeta; }
            set { SetProperty(ref _CanSaveMeta, value); }
        }

        #endregion

        #region CanImportNsp

        private bool _CanImportNsp;

        public bool CanImportNsp
        {
            get { return _CanImportNsp; }
            set { SetProperty(ref _CanImportNsp, value); }
        }

        #endregion

        #region CanSaveAsNsp

        private bool _CanSaveAsNsp;

        public bool CanSaveAsNsp
        {
            get { return _CanSaveAsNsp; }
            set { SetProperty(ref _CanSaveAsNsp, value); }
        }

        #endregion

        #region CanMakeNspPatch

        private bool _CanMakeNspPatch;

        public bool CanMakeNspPatch
        {
            get { return _CanMakeNspPatch; }
            set { SetProperty(ref _CanMakeNspPatch, value); }
        }

        #endregion

        #region CanCompare

        private bool _CanCompare;

        public bool CanCompare
        {
            get { return _CanCompare; }
            set { SetProperty(ref _CanCompare, value); }
        }

        #endregion

        #region CanShowNspEntries

        private bool _CanShowNspEntries;

        public bool CanShowNspEntries
        {
            get { return _CanShowNspEntries; }
            set { SetProperty(ref _CanShowNspEntries, value); }
        }

        #endregion

        #region CanImportPartialMeta

        private bool _CanImportPartialMeta;

        public bool CanImportPartialMeta
        {
            get { return _CanImportPartialMeta; }
            set { SetProperty(ref _CanImportPartialMeta, value); }
        }

        #endregion

        ////////////////////////////////////////////////////////////////////////////////

        #region TargetFileType

        private ImportableFileType _targetFileType = ImportableFileType.Project;

        public ImportableFileType TargetFileType
        {
            get { return _targetFileType; }
            set { SetProperty(ref _targetFileType, value); }
        }

        #endregion

        ////////////////////////////////////////////////////////////////////////////////

        #region AppMode

        private AppModeType _AppMode;

        public AppModeType AppMode
        {
            get { return _AppMode; }
            set { SetProperty(ref _AppMode, value); }
        }

        #endregion

        public bool IsAocAppMode => AppMode == AppModeType.AocMeta || AppMode == AppModeType.AocNsp;

        ////////////////////////////////////////////////////////////////////////////////

        public AppProfile()
        {
            this.ObserveProperty(x => x.AppMode)
                .Subscribe(SetPropertires)
                .AddTo(CompositeDisposable);
        }

        private void SetPropertires(AppModeType mode)
        {
            switch (mode)
            {
                case AppModeType.Startup:
                case AppModeType.Comparison:
                    TargetFileType = ImportableFileType.Project;
                    IsEnableCorePageGroup = false;
                    CanEditCardSpec = false;
                    CanImportProject = false;
                    CanExportProject = false;
                    CanImportMeta = false;
                    CanSaveMeta = false;
                    CanSaveAsMeta = false;
                    CanImportNsp = false;
                    CanSaveAsNsp = false;
                    CanMakeNspPatch = false;
                    CanCompare = mode == AppModeType.Comparison;
                    CanShowNspEntries = false;
                    CanImportPartialMeta = false;

                    break;

                case AppModeType.Project:
                    TargetFileType = ImportableFileType.Project;
                    IsEnableCorePageGroup = true;
                    CanEditCardSpec = true;
                    CanImportProject = true;
                    CanExportProject = true;
                    CanImportMeta = false;
                    CanSaveMeta = false;
                    CanSaveAsMeta = true;
                    CanImportNsp = false;
                    CanSaveAsNsp = false;
                    CanMakeNspPatch = false;
                    CanCompare = false;
                    CanShowNspEntries = false;
                    CanImportPartialMeta = true;

                    break;

                case AppModeType.ApplicationMeta:
                    SetMetaProperties();
                    CanSaveAsNsp = false;
                    CanImportPartialMeta = true;

                    break;

                case AppModeType.AocMeta:
                    SetMetaProperties();
                    CanSaveAsNsp = true;
                    CanImportPartialMeta = false;

                    break;

                case AppModeType.ApplicationNsp:
                    TargetFileType = ImportableFileType.Nsp;
                    IsEnableCorePageGroup = false;
                    CanEditCardSpec = false;
                    CanImportProject = false;
                    CanExportProject = false;
                    CanImportMeta = false;
                    CanSaveMeta = false;
                    CanSaveAsMeta = false;
                    CanImportNsp = true;
                    CanSaveAsNsp = true;
                    CanMakeNspPatch = true;
                    CanCompare = false;
                    CanShowNspEntries = true;
                    CanImportPartialMeta = true;

                    break;

                case AppModeType.AocNsp:
                    TargetFileType = ImportableFileType.Nsp;
                    IsEnableCorePageGroup = false;
                    CanEditCardSpec = false;
                    CanImportProject = false;
                    CanExportProject = false;
                    CanImportMeta = false;
                    CanSaveMeta = false;
                    CanSaveAsMeta = false;
                    CanImportNsp = true;
                    CanSaveAsNsp = true;
                    CanMakeNspPatch = false;
                    CanCompare = false;
                    CanShowNspEntries = true;
                    CanImportPartialMeta = false;

                    break;

                case AppModeType.PatchNsp:
                    TargetFileType = ImportableFileType.Nsp;
                    IsEnableCorePageGroup = false;
                    CanEditCardSpec = false;
                    CanImportProject = false;
                    CanExportProject = false;
                    CanImportMeta = false;
                    CanSaveMeta = false;
                    CanSaveAsMeta = false;
                    CanImportNsp = true;
                    CanSaveAsNsp = false;
                    CanMakeNspPatch = false;
                    CanCompare = false;
                    CanShowNspEntries = true;
                    CanImportPartialMeta = false;

                    break;
                default:
                    throw new ArgumentOutOfRangeException(nameof(mode), mode, null);
            }
        }

        private void SetMetaProperties()
        {
            TargetFileType = ImportableFileType.Meta;
            IsEnableCorePageGroup = true;
            CanEditCardSpec = true;
            CanImportProject = false;
            CanExportProject = false;
            CanImportMeta = true;
            CanSaveMeta = true;
            CanSaveAsMeta = true;
            CanImportNsp = false;
            CanMakeNspPatch = false;
            CanCompare = false;
            CanShowNspEntries = false;
        }
    }
}
