﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.ObjectModel;
using System.Linq;
using System.Xml.Serialization;
using BezelEditor.Mvvm;
using Reactive.Bindings.Extensions;
using SimpleInjector;
using YamlDotNet.Serialization;

namespace Nintendo.Authoring.AuthoringEditor.Core
{
    public partial class AocMeta : DisposableModelBase
    {
        [YamlIgnore]
        public Container DiContainer
        {
            get { return _DiContainer; }
            set
            {
                _DiContainer = value;

                Contents.ObserveAddChanged()
                    .Subscribe(t => t.DiContainer = value)
                    .AddTo(CompositeDisposable);

                foreach (var t in Contents)
                    t.DiContainer = value;

                // ReSharper disable once ExplicitCallerInfoArgument
                RaisePropertyChanged(nameof(DiContainer));
            }
        }

        private Container _DiContainer;

        #region ApplicationId

        private ulong _ApplicationId = Constants.DefaultMetaProgramId;

        public ulong ApplicationId
        {
            get { return _ApplicationId; }
            set { SetProperty(ref _ApplicationId, value); }
        }

        #endregion

        #region Contents

        private ObservableCollection<AocContent> _Contents = new ObservableCollection<AocContent>();

        public ObservableCollection<AocContent> Contents
        {
            get { return _Contents; }
            set { SetProperty(ref _Contents, value); }
        }

        #endregion

        public AocMeta()
        {
            CompositeDisposable.Add(() => Contents.ForEach(x => x.Dispose()));

            InitializeValidation();
        }

        public void AddContent()
        {
            var maxIndex = Contents.Any() ? Contents.Max(x => x.Index) : 0;

            Contents.Add(new AocContent(DiContainer, this) {Index = maxIndex + 1});
        }

        public void DeleteContent(AocContent content)
        {
            Contents.Remove(content);

            content.Dispose();
        }
    }

    public partial class AocContent : DisposableModelBase
    {
        #region DiContainer

        private Container _DiContainer;

        [YamlIgnore]
        [XmlIgnore]
        public Container DiContainer
        {
            get { return _DiContainer; }
            set { SetProperty(ref _DiContainer, value); }
        }

        #endregion

        public AocMeta Parent { get; }

        // シリアライズ用
        public AocContent()
        {
        }

        public AocContent(Container diContainer, AocMeta parent)
        {
            Parent = parent;
            DiContainer = diContainer;

            InitializeValidation();
        }

        #region Index

        private ulong _Index = 1;

        public ulong Index
        {
            get { return _Index; }
            set { SetProperty(ref _Index, value); }
        }

        #endregion

        #region ReleaseVersion

        private ulong _ReleaseVersion;

        public ulong ReleaseVersion
        {
            get { return _ReleaseVersion; }
            set { SetProperty(ref _ReleaseVersion, value); }
        }

        #endregion

        #region RequiredApplicationReleaseVersion

        private ulong _RequiredApplicationReleaseVersion;

        public ulong RequiredApplicationReleaseVersion
        {
            get { return _RequiredApplicationReleaseVersion; }
            set { SetProperty(ref _RequiredApplicationReleaseVersion, value); }
        }

        #endregion

        public const int MaxTagLength = 128;

        #region Tag

        private string _Tag = string.Empty;

        public string Tag
        {
            get { return _Tag; }
            set { SetProperty(ref _Tag, value); }
        }

        #endregion

        #region DataPath

        private ExpandablePath _DataPath = string.Empty;

        [YamlIgnore]
        public ExpandablePath DataPath
        {
            get { return _DataPath; }
            set
            {
                if (_DataPath.SetExpandablePath(value))
                    RaisePropertyChanged(nameof(DataPath));
            }
        }

        [XmlIgnore]
        [YamlMember(Alias = nameof(DataPath))]
        public string DataPathForYaml
        {
            get { return DataPath.Path; }
            set { DataPath.Path = value; }
        }

        [XmlIgnore]
        public bool DataPathIsExpandEnvironmentVariableForYaml
        {
            get { return DataPath.IsExpandEnvironmentVariable; }
            set { DataPath.IsExpandEnvironmentVariable = value; }
        }

        #endregion

        #region Digest

        private string _Digest = string.Empty;

        public string Digest
        {
            get { return _Digest; }
            set { SetProperty(ref _Digest, value); }
        }

        #endregion

        #region ContentDataSize

        private ulong _ContentDataSize;

        public ulong ContentDataSize
        {
            get { return _ContentDataSize; }
            set { SetProperty(ref _ContentDataSize, value); }
        }

        #endregion
    }
}
