﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.ObjectModel;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Reactive;
using System.Reactive.Linq;
using System.Xml.Serialization;
using Reactive.Bindings.Extensions;
using YamlDotNet.Serialization;

namespace Nintendo.Authoring.AuthoringEditor.Core
{
    public partial class AocMeta
    {
        #region ApplicationId

        [YamlIgnore]
        [XmlIgnore]
        public ApplicationIdValidationType ValidationApplicationId
            => ValidateApplicationId(ApplicationId);

        public static ApplicationIdValidationType ValidateApplicationId(ulong id)
        {
            return
                id >= Constants.ProgramIdMinimum &&
                id <= Constants.ProgramIdMaximum
                    ? ApplicationIdValidationType.Ok
                    : ApplicationIdValidationType.RangeOver;
        }

        public enum ApplicationIdValidationType
        {
            Ok,
            RangeOver
        }

        #endregion

        #region HasErrors

        private bool _hasErrors;

        [YamlIgnore]
        [XmlIgnore]
        public bool HasErrors
        {
            get { return _hasErrors; }
            set { SetProperty(ref _hasErrors, value); }
        }

        #endregion

        private IDisposable _contentsObserver;
        private IDisposable _contentsChangedObserver;

        private void InitializeValidation()
        {
            _contentsObserver?.Dispose();

            _contentsObserver = this.ObserveProperty(x => x.Contents)
                .Subscribe(_ =>
                {
                    _contentsChangedObserver?.Dispose();
                    _contentsChangedObserver = Contents.CollectionChangedAsObservable()
                        .Subscribe(__ => UpdateHasErrors());
                });

            UpdateHasErrors();
        }

        public void UpdateHasErrors()
        {
            if (Contents.Any() == false)
            {
                HasErrors = true;
                return;
            }

            var hasErrors = false;

            foreach (var c in Contents)
            {
                c.UpdateHasErrors();

                if (c.HasErrors)
                    hasErrors = true;
            }

            HasErrors = hasErrors;
        }

        public static ContentsValidationType ValidateContents(ObservableCollection<AocContent> contents)
        {
            if (contents == null)
                return ContentsValidationType.EmptyError;

            return contents.Any()
                ? ContentsValidationType.Ok
                : ContentsValidationType.EmptyError;
        }

        public enum ContentsValidationType
        {
            Ok,
            EmptyError
        }
    }

    public partial class AocContent
    {
        private IObservable<ulong> _indexO;
        private IObservable<string> _tagO;
        private IObservable<string> _dataPathO;

        private void InitializeValidation()
        {
            // ReSharper disable ExplicitCallerInfoArgument

            _indexO = this.ObserveProperty(x => x.Index);
            _tagO = this.ObserveProperty(x => x.Tag);
            _dataPathO = DataPath.ObserveProperty(x => x.Path);

            _indexO
                .Subscribe(_ => RaisePropertyChanged(nameof(ValidationIndex)))
                .AddTo(CompositeDisposable);

            _tagO
                .Subscribe(_ => RaisePropertyChanged(nameof(ValidationTag)))
                .AddTo(CompositeDisposable);

            _dataPathO
                .Subscribe(_ => RaisePropertyChanged(nameof(ValidationDataPath)))
                .AddTo(CompositeDisposable);

            Observable
                .Merge(_indexO.ToUnit())
                .Merge(_tagO.ToUnit())
                .Merge(_dataPathO.ToUnit())
                .Merge(this.ObserveProperty(x => x.ValidationIndex).ToUnit())
                .Merge(this.ObserveProperty(x => x.ValidationTag).ToUnit())
                .Merge(this.ObserveProperty(x => x.ValidationDataPath).ToUnit())
                .Merge(this.ObserveProperty(x => x.DiContainer).ToUnit())
                .Merge(DiContainer.GetInstance<App>().ObserveProperty(x => x.Project).ToUnit())
                .Merge(Observable.Return(Unit.Default))
                .Subscribe(_ => Parent.UpdateHasErrors())
                .AddTo(CompositeDisposable);

            // ReSharper restore ExplicitCallerInfoArgument
        }

        public void UpdateHasErrors()
        {
            HasErrors =
                ValidationIndex != IndexValidationType.Ok ||
                ValidationTag != TagValidationType.Ok ||
                ValidationDataPath != DataPathValidationType.Ok;
        }

        #region HasErrors

        private bool _hasErrors;

        [YamlIgnore]
        [XmlIgnore]
        public bool HasErrors
        {
            get { return _hasErrors; }
            set { SetProperty(ref _hasErrors, value); }
        }

        #endregion

        #region Index

        [YamlIgnore]
        [XmlIgnore]
        public IndexValidationType ValidationIndex
            => ValidateIndex(Index, this, DiContainer?.GetInstance<Project>()?.AocMeta?.Contents);

        public static IndexValidationType ValidateIndex(ulong s, AocContent ignore,
            ObservableCollection<AocContent> allContents)
        {
            if (allContents == null)
                return IndexValidationType.Ok;

            if (s < Constants.AocIndexMinimum ||
                s > Constants.AocIndexMaximum)
                return IndexValidationType.RangeOver;

            var isExists = false;

            foreach (var c in allContents)
            {
                if (c == ignore)
                    continue;

                if (c.Index == s)
                {
                    isExists = true;
                    break;
                }
            }

            return isExists
                ? IndexValidationType.AlreadyExist
                : IndexValidationType.Ok;
        }

        public enum IndexValidationType
        {
            Ok,
            AlreadyExist,
            RangeOver
        }

        #endregion

        #region Tag

        public TagValidationType ValidationTag
            => ValidateTag(Tag, this, DiContainer?.GetInstance<Project>()?.AocMeta?.Contents);

        public static TagValidationType ValidateTag(string s, AocContent ignore,
            ObservableCollection<AocContent> allContents)
        {
            if (allContents == null)
                return TagValidationType.Ok;

            if (string.IsNullOrEmpty(s))
                return TagValidationType.EmptyError;

            var isExists = false;

            foreach (var c in allContents)
            {
                if (c == ignore)
                    continue;

                if (c.Tag == s)
                {
                    isExists = true;
                    break;
                }
            }

            if (isExists)
                return TagValidationType.AlreadyExist;

            if (new StringInfo(s).LengthInTextElements > MaxTagLength)
                return TagValidationType.LengthError;

            return TagValidationType.Ok;
        }

        public enum TagValidationType
        {
            Ok,
            AlreadyExist,
            EmptyError,
            LengthError
        }

        #endregion

        #region DataPath

        [YamlIgnore]
        [XmlIgnore]
        public DataPathValidationType ValidationDataPath
        {
            get
            {
                if (DiContainer?.GetInstance<AppProfile>()?.AppMode != AppModeType.AocMeta)
                    return DataPathValidationType.Ok;

                if (DataPath.IsExpandEnvironmentVariable)
                    return DataPathValidationType.Ok;

                return ValidateDataPath(DiContainer?.GetInstance<Project>()?.ToAbsolutePath(DataPath.Path));
            }
        }

        public static DataPathValidationType ValidateDataPath(string path)
        {
            if (string.IsNullOrEmpty(path))
                return DataPathValidationType.PathIsEmpty;

            if (Directory.Exists(path) == false)
                return DataPathValidationType.DirectoryNotFound;

            return DataPathValidationType.Ok;
        }

        public enum DataPathValidationType
        {
            Ok,
            DirectoryNotFound,
            PathIsEmpty
        }

        #endregion
    }
}
