﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Threading.Tasks;

namespace SystemAudioMonitor
{
    public class ChannelLoudnessInfo
    {
        public float PeakValue { get; set; }
        public float TruePeakValue { get; set; }
        public float RmsValue { get; set; }

        public ChannelLoudnessInfo()
        {
        }
    }

    public class LoudnessInfo
    {
        static public readonly int ChannelCountMax = 6;
        public long SampleIndex { get; set; }
        public int LoudnessIndex { get; set; }
        public float MomentaryLoudnessValue { get; set; }
        public float ShortTermLoudnessValue { get; set; }
        public float AbsGatedLoudnessValue { get; set; }
        public float RelGatedLoudnessValue { get; set; }
        public ChannelLoudnessInfo[] Channels { get; } =
            new ChannelLoudnessInfo[ChannelCountMax].Populate(_ => new ChannelLoudnessInfo());

        public LoudnessInfo()
        {
        }
    }

    public class LoudnessAccumulator
    {
        static public readonly int ChannelCountMax = LoudnessInfo.ChannelCountMax;
        private int _channelCount;
        public int ChannelCount
        {
            get
            {
                return _channelCount;
            }
            set
            {
                Debug.Assert(value > 0);
                Debug.Assert(value <= ChannelCountMax);
                _channelCount = value;
            }
        }

        private class WaveformRegion
        {
            public short[][] Samples { get; }
            public int From { get; }
            public int To { get; }

            public WaveformRegion(short[][] samples, int from, int to)
            {
                this.Samples = samples;
                this.From = from;
                this.To = to;
            }
        }

        private readonly LoudnessCalculator _loudnessCalculator = new LoudnessCalculator();
        private readonly Loudness.LoudnessContext[] _loudnessContext = new Loudness.LoudnessContext[Loudness.ChannelCount].Populate(_ => new Loudness.LoudnessContext());
        private readonly double[] _channelValue = new double[Loudness.ChannelCount];
        private readonly float[] _sampleSquareSum = new float[Loudness.ChannelCount];
        private readonly float[] _samplePeak = new float[Loudness.ChannelCount];
        private readonly float[] _sampleTruePeak = new float[Loudness.ChannelCount];
        private readonly List<WaveformRegion> _waveformRegions = new List<WaveformRegion>();
        private long _sampleIndex;
        private int _accumulatedSampleCount;
        private Loudness.Parameter _parameter;

        public LoudnessAccumulator()
        {
            Debug.Assert(Loudness.ChannelCount <= ChannelCountMax);
            ChannelCount = ChannelCountMax;
        }

        public void SetParameter(Loudness.Parameter parameter)
        {
            _parameter = parameter;
            _loudnessCalculator.SetParameter(parameter);
        }

        public IEnumerable<LoudnessInfo> Accumulate(byte[] buffer, int offset, int size)
        {
            int sampleLength = (buffer.Length - offset) / (sizeof(short) * _channelCount);
            short[][] samples = new short[ChannelCountMax][];
            for (int i = 0; i < ChannelCountMax; ++i)
            {
                samples[i] = new short[sampleLength];
            }

            for (int i = 0; i < buffer.Length / 2; ++i)
            {
                int channelIndex = i % _channelCount;
                int sampleIndex = i / _channelCount;

                samples[channelIndex][sampleIndex] = BitConverter.ToInt16(buffer, i * 2);
            }

            return AccumulateImpl(samples, 0, sampleLength);
        }

        private IEnumerable<LoudnessInfo> AccumulateImpl(short[][] samples, int offset, int count)
        {
            int sampleIndex = 0;
            while (sampleIndex < count)
            {
                // ラウドネスの計算に足りないサンプル数だけ累積
                int sampleCount = Math.Min(count - sampleIndex, _parameter.StepSampleCount - _accumulatedSampleCount);
                Debug.Assert(sampleCount > 0);

                _waveformRegions.Add(new WaveformRegion(samples, sampleIndex, sampleIndex + sampleCount));
                sampleIndex += sampleCount;
                _accumulatedSampleCount += sampleCount;

                // ラウドネスの計算にサンプル数が足りないときはここで終了
                if (_accumulatedSampleCount < _parameter.StepSampleCount)
                {
                    break;
                }

                // ラウドネスを計算
                {
                    this.AccumulateSample();

                    double totalMeanSquare = 0;
                    float[] sampleSquareSumArray = new float[Loudness.ChannelCount];
                    float[] samplePeakArray = new float[Loudness.ChannelCount];
                    float[] sampleTruePeakArray = new float[Loudness.ChannelCount];
                    for (int channelIndex = 0; channelIndex < Loudness.ChannelCount; channelIndex++)
                    {
                        totalMeanSquare += _channelValue[channelIndex] * Loudness.ChannelWeightArray[channelIndex];
                        sampleSquareSumArray[channelIndex] = _sampleSquareSum[channelIndex];
                        samplePeakArray[channelIndex] = _samplePeak[channelIndex];
                        sampleTruePeakArray[channelIndex] = _sampleTruePeak[channelIndex];
                    }

                    _loudnessCalculator.AddStep((float)totalMeanSquare, sampleSquareSumArray, samplePeakArray, sampleTruePeakArray);
                }

                _sampleIndex += _parameter.StepSampleCount;

                var loudnessInfo = ReadLoudnessInfo();

                // 計算の完了したサンプルのインデックスを記録します。
                loudnessInfo.SampleIndex = _sampleIndex;

                ResetValue();

                yield return loudnessInfo;
            }
        }

        private void ResetValue()
        {
            for (int i = 0; i < ChannelCountMax; ++i)
            {
                _channelValue[i] = 0;
                _sampleSquareSum[i] = 0;
                _samplePeak[i] = 0;
                _sampleTruePeak[i] = 0;
            }

            _accumulatedSampleCount = 0;
        }

        private void AccumulateSample()
        {
            Parallel.For(0, ChannelCountMax, channelIndex =>
            {
                var samplePeak = float.MinValue;
                var sampleTruePeak = float.MinValue;
                var sampleSquareSum = 0.0f;
                var channelValue = 0.0;
                var loudnessContext = _loudnessContext[channelIndex];

                foreach (var region in _waveformRegions)
                {
                    short[] sample = region.Samples[channelIndex];

                    for (int i = region.From; i < region.To; i++)
                    {
                        float sampleValue = sample[i] / 32768.0f;

                        samplePeak = Math.Max(samplePeak, Math.Abs(sampleValue));
                        sampleTruePeak = Math.Max(sampleTruePeak, Loudness.CalcTruePeak(loudnessContext, sampleValue));
                        sampleSquareSum += sampleValue * sampleValue;
                        float kOut = Loudness.CalcKWeightingFilter(loudnessContext, sampleValue, _parameter);
                        channelValue += kOut * kOut;
                    }
                }

                _samplePeak[channelIndex] = Math.Max(_samplePeak[channelIndex], samplePeak);
                _sampleTruePeak[channelIndex] = Math.Max(_sampleTruePeak[channelIndex], sampleTruePeak);
                _sampleSquareSum[channelIndex] += sampleSquareSum;
                _channelValue[channelIndex] += channelValue;
            });

            _waveformRegions.Clear();
        }

        private LoudnessInfo ReadLoudnessInfo()
        {
            var loudness = new LoudnessInfo();

            loudness.MomentaryLoudnessValue = _loudnessCalculator.MomentaryLoudnessValue;
            loudness.ShortTermLoudnessValue = _loudnessCalculator.ShortTermLoudnessValue;
            loudness.AbsGatedLoudnessValue = _loudnessCalculator.AbsGatedLoudnessValue;
            loudness.RelGatedLoudnessValue = _loudnessCalculator.RelGatedLoudnessValue;

            for (int channelIndex = 0; channelIndex < Loudness.ChannelCount; channelIndex++)
            {
                var channelInfo = loudness.Channels[channelIndex];
                channelInfo.PeakValue = _loudnessCalculator.PeakValue[channelIndex];
                channelInfo.TruePeakValue = _loudnessCalculator.TruePeakValue[channelIndex];
                channelInfo.RmsValue = _loudnessCalculator.RmsValue[channelIndex];
            }

            return loudness;
        }
    }
}
