﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <locale.h>
#include <stdio.h>
#include <string.h>
#include <windows.h>
#include <assert.h>
#include "dspadpcm.h"
#include "dspheader.h"

#define FORCE_ENCODE_MODE

#define MAX_PATH_SIZE   FILENAME_MAX
#define DEFAULT_INTERLEAVED_FRAME_COUNT 0x400

u8 input_path  [MAX_PATH_SIZE]; // path to input file
u8 output_path [MAX_PATH_SIZE]; // path to output file
u8 split_path  [MAX_PATH_SIZE]; // path to output file
u8 coef_path   [MAX_PATH_SIZE]; // path to DSPADPCMHEADER text dump file

FILE *input_file;
FILE *output_file;
FILE *split_file;
FILE *coef_file;

BOOL  loop_flag;                // TRUE if loop points were specified on commandline
BOOL  encode_flag;              // TRUE for encoding
BOOL  decode_flag;              // TRUE for decoding
BOOL  split_flag;               // TRUE for splitting multicahnnel input files
BOOL  coef_flag;                // TRUE for text dump of DSPADPCMHEADER to file
BOOL  verbose_flag;             // TRUE is user specified verbose mode
BOOL  ea_flag;                  // TRUE if user specified end address

u32   loopStart;                // user specified loop start sample
u32   loopEnd;                  // user specified loop end sample
u32   sampleEndAddr;            // user specified end address

#define DECODE_WAV  0
#define DECODE_AIFF 1

#define INTERLEAVE_MODE      TRUE
#define NON_INTERLEAVE_MODE  FALSE

u32 decodeFormat;               // user specified decode fromat
BOOL interleave_flag;              // user specified encode a multi-channel WAV into an interleaved DSP file


/*--------------------------------------------------------------------------*
        dsptool DLL interface
 *--------------------------------------------------------------------------*/
HINSTANCE hDllDsptool;

typedef u32 (*lpFunc1)(u32);
typedef u32 (*lpFunc2)(void);
typedef void (*lpFunc3)(s16*, u8*, ADPCMINFO*, u32);
typedef void (*lpFunc4)(u8*, s16*, ADPCMINFO*, u32);
typedef void (*lpFunc5)(u8*, ADPCMINFO*, u32);

lpFunc1 getBytesForAdpcmBuffer;
lpFunc1 getBytesForAdpcmSamples;
lpFunc1 getBytesForPcmBuffer;
lpFunc1 getBytesForPcmSamples;
lpFunc1 getNibbleAddress;
lpFunc1 getNibblesForNSamples;
lpFunc1 getSampleForAdpcmNibble;
lpFunc2 getBytesForAdpcmInfo;
lpFunc3 encode;
lpFunc4 decode;
lpFunc5 getLoopContext;


/*--------------------------------------------------------------------------*
        soundfile DLL interface
 *--------------------------------------------------------------------------*/
HINSTANCE hDllSoundfile;

typedef int (*lpFunc6)(u8 *path, SOUNDINFO *soundinfo);
typedef int (*lpFunc7)(u8 *path, SOUNDINFO *soundinfo, void *dest);
typedef int (*lpFunc8)(char *path, SOUNDINFO *info, void *samples);
lpFunc6 getSoundInfo;
lpFunc7 getSoundSamples;
lpFunc8 writeWaveFile;
lpFunc8 writeAiffFile;

#define LOG_ERROR(fmt, ...) fprintf(stderr, "ERROR: "fmt, ##__VA_ARGS__)

/*--------------------------------------------------------------------------*/
void print_usage(void)
{
    printf("\n");
    printf("Usage:\n\n");
    printf("AdpcmEncode <inputfile> [-opt<argument>]\n\n");
    printf("Where:\n");
    printf("   <inputfile>........Input file (required)\n");

    printf("Options are:\n");
    printf("   -o|--output <outputfile>....Output file\n");
    printf("   --loop-begin <sample-index>.Beginning position of the loop in sample, counting from zero.\n");
    printf("                               Should be used with --loop-end.\n");
    printf("   --loop-end <sample-index>...Ending position of the loop in sample, counting from zero.\n");
    printf("                               Should be used with --loop-begin.\n");
    printf("   -h|--help...................This help text.\n");
    printf("   -v|--verbose................Verbose mode.\n");
    printf("\n\n");
    printf("This tool generates DSPADPCM data from MONO, 16-bit PCM AIFF or n-channel\n");
    printf("16 bit PCM WAV files. The output is at the same sample rate as the input.\n");
    printf("\n\n");
    printf("For > 1 channel WAV files, the additional channels are encoded to separate\n");
    printf("files with the suffix '_2' through '_n' appended to the root filename.\n");
    printf("\n\n");
    printf("The DSPADPCM output file (*.adpcm) has a %zd byte data header \n", sizeof(DSPADPCMHEADER));
    printf("which contains the ADPCM coefficients, loop context (if any), \n");
    printf("and other sample info. The format of this header is described \n");
    printf("in the DSPADPCM documentation.\n\n");

} // end print_usage()


/*--------------------------------------------------------------------------*/

void init(void)
{
    input_file    = NULL;
    output_file   = NULL;
    coef_file     = NULL;
    split_file    = NULL;
    decodeFormat  = DECODE_WAV;
    interleave_flag  = FALSE;

    hDllDsptool   = NULL;
    hDllSoundfile = NULL;

} // end init()


/*--------------------------------------------------------------------------*/
void clean_up(void)
{
    if (input_file)    fclose(input_file);
    if (output_file)   fclose(output_file);
    if (coef_file)     fclose(coef_file);
    if (split_file)    fclose(split_file);
    if (hDllDsptool)   FreeLibrary(hDllDsptool);
    if (hDllSoundfile) FreeLibrary(hDllSoundfile);

} // end clean_up()


/*--------------------------------------------------------------------------*/

void coefficients(u16 *p)
{
    int i;

    for (i = 0; i < 16; i++)
        printf("[%d]: 0x%04X\n", i, *p++ & 0xffff);
}


/*--------------------------------------------------------------------------
 * findbase() - returns pointer to first character of file basename
 *--------------------------------------------------------------------------*/
BYTE *findbase(BYTE *string)
{

    BYTE *ptr = string;

    while (*ptr++); // find end of string

    while ((--ptr >= string) && (*ptr != '\\') && (*ptr != '/') && (*ptr != ':'));

    return(++ptr);

} // end findbase()


/*--------------------------------------------------------------------------
 * findext() - returns pointer to start of file extension (including '.')
 *--------------------------------------------------------------------------*/
BYTE *findext(BYTE *string)
{
    BYTE *ptr;
    BYTE *end;

    ptr = string;

    while (*ptr++); // find end of string

    end = ptr;

    while ((--ptr >= string) && (*ptr != '.'));

    if (ptr <= string)
        return(end);
    else
        return(ptr);

} // end findext()

int GetOptionAndArgument(int argc, char* argv[], int index, char* pOutOption, int optionLength, char* pOutArgument, int argumentLength)
{
   	int parsedCount = 0;

    char* p = argv[index];

    if (index >= argc)
    {
        return 0;
    }

    memset(pOutOption, 0, optionLength);
    memset(pOutArgument, 0, argumentLength);

    if (p[0] == '-' && p[1] != '-')  // short option
    {
        if (p[2] == '\0')
        {
            strncpy(pOutOption, p, optionLength - 1);
            pOutOption[optionLength - 1] = '\0';
            ++parsedCount;
            if (index + 1 < argc)
            {
                char* pNext = argv[index + 1];
                if (pNext[0] != '-')
                {
                    strncpy(pOutArgument, pNext, argumentLength - 1);
                    pOutArgument[argumentLength - 1] = '\0';
                    ++parsedCount;
                }
            }
        }
        else
        {
//			assert(optionLength >= 3);
            memcpy(pOutOption, p, 2);
            pOutOption[2] = '\0';
            strncpy(pOutArgument, p + 2, argumentLength - 1);
            pOutArgument[argumentLength - 1] = '\0';
            ++parsedCount;
        }
    }
    else if (p[0] == '-' && p[1] == '-')  // long option
    {
        strncpy(pOutOption, p, optionLength - 1);
        pOutOption[optionLength - 1] = '\0';
        ++parsedCount;
        if (index + 1 < argc)
        {
            char* pNext = argv[index + 1];
            if (pNext[0] != '-')
            {
                strncpy(pOutArgument, pNext, argumentLength - 1);
                pOutArgument[argumentLength - 1] = '\0';
                ++parsedCount;
            }
        }
    }
    else  // not an option
    {
        strncpy(pOutArgument, p, argumentLength - 1);
        pOutArgument[argumentLength - 1] = '\0';
        ++parsedCount;
    }

    return parsedCount;
}

/*--------------------------------------------------------------------------
 * parse_args() - parse the command line arguments
 *--------------------------------------------------------------------------*/
BOOL parse_args(int argc, char *argv[])
{
    BOOL rval;
    int index;

    if (argc < 2)
    {
        // must specify input file at least
        printf("\nERROR: Missing parameter\n");

        print_usage();

        return(FALSE);
    }

    rval         = TRUE;
    loop_flag    = FALSE;
    encode_flag  = FALSE;
    decode_flag  = FALSE;
    split_flag   = FALSE;
    coef_flag    = FALSE;
    verbose_flag = FALSE;
    ea_flag      = FALSE;

#if defined(FORCE_ENCODE_MODE)
    encode_flag  = TRUE;
#endif  // defined(FORCE_ENCODE_MODE)

    memset(input_path,   0x00, MAX_PATH_SIZE);
    memset(output_path,  0x00, MAX_PATH_SIZE);
    memset(coef_path,    0x00, MAX_PATH_SIZE);

    loopStart     = 0xFFFFFFFF;
    loopEnd       = 0xFFFFFFFF;
    sampleEndAddr = 0x00000000;

    index = 1;
    while (index < argc)
    {
        char option[512];
    	char argument[512];

        int readCount = GetOptionAndArgument(argc, argv, index, option, sizeof(option), argument, sizeof(argument));
        index += readCount;
        if (strnlen(option, sizeof(option)) == 0)
        {
            if (0 == input_path[0])
            {
                // input path not specified yet, so argument must be the input path
                strncpy(input_path, argument, sizeof(input_path));
            }
            else
            {
                // unknown unswitched argument
                LOG_ERROR("extraneous argument '%s'\n", argument);
                rval = FALSE;
            }
        }
        else
        {
            if (strncmp(option, "-o", sizeof(option)) == 0 || strncmp(option, "--output", sizeof(option)) == 0)
            {
                strncpy(output_path, argument, sizeof(output_path));
                // TODO: 拡張子名は .adpcm である必要はないが、実装の理由
                //       （拡張子が 6 文字であるという前提がある）によりこれに制限しており、要修正
                if (!(strncmp((const char*)findext(output_path), ".adpcm", strlen(".adpcm")) == 0 &&
                      strnlen((const char*)findext(output_path), FILENAME_MAX) == strlen(".adpcm")))
                 {
                      LOG_ERROR("output file extension should be \".adpcm\"\n");
                      rval = FALSE;
                 }
            }
            else if (strncmp(option, "--loop-begin", sizeof(option)) == 0)
            {
                loopStart = atoi(argument);
            }
            else if (strncmp(option, "--loop-end", sizeof(option)) == 0)
            {
                loopEnd = atoi(argument);
            }
            else if (strncmp(option, "-h", sizeof(option)) == 0 || strncmp(option, "--help", sizeof(option)) == 0)
            {
                print_usage();
                return FALSE;
            }
            else if (strncmp(option, "-v", sizeof(option)) == 0 || strncmp(option, "--verbose", sizeof(option)) == 0)
            {
                verbose_flag = TRUE;
                --index;  // オプション引数は取らないので、インデックスを戻す
            }
            else
            {
                LOG_ERROR("unknown option '%s'\n", option);
                rval = FALSE;
            }
        }
    }
    if (loopStart != 0xFFFFFFFF && loopEnd != 0xFFFFFFFF)
    {
        loop_flag = TRUE;
    }
    else if (!(loopStart == 0xFFFFFFFF && loopEnd == 0xFFFFFFFF))
    {
        LOG_ERROR("both --loop-begin and --loop-end should be specified\n");
        rval = FALSE;
    }

    // now perform sanity check
    if (0 == input_path[0])
    {
        // no input file specified!
        printf("\nERROR: No input file specified!\n");
        rval = FALSE;
    }

    if (0 == output_path[0])
    {
        // output path not specified yet, use default
        strcpy(output_path, findbase(input_path));

        if (TRUE == decode_flag)
        {
            if (decodeFormat != DECODE_AIFF)
            {
                // add default extension of '.wav'
                strcpy(findext(output_path), ".wav");
            }
            else
            {
                // add default extension of '.aif'
                strcpy(findext(output_path), ".aif");
            }
        }
        else if (encode_flag)
        {
            // add default extension of '.adpcm'
            strcpy(findext(output_path), ".adpcm");
        }
        else if (interleave_flag)
        {
            // add default extension of '.adpcm'
            strcpy(findext(output_path), ".madpcm");
        }
        else
        {
            printf("\nERROR: MODE directive not specified!\n");
            rval = FALSE;
        }
    }

    if ((0 == coef_path[0]) && (coef_flag == TRUE))
    {
        // coefficient output path not specified, use default
        strcpy(coef_path, findbase(input_path));

        // add default extension of '.txt'
        strcpy(findext(coef_path), ".txt");
    }

#define ARAM_MAX_RANGE  0x4000000   // 64MB of ARAM - in my dreams

    if (TRUE == loop_flag)
    {
        if (loopStart > loopEnd)
        {
            printf("\nWARNING: loopStart is greater than loopEnd\n");
        }

        if (loopStart > ARAM_MAX_RANGE)
        {
            printf("\nWARNING: loop-start is beyond valid ARAM range! (0x%08X)\n", loopStart);
        }

        if (loopEnd > ARAM_MAX_RANGE)
        {
            printf("\nWARNING: loop=end is beyond valid ARAM range! (0x%08X)\n", loopEnd);
        }

        if (TRUE == ea_flag)
        {
            printf("\nWARNING: '-a' argument ignored for looped sample.\n");
        }
    }

    if (sampleEndAddr > ARAM_MAX_RANGE)
    {
        printf("\nWARNING: sample-end address is beyond valid ARAM range! (0x%08X)\n", sampleEndAddr);
    }

   if (interleave_flag && split_flag)
   {
      printf("\nWARNING: split flag ignored when mode set to interleave encode.\n");
      split_flag = FALSE;
   }

    if (verbose_flag)
    {
        printf("\tinput file : '%s'\n", input_path);
        printf("\toutput file: '%s'\n", output_path);
        if (coef_flag)
        {
            printf("\tcoef file  : '%s'\n", coef_path);
        }
        printf("\n");
    }

    return(rval);

} // end parse args


/*--------------------------------------------------------------------------
 *      dump DSPHEADER to specified file
 *--------------------------------------------------------------------------*/
void dump_header(DSPADPCMHEADER *h, FILE *handle, int channel)
{
    u16 i;
    u16 j;

    fprintf(handle, "\n");

    fprintf(handle, "Header size: %zd bytes\n\n", sizeof(DSPADPCMHEADER));

    fprintf(handle, "Sample     : '%s'\n", input_path);
    if (channel) { fprintf(handle, "Channel    : %d\n", channel-1); }
    fprintf(handle, "Length     : %d samples\n", h->num_samples);
    fprintf(handle, "Num nibbles: %d ADPCM nibbles\n", h->num_adpcm_nibbles);

    fprintf(handle, "Sample Rate: %d Hz\n", h->sample_rate);
    fprintf(handle, "Loop Flag  : %s\n", VoiceType_Looped == h->loop_flag ? "LOOPED" : "NOT LOOPED");

    fprintf(handle, "\n");

    fprintf(handle, "Start Addr : 0x%08X + ARAM_offset (ADPCM nibble mode)\n", h->sa);
    fprintf(handle, "End Addr   : 0x%08X + ARAM_offset (ADPCM nibble mode)\n", h->ea);
    fprintf(handle, "Curr Addr  : 0x%08X + ARAM_offset (ADPCM nibble mode)\n", h->ca);

    fprintf(handle, "\n");

    j = 0;
    for (i=0; i<16; i+=2)
    {
        fprintf(handle, "a1[%d]: 0x%04X a2[%d]: 0x%04X \n", j, h->coef[i], j, h->coef[i+1]);
        j++;
    }

    fprintf(handle, "\n");

    fprintf(handle, "Gain      : 0x%04X\n", h->gain);
    fprintf(handle, "Pred/Scale: 0x%04X\n", h->ps);
    fprintf(handle, "y[n-1]    : 0x%04X\n", h->yn1);
    fprintf(handle, "y[n-2]    : 0x%04X\n", h->yn2);

    fprintf(handle, "\n");

    fprintf(handle, "Loop Pred/Scale: 0x%04X\n", h->lps);
    fprintf(handle, "Loop y[n-1]    : 0x%04X\n", h->lyn1);
    fprintf(handle, "Loop y[n-2]    : 0x%04X\n", h->lyn2);

} // end dump_header()


/*--------------------------------------------------------------------------*
 *
 *--------------------------------------------------------------------------*/
void encode_soundfile_channel(
    SOUNDINFO *soundinfo,
    DSPADPCMHEADER dspadpcmheader,
    void *soundbuffer,
    void *outputbuffer
    )
{
    ADPCMINFO      adpcminfo;

    dspadpcmheader.num_samples       = soundinfo->samples;
    dspadpcmheader.num_adpcm_nibbles = getNibblesForNSamples(soundinfo->samples);
    dspadpcmheader.sample_rate       = soundinfo->sampleRate;

    // if the user specified loop points on the commandline use them
    // or else look for loop points in the adpcminfo
    if (loop_flag)
    {
        u32 nibbleLoopStart, nibbleLoopEnd, nibbleCurrent;

        nibbleLoopStart = getNibbleAddress(loopStart);
        nibbleLoopEnd   = getNibbleAddress(loopEnd);
        nibbleCurrent   = getNibbleAddress(0);

        dspadpcmheader.loop_flag = VoiceType_Looped;
        dspadpcmheader.format    = DecodeMode_Adpcm;
        dspadpcmheader.sa        = nibbleLoopStart;
        dspadpcmheader.ea        = nibbleLoopEnd;
        dspadpcmheader.ca        = nibbleCurrent;
    }
    else if (soundinfo->loopEnd) // the sound info has loops form AIFF
    {
        u32 nibbleLoopStart, nibbleLoopEnd, nibbleCurrent;

        nibbleLoopStart = getNibbleAddress(soundinfo->loopStart);
        nibbleLoopEnd   = getNibbleAddress(soundinfo->loopEnd - 1); // AIFF loop end is 1 based
        nibbleCurrent   = getNibbleAddress(0);

        dspadpcmheader.loop_flag = VoiceType_Looped;
        dspadpcmheader.format    = DecodeMode_Adpcm;
        dspadpcmheader.sa        = nibbleLoopStart;
        dspadpcmheader.ea        = nibbleLoopEnd;
        dspadpcmheader.ca        = nibbleCurrent;
    }
    else // not looped
    {
        u32 nibbleLoopStart, nibbleLoopEnd, nibbleCurrent;

        nibbleLoopStart = getNibbleAddress(0);

        // if the user specified end address use it
        if (ea_flag)
            nibbleLoopEnd = getNibbleAddress(sampleEndAddr);
        else
            nibbleLoopEnd = getNibbleAddress(soundinfo->samples - 1);

        nibbleCurrent = getNibbleAddress(0);

        dspadpcmheader.loop_flag = VoiceType_OneShot;
        dspadpcmheader.format    = DecodeMode_Adpcm;
        dspadpcmheader.sa        = nibbleLoopStart;
        dspadpcmheader.ea        = nibbleLoopEnd;
        dspadpcmheader.ca        = nibbleCurrent;
    }

    if (verbose_flag) printf(" Done.\nEncoding samples...");

    encode(soundbuffer, outputbuffer, &adpcminfo, soundinfo->samples);

    // if the user specified loops get loop context
    if (loop_flag)
        getLoopContext(outputbuffer, &adpcminfo, loopStart);
    // see if the aiff file has loop points
    else if (soundinfo->loopEnd)
        getLoopContext(outputbuffer, &adpcminfo, soundinfo->loopStart);
    // no loop make sure loop context is 0
    else
        adpcminfo.loop_pred_scale = adpcminfo.loop_yn1 = adpcminfo.loop_yn2 = 0;

    // put the adpcm info on the dsp_header
    dspadpcmheader.coef[0]  = adpcminfo.coef[0];
    dspadpcmheader.coef[1]  = adpcminfo.coef[1];
    dspadpcmheader.coef[2]  = adpcminfo.coef[2];
    dspadpcmheader.coef[3]  = adpcminfo.coef[3];
    dspadpcmheader.coef[4]  = adpcminfo.coef[4];
    dspadpcmheader.coef[5]  = adpcminfo.coef[5];
    dspadpcmheader.coef[6]  = adpcminfo.coef[6];
    dspadpcmheader.coef[7]  = adpcminfo.coef[7];
    dspadpcmheader.coef[8]  = adpcminfo.coef[8];
    dspadpcmheader.coef[9]  = adpcminfo.coef[9];
    dspadpcmheader.coef[10] = adpcminfo.coef[10];
    dspadpcmheader.coef[11] = adpcminfo.coef[11];
    dspadpcmheader.coef[12] = adpcminfo.coef[12];
    dspadpcmheader.coef[13] = adpcminfo.coef[13];
    dspadpcmheader.coef[14] = adpcminfo.coef[14];
    dspadpcmheader.coef[15] = adpcminfo.coef[15];
    dspadpcmheader.gain     = adpcminfo.gain;
    dspadpcmheader.ps       = adpcminfo.pred_scale;
    dspadpcmheader.yn1      = adpcminfo.yn1;
    dspadpcmheader.yn2      = adpcminfo.yn2;
    dspadpcmheader.lps      = adpcminfo.loop_pred_scale;
    dspadpcmheader.lyn1     = adpcminfo.loop_yn1;
    dspadpcmheader.lyn2     = adpcminfo.loop_yn2;

    if (verbose_flag) printf(" Done.\nWriting DSPADPCM file...");

    // write the output file
    fwrite(&dspadpcmheader, 1, sizeof(DSPADPCMHEADER), output_file);
    fwrite(outputbuffer, getBytesForAdpcmSamples(soundinfo->samples), sizeof(u8), output_file);

    if (coef_flag)
    {
        if (verbose_flag) printf(" Done.\nWriting text dump file ...");
        dump_header(&dspadpcmheader, coef_file, 0);
    }

    if (verbose_flag) printf(" Done.\n");
}
/*--------------------------------------------------------------------------*
 *
 *--------------------------------------------------------------------------*/
void encode_soundfile(char *path, SOUNDINFO *soundinfo)
{
    DSPADPCMHEADER dspadpcmheader;
    SOUNDINFO      splitinfo;
    void           *soundbuffer,
                   *channelBuffer,
                   *outputbuffer;
   u16 *p_sound_buffer;
   u16 *p_channel_buffer;
   int channel;
   int index;

   soundbuffer   = malloc(soundinfo->bufferLength);
   channelBuffer = malloc(soundinfo->bufferLength / soundinfo->channels);
   outputbuffer  = malloc(getBytesForAdpcmBuffer(soundinfo->samples));

   //if (verbose_flag) printf("\n:::::::: channelBuffer size %d  outputbuffer size %d", soundinfo->bufferLength / soundinfo->channels, getBytesForAdpcmBuffer(soundinfo->samples));

   if (!soundbuffer || !channelBuffer || !outputbuffer)
   {
      if (soundbuffer)   free(soundbuffer);
      if (channelBuffer) free(channelBuffer);
      if (outputbuffer)  free(outputbuffer);

      printf("Cannot allocate buffers for encode!\n");
      return;
   }

   // now tease apart the channels
   // not worrying about super-efficient code here
   // this is an offline tool
   getSoundSamples(path, soundinfo, soundbuffer);

   for (channel = 0; channel < soundinfo->channels; channel++)
   {
      p_sound_buffer = &((u16*)soundbuffer)[channel];
      p_channel_buffer = (u16*)channelBuffer;

      for (index = 0; (index < soundinfo->samples); index++)
      {

         *p_channel_buffer = *p_sound_buffer;
         p_sound_buffer += soundinfo->channels;
         p_channel_buffer++;
      }

      if (0 != channel)
      {
         // multi-channel file,
         // so close previous file
         if (output_file) fclose(output_file);
         output_file = NULL;
         if (coef_file) fclose(coef_file);
         coef_file = NULL;

         // synthesize name for this channel
         if (channel > 1)
         {
            sprintf(&output_path[strlen(output_path)-7], "%c.adpcm", '1'+channel);
            if (coef_flag)
            {
                sprintf(&coef_path[strlen(coef_path)-5], "%c.txt", '1'+channel);
            }
         }
         else
         {
            sprintf(&output_path[strlen(output_path)-6], "_%c.adpcm", '1'+channel);
            if (coef_flag)
            {
                sprintf(&coef_path[strlen(coef_path)-4], "_%c.txt", '1'+channel);
            }
         }
         // and re-open output_file
         output_file = fopen(output_path, "wb");
         if (!output_file)
         {
            printf("Unable to creat file %s\r\n", output_path);
            break;
         }
         if (coef_flag)
         {
             coef_file = fopen(coef_path, "w");
             if (!coef_path)
             {
                 printf("Unable to creat file %s\r\n", coef_path);
                 break;
             }
         }
      }

      if (split_flag)
      {
         strcpy(split_path, output_path);
         if (channel > 0)
         {
            strcpy(&split_path[strlen(split_path)-3], "wav");
         }
         else
         {
            sprintf(&split_path[strlen(split_path)-4], "_1.wav");
         }
         splitinfo = *soundinfo;
         splitinfo.channels = 1;
         splitinfo.bufferLength = soundinfo->samples*2;
         printf("splitting channel %d to file %s\r\n", channel, output_path);
         writeWaveFile(split_path, &splitinfo, channelBuffer);
      }

      if (encode_flag)
      {
          if (verbose_flag)
          {
              printf("generating DSPADPCM for channel %d to file %s\r\n", channel, output_path);
          }
          memset(&dspadpcmheader, 0, sizeof(DSPADPCMHEADER));
          encode_soundfile_channel(soundinfo, dspadpcmheader, channelBuffer, outputbuffer);
      }
   }

   if (soundbuffer)
   {
       free(soundbuffer);
   }
   if (channelBuffer)
   {
       free(channelBuffer);
   }
   if (outputbuffer)
   {
       free(outputbuffer);
   }
}



/*--------------------------------------------------------------------------*
 *
 *--------------------------------------------------------------------------*/
void encode_interleaved_soundfile (
    SOUNDINFO *soundinfo,
    u16 *soundbuffers,
    u8  *outputbuffer
    )
{
   u32 ch;
   u32 channels    = soundinfo->channels;
   u32 samples     = soundinfo->samples;
   DSPADPCMHEADER dspadpcmheader[6];
   ADPCMINFO      adpcminfo;
   u32 totalWritten=0;
   u32 interleave_frame_byte_count = DEFAULT_INTERLEAVED_FRAME_COUNT * 8;
   u32 frame;
   u32 num_full_frames_to_write;
   u32 size_last_frame_to_write;

   u32 outputbufferChannelSize = getBytesForAdpcmBuffer(samples);

   if (verbose_flag) printf("Encoding %zd byte headers and %d audio samples into %d padded ADPCM bytes for %d channels\n", sizeof(DSPADPCMHEADER), samples, outputbufferChannelSize, channels);

   for (ch=0; ch<channels; ++ch)
   {
      if (verbose_flag) printf("\tCalculating ADPCM headers for channel %d...", ch);

      memset(&dspadpcmheader[ch], 0, sizeof(DSPADPCMHEADER));

      dspadpcmheader[ch].multi_ch_count = channels;
      dspadpcmheader[ch].block_frame_count  = DEFAULT_INTERLEAVED_FRAME_COUNT; // Default to 1024 ADPCM frames per interleaved channel frame

      dspadpcmheader[ch].num_samples       = samples;
      dspadpcmheader[ch].num_adpcm_nibbles = getNibblesForNSamples(samples);
      dspadpcmheader[ch].sample_rate       = soundinfo->sampleRate;

      // if the user specified loop points on the commandline use them
      // or else look for loop points in the adpcminfo
      if (loop_flag)
      {
          u32 nibbleLoopStart, nibbleLoopEnd, nibbleCurrent;

          nibbleLoopStart = getNibbleAddress(loopStart);
          nibbleLoopEnd   = getNibbleAddress(loopEnd);
          nibbleCurrent   = getNibbleAddress(0);

          dspadpcmheader[ch].loop_flag = VoiceType_Looped;
          dspadpcmheader[ch].format    = DecodeMode_Adpcm;
          dspadpcmheader[ch].sa        = nibbleLoopStart;
          dspadpcmheader[ch].ea        = nibbleLoopEnd;
          dspadpcmheader[ch].ca        = nibbleCurrent;
      }
      else if (soundinfo->loopEnd) // the sound info has loops form AIFF
      {
          u32 nibbleLoopStart, nibbleLoopEnd, nibbleCurrent;

          nibbleLoopStart = getNibbleAddress(soundinfo->loopStart);
          nibbleLoopEnd   = getNibbleAddress(soundinfo->loopEnd - 1); // AIFF loop end is 1 based
          nibbleCurrent   = getNibbleAddress(0);

          dspadpcmheader[ch].loop_flag = VoiceType_Looped;
          dspadpcmheader[ch].format    = DecodeMode_Adpcm;
          dspadpcmheader[ch].sa        = nibbleLoopStart;
          dspadpcmheader[ch].ea        = nibbleLoopEnd;
          dspadpcmheader[ch].ca        = nibbleCurrent;
      }
      else // not looped
      {
          u32 nibbleLoopStart, nibbleLoopEnd, nibbleCurrent;

          nibbleLoopStart = getNibbleAddress(0);

          // if the user specified end address use it
          if (ea_flag)
          {
              nibbleLoopEnd = getNibbleAddress(sampleEndAddr);
          }
          else
          {
              nibbleLoopEnd = getNibbleAddress(samples - 1);
          }

          nibbleCurrent = getNibbleAddress(0);

          dspadpcmheader[ch].loop_flag = VoiceType_OneShot;
          dspadpcmheader[ch].format    = DecodeMode_Adpcm;
          dspadpcmheader[ch].sa        = nibbleLoopStart;
          dspadpcmheader[ch].ea        = nibbleLoopEnd;
          dspadpcmheader[ch].ca        = nibbleCurrent;
      }

      memset(&adpcminfo, 0, sizeof(ADPCMINFO));

      encode((void*)(soundbuffers + ch*samples), (void*)(outputbuffer + ch * outputbufferChannelSize), &adpcminfo, samples);

      if (verbose_flag) printf("Getting loop context...");

      // if the user specified loops get loop context
      if (loop_flag)
         getLoopContext((outputbuffer + ch*outputbufferChannelSize), &adpcminfo, loopStart);
      // see if the aiff file has loop points
      else if (soundinfo->loopEnd)
         getLoopContext((outputbuffer + ch*outputbufferChannelSize), &adpcminfo, soundinfo->loopStart);
      // no loop make sure loop context is 0
      else
         adpcminfo.loop_pred_scale = adpcminfo.loop_yn1 = adpcminfo.loop_yn2 = 0;

      if (verbose_flag) printf("Filling rest of dspadpcmheader...");

      dspadpcmheader[ch].coef[0]  = adpcminfo.coef[0];
      dspadpcmheader[ch].coef[1]  = adpcminfo.coef[1];
      dspadpcmheader[ch].coef[2]  = adpcminfo.coef[2];
      dspadpcmheader[ch].coef[3]  = adpcminfo.coef[3];
      dspadpcmheader[ch].coef[4]  = adpcminfo.coef[4];
      dspadpcmheader[ch].coef[5]  = adpcminfo.coef[5];
      dspadpcmheader[ch].coef[6]  = adpcminfo.coef[6];
      dspadpcmheader[ch].coef[7]  = adpcminfo.coef[7];
      dspadpcmheader[ch].coef[8]  = adpcminfo.coef[8];
      dspadpcmheader[ch].coef[9]  = adpcminfo.coef[9];
      dspadpcmheader[ch].coef[10] = adpcminfo.coef[10];
      dspadpcmheader[ch].coef[11] = adpcminfo.coef[11];
      dspadpcmheader[ch].coef[12] = adpcminfo.coef[12];
      dspadpcmheader[ch].coef[13] = adpcminfo.coef[13];
      dspadpcmheader[ch].coef[14] = adpcminfo.coef[14];
      dspadpcmheader[ch].coef[15] = adpcminfo.coef[15];
      dspadpcmheader[ch].gain     = adpcminfo.gain;
      dspadpcmheader[ch].ps       = adpcminfo.pred_scale;
      dspadpcmheader[ch].yn1      = adpcminfo.yn1;
      dspadpcmheader[ch].yn2      = adpcminfo.yn2;
      dspadpcmheader[ch].lps      = adpcminfo.loop_pred_scale;
      dspadpcmheader[ch].lyn1     = adpcminfo.loop_yn1;
      dspadpcmheader[ch].lyn2     = adpcminfo.loop_yn2;

      if (verbose_flag) printf("Done.\n");
   }

  if (verbose_flag) { printf("\tWriting DSPADPCM headers (%zd bytes each) for %d channels", sizeof(DSPADPCMHEADER), channels); }

   for (ch=0; (ch < channels); ++ch)
   {
      fwrite(&dspadpcmheader[ch], 1, sizeof(DSPADPCMHEADER), output_file);
      totalWritten += sizeof(DSPADPCMHEADER);
   }

   // Write interleaved data

   num_full_frames_to_write = outputbufferChannelSize / interleave_frame_byte_count;
   size_last_frame_to_write =  outputbufferChannelSize % interleave_frame_byte_count;

   if (verbose_flag)
   {
      printf("\n\tWriting %d DSPADPCM interleaved frames (%d bytes each)", channels*num_full_frames_to_write, interleave_frame_byte_count);
      if (size_last_frame_to_write) { printf(" and %d bytes each in the last %d frames)...", size_last_frame_to_write, channels); }
      else printf("...");
   }

   for (frame=0; (frame < num_full_frames_to_write); ++frame)
   {
      for (ch=0; (ch < channels); ++ch)
      {
         //if (verbose_flag) { if (ch==0) printf("\n\tWriting DSPADPCM interleave frame %x bytes %08x for channel %d", frame, interleave_frame_byte_count, ch); else printf(" %d", ch); }
         fwrite((outputbuffer + ch*outputbufferChannelSize + frame*interleave_frame_byte_count), interleave_frame_byte_count, sizeof(u8), output_file);
         totalWritten += interleave_frame_byte_count;
      }
   }

   // Write last frames
   if (size_last_frame_to_write)
   {
      for (ch=0; (ch < channels); ++ch)
      {
         //if (verbose_flag) { if (ch==0) printf("\n\tFinal   DSPADPCM interleave frame %x bytes %08x for channel %d", frame, size_last_frame_to_write, ch); else printf(" %d", ch); }
         fwrite((outputbuffer + ch*outputbufferChannelSize + frame*interleave_frame_byte_count), size_last_frame_to_write, sizeof(u8), output_file);
         totalWritten += size_last_frame_to_write;
      }
   }
   else
   {
      printf("\n\tNo trailing frames to write.");
   }

   if (coef_flag)
   {
       if (verbose_flag) printf("\nWriting text dump file...");
       for (ch=0; (ch < channels); ++ch)
       {
           dump_header(dspadpcmheader+ch, coef_file, ch+1);
       }
   }

   if (verbose_flag) printf("\nWrote %d bytes total.  Done.\n", totalWritten);
}
/*--------------------------------------------------------------------------*
 *
 *--------------------------------------------------------------------------*/
void interleave_soundfile(char *path, SOUNDINFO *soundinfo)
{
   u16   *soundbuffer;
   u16   *channelBuffers;
   u8    *outputbuffer;
   u32 ch;
   u32 index;
   u32 bufferLength = soundinfo->bufferLength;
   u32 samples      = soundinfo->samples;
   u32 channels     = soundinfo->channels;

   soundbuffer   = malloc(bufferLength);
   channelBuffers= malloc(bufferLength);
   outputbuffer  = malloc(getBytesForAdpcmBuffer(samples)*channels);

   //if (verbose_flag) printf("\n:::::::: channels %d  channelBuffer size %d  outputbuffer size %d", channels, bufferLength, getBytesForAdpcmBuffer(samples)*channels);

   if (!soundbuffer || !channelBuffers || !outputbuffer)
   {
      if (soundbuffer)    { free(soundbuffer);   }
      if (channelBuffers) { free(channelBuffers); }
      if (outputbuffer)   { free(outputbuffer);  }
      printf("Cannot allocate buffers for encode!\n");
      return;
   }

   if (verbose_flag) printf("Processing sound file...");

   // Tease apart the channels.  Not worrying about super-efficient code here as this is an offline tool.
   getSoundSamples(path, soundinfo, soundbuffer);

   // Convert the sample interleaved buffer into channle interleaved
   // 012345012345012345...  =>  000...00111...11222...22333...33444...44555...55
   if (verbose_flag) { printf("Un-interleaving samples to interleaved channels..."); }
   for (index = 0; (index < samples); index++)
   {
      for (ch = 0; (ch < channels); ch++)
      {
         channelBuffers[index+ch*samples] = soundbuffer[index*channels+ch];
      }
   }

   if (verbose_flag) printf("done.\n");

   // Write the headers...
   encode_interleaved_soundfile(soundinfo, channelBuffers, outputbuffer);
}



/*--------------------------------------------------------------------------*
 *
 *--------------------------------------------------------------------------*/
int encode_input_file(BOOL interleave_mode)
{
   SOUNDINFO soundinfo;

   if (verbose_flag) printf("Getting sound header from '%s'...\n", input_path);

   switch (getSoundInfo(input_path, &soundinfo))
   {
      case SoundFileResult_Success:

         if (soundinfo.bitsPerSample != 16)
         {
             printf("Sound file buffer not 16bit samples!\n");
             return FALSE;
         }
 /*
         if (soundinfo.channels != 1)
         {
             printf("Soundfile buffer not mono!\n");
             return;
         }
 */
         if (verbose_flag) printf("done.\n");

         if (interleave_mode)
         {
            interleave_soundfile(input_path, &soundinfo);
         }
         else
         {
            encode_soundfile(input_path, &soundinfo);
         }

         break;

      case SoundFileResult_InvalidFormat:

         printf("Sound file format not supported!\n");

         return FALSE;

      case SoundFileResult_OperationFailed:

         printf("fopen error for sound file!\n");

         return FALSE;
      case SoundFileResult_FileCorrupted:

         printf("Sound file corrupted!\n");

         return FALSE;
   }
   return TRUE;
}

u32 roundUpByX (u32 num, s32 round)
{
    return (num - 1 + round) & -round;
}

/*--------------------------------------------------------------------------*
 * Decodes one of the 6 channels.  Passing in ch=0 implies a non-interlaced
 * encoded file thus just one channel.
 *--------------------------------------------------------------------------*/
void decode_input_file_channel (DSPADPCMHEADER *dspadpcmheader, u32 multiChCount, u32 blockFrameCount, u32 ch)
{
    u32  samples, sampleRate;
    u8  *inputBuffer;
    u16 *outputBuffer;
    u32  blockByteCount;  // Number of bytes per interleaved block
    u32  numBytesPerChannel; // Number of bytes of each interleaved channel (excluding the headers)
    u32  bytesToRead; // Number of bytes of each interleaved channel (excluding the headers)
    u32  bytesRead=0; // Number of bytes of each interleaved channel (excluding the headers)
    char filename[256];
    size_t  result;
    BOOL channelCount; // Keep track of channel count separate from multiChCount which might be 0 indicating a non-interlaced file.

    if (verbose_flag) printf("Getting DSPADPCM header...");

    channelCount = (0 == multiChCount) ? 1 : multiChCount;

    blockByteCount  = blockFrameCount * 8;

    samples         = dspadpcmheader->num_samples;
    sampleRate      = dspadpcmheader->sample_rate;
    numBytesPerChannel = roundUpByX(getBytesForAdpcmSamples(samples), 8); // Account for the last ADPCM frame padding in the last interleaved block.

    //printf("\n\n::"__FUNCTION__" HEADER %p  ch %d/%d  blockFrameCount %d  blockByteCount %d  samples %d", dspadpcmheader, ch, multiChCount, blockFrameCount, blockByteCount, samples);

    // allocate buffers
    if (inputBuffer = malloc(getBytesForAdpcmSamples(samples)))
    {
        if (verbose_flag) printf(" Done.\nGetting ADPCM samples from channel %d out of %d...", ch, channelCount);

        if (outputBuffer = malloc(samples * 2))
        {
            SOUNDINFO soundinfo;
            ADPCMINFO adpcminfo;

            // prepare adpcminfo for decoding
            adpcminfo.coef[0]    = (s16)dspadpcmheader->coef[0];
            adpcminfo.coef[1]    = (s16)dspadpcmheader->coef[1];
            adpcminfo.coef[2]    = (s16)dspadpcmheader->coef[2];
            adpcminfo.coef[3]    = (s16)dspadpcmheader->coef[3];
            adpcminfo.coef[4]    = (s16)dspadpcmheader->coef[4];
            adpcminfo.coef[5]    = (s16)dspadpcmheader->coef[5];
            adpcminfo.coef[6]    = (s16)dspadpcmheader->coef[6];
            adpcminfo.coef[7]    = (s16)dspadpcmheader->coef[7];
            adpcminfo.coef[8]    = (s16)dspadpcmheader->coef[8];
            adpcminfo.coef[9]    = (s16)dspadpcmheader->coef[9];
            adpcminfo.coef[10]   = (s16)dspadpcmheader->coef[10];
            adpcminfo.coef[11]   = (s16)dspadpcmheader->coef[11];
            adpcminfo.coef[12]   = (s16)dspadpcmheader->coef[12];
            adpcminfo.coef[13]   = (s16)dspadpcmheader->coef[13];
            adpcminfo.coef[14]   = (s16)dspadpcmheader->coef[14];
            adpcminfo.coef[15]   = (s16)dspadpcmheader->coef[15];
            adpcminfo.gain       = dspadpcmheader->gain;
            adpcminfo.pred_scale = 0;
            adpcminfo.yn1        = 0;
            adpcminfo.yn2        = 0;

            // read adpcm samples into input buffer

            fseek(input_file, (channelCount * sizeof(DSPADPCMHEADER)), SEEK_SET); // Skip all headers

            // If the blockByteCount is 0 then this is a non interleaved file so just assume the data is one big block.
            if (0 == blockByteCount) { blockByteCount = numBytesPerChannel; }

            /* Exctract out all data of a single channel from the interleaved ADPCM blocks.  This requires skipping
               a number of blocks, reading the channel block, then skpping more again to start the process over again.
               Ultimatley the last set of blocks (each less than the block size and possibly padded) are copied.
               Read this many bytes (in a block) at a time
            */

            bytesToRead = (numBytesPerChannel < blockByteCount) ? numBytesPerChannel : blockByteCount;

            while (numBytesPerChannel)
            {
                result = fseek(input_file, (ch * bytesToRead), SEEK_CUR); // Skip to this channel's next interleaved block
                result = fread((inputBuffer + bytesRead), 1, bytesToRead, input_file); // Read an interleaved block.  If it's the last block, might read in less than a full block
                result = fseek(input_file, ((channelCount-(ch+1)) * bytesToRead), SEEK_CUR); // Skip to next set of blocks

                bytesRead += bytesToRead;
                numBytesPerChannel -= bytesToRead;

                // Compute next block size.  It may not be a full block size since the last block of each channel could be partially filled.
                bytesToRead = (numBytesPerChannel < blockByteCount) ? numBytesPerChannel : blockByteCount;
            }

            if (verbose_flag) printf(" Done.\nDecoding samples...");

            // decode samples to buffer
            decode(
                inputBuffer,
                outputBuffer,
                &adpcminfo,
                samples
                );

            soundinfo.bitsPerSample = 16;
            soundinfo.bufferLength  = samples * 2;
            soundinfo.channels      = 1;
            soundinfo.sampleRate    = sampleRate;
            soundinfo.samples       = samples;

            if (dspadpcmheader->loop_flag)
            {
                soundinfo.loopStart = getSampleForAdpcmNibble(dspadpcmheader->sa);
                soundinfo.loopEnd   = getSampleForAdpcmNibble(dspadpcmheader->ea) + 1;
            }
            else
            {
                soundinfo.loopStart = 0;
                soundinfo.loopEnd   = 0;
            }

            if (verbose_flag) printf(" Done.\nWriting sound file...");

            // In multi channel decoding mode, append "_channel" to the file name or before the last '.'
            if (multiChCount)
            {
                char *dot = strrchr(output_path, '.');
                int   len;
                if (dot)
                {
                    len = (int)(dot-output_path);
                    strncpy(filename, output_path, len);
                    sprintf(filename+len, "_%d%s", ch, dot+(len-1));
                }
                else
                {
                    sprintf(filename, "%s_%d", output_path, ch);
                }
            }
            else
            {
                sprintf(filename, output_path);
            }

            switch (decodeFormat)
            {
                case DECODE_WAV:

                    writeWaveFile(filename, &soundinfo, outputBuffer);

                    break;

                case DECODE_AIFF:

                    writeAiffFile(filename, &soundinfo, outputBuffer);

                    break;
            }
        }
        else
        {
            printf("\nERROR: Cannot allocate output buffer!\n");
            clean_up();
            exit(1);
        }
    }
    else
    {
        printf("\nERROR: Cannot allocate input buffer!\n");
        clean_up();
        exit(1);
    }

    // free buffers
    if (inputBuffer)  free(inputBuffer);
    if (outputBuffer) free(outputBuffer);

    // see if we should write a coefficient file
    if (coef_flag)
    {
        if (verbose_flag) printf(" Done.\nWriting text dump file...");
        dump_header(dspadpcmheader, coef_file, 0);
    }
    if (verbose_flag) printf(" Done.\n");
}


/* Decode dsp or mdsp file into multiple wav files.
 */
BOOL decode_input_file (void)
{
    DSPADPCMHEADER dspadpcmheader[6];
    u32 multi_ch_count; // Number of interleaved channels
    u32 block_frame_count; // Number of 8-byte ADPCM frames per interleaved block
    u32 i;

    if (verbose_flag) printf("Getting initial DSPADPCM header...");

    fread(dspadpcmheader+0, 1, sizeof(DSPADPCMHEADER), input_file);

    multi_ch_count = dspadpcmheader[0].multi_ch_count;
    block_frame_count = dspadpcmheader[0].block_frame_count;

    if (multi_ch_count > 6)
    {
        return FALSE;
    }

    // Read the remaining DSPADPCM headers...
    if (1 < multi_ch_count)
    {
        for (i=1; (i < multi_ch_count); ++i)
        {
            fread(dspadpcmheader+i, 1, sizeof(DSPADPCMHEADER), input_file);
        }
    }

    if (0 == multi_ch_count)
    {
        decode_input_file_channel(dspadpcmheader, 0, 0, 0);
    }
    else
    {
        for (i=0; (i < multi_ch_count); ++i)
        {
            decode_input_file_channel(&dspadpcmheader[i], multi_ch_count, block_frame_count, i);
        }
    }

    return TRUE;
}


/*--------------------------------------------------------------------------*/
int get_dll(void)
{
    hDllDsptool = LoadLibrary("AdpcmCore.dll");
    if (hDllDsptool)
    {
        getBytesForAdpcmBuffer  = (lpFunc1)GetProcAddress(hDllDsptool, "getBytesForAdpcmBuffer");
        getBytesForAdpcmSamples = (lpFunc1)GetProcAddress(hDllDsptool, "getBytesForAdpcmSamples");
        getBytesForPcmBuffer    = (lpFunc1)GetProcAddress(hDllDsptool, "getBytesForPcmBuffer");
        getBytesForPcmSamples   = (lpFunc1)GetProcAddress(hDllDsptool, "getBytesForPcmSamples");
        getNibbleAddress        = (lpFunc1)GetProcAddress(hDllDsptool, "getNibbleAddress");
        getNibblesForNSamples   = (lpFunc1)GetProcAddress(hDllDsptool, "getNibblesForNSamples");
        getSampleForAdpcmNibble = (lpFunc1)GetProcAddress(hDllDsptool, "getSampleForAdpcmNibble");
        getBytesForAdpcmInfo    = (lpFunc2)GetProcAddress(hDllDsptool, "getBytesForAdpcmInfo");
        encode                  = (lpFunc3)GetProcAddress(hDllDsptool, "encode");
        decode                  = (lpFunc4)GetProcAddress(hDllDsptool, "decode");
        getLoopContext          = (lpFunc5)GetProcAddress(hDllDsptool, "getLoopContext");

        if (
            !getBytesForAdpcmBuffer  ||
            !getBytesForAdpcmSamples ||
            !getBytesForPcmBuffer    ||
            !getBytesForPcmSamples   ||
            !getNibbleAddress        ||
            !getNibblesForNSamples   ||
            !getSampleForAdpcmNibble ||
            !getBytesForAdpcmInfo    ||
            !encode                  ||
            !decode                  ||
            !getLoopContext
            ) return FALSE;
    }

    hDllSoundfile = LoadLibrary("AudioFile.dll");
    if (hDllSoundfile)
    {
        getSoundInfo    = (lpFunc6)GetProcAddress(hDllSoundfile, "getSoundInfo");
        getSoundSamples = (lpFunc7)GetProcAddress(hDllSoundfile, "getSoundSamples");
        writeWaveFile   = (lpFunc8)GetProcAddress(hDllSoundfile, "writeWaveFile");
        writeAiffFile   = (lpFunc8)GetProcAddress(hDllSoundfile, "writeAiffFile");

        if (
            !getSoundInfo    ||
            !getSoundSamples ||
            !writeWaveFile   ||
            !writeAiffFile
            ) return FALSE;
    }

    if (hDllDsptool && hDllSoundfile)
    {
        return TRUE;
    }
    else
    {
        return FALSE;
    }
}


/*--------------------------------------------------------------------------*
 * main()
 *--------------------------------------------------------------------------*/
int main(int argc, char *argv[])
{
    setlocale(LC_CTYPE, "");

   init();

   if (FALSE == parse_args(argc, argv))
   {
      // don't print usage help, it just adds more junk to
      // log files. print usage only for missing parameters or explicit
      // requests for help.
      exit(1);
   }

   // load the dspadpcm.dll
   if (FALSE == get_dll())
   {
      printf("\nERROR: Failed to load dll\n");
      exit(1);
   }

   // open files
   if (decode_flag)
   {
      if ((input_file = fopen(input_path, "rb")) == NULL)
      {
         printf("\nERROR: Cannot open %s for reading!\n", input_path);
         clean_up();
         exit(1);
      }
   }

   if (encode_flag || interleave_flag)
   {
      if ((output_file = fopen(output_path, "wb")) == NULL)
      {
         printf("\nERROR: Cannot open %s for writing!\n", output_path);
         clean_up();
         exit(1);
      }
   }

   if (coef_path && coef_flag)
   {
      if ((coef_file = fopen(coef_path, "w")) == NULL)
      {
         printf("\nERROR: Cannot open %s for writing!\n", coef_path);
         clean_up();
         exit(1);
      }
   }

   // encode or decode
   if (interleave_flag)
   {
      if (!encode_input_file(INTERLEAVE_MODE))
      {
         printf("\nERROR: encode failed!\n");
         clean_up();
         remove(output_path);
         if (coef_flag)
         {
             remove(coef_path);
         }
         exit(1);
      }
   }
   else if (encode_flag || split_flag)
   // bug 5830: capture and obey error return from encode_input_file (used to be void)
   {
      if (!encode_input_file(NON_INTERLEAVE_MODE))
      {
         printf("\nERROR: encode failed!\n");
         clean_up();
         remove(output_path);
         if (coef_flag)
         {
            remove(coef_path);
         }
         exit(1);
      }
   }
   else if (decode_flag)
   {
      if (decode_input_file() == FALSE)
      {
         exit(1);
      }
   }

   // clean up
   clean_up();

   // exit with a clean bill of health
   exit(0);

} // end main()
