﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "coredump_FormatWriter.h"

#ifdef NN_NINTENDO_SDK
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_File.h>
#include <nn/fs.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#define REPORT( ... ) NN_SDK_LOG(__VA_ARGS__)
#else
#define REPORT( ... ) printf( __VA_ARGS__ )
#endif
#include <cstdlib>

//==============================================================================
namespace coredump {
//==============================================================================

enum: u64
{
    INVALID_MEMORY_SECTION = 0xFFFFFFFFFFFFFFFF
};

coredump_writer::coredump_writer()
: m_pOutput(NULL)
, m_OpenMemorySection(INVALID_MEMORY_SECTION)
, m_FileOffset(0)
#ifndef WIN32
, m_FileBufferOffset(0)
, m_FileWriteOffset(0)
#endif
, m_OpenImageSectionSize(0)
, m_OpenVideoSectionSize(0)
{
}

//==============================================================================

bool coredump_writer::Init(coredump_file_header* pInfo, char* pFileName)
{
    if (m_pOutput != NULL)
    {
        Close();
    }

    if (m_pOutput == NULL)
    {
#ifndef WIN32
        m_FileOffset = 0;
        m_FileWriteOffset = 0;
#endif
        m_OpenMemorySection = INVALID_MEMORY_SECTION;
        m_OpenImageSectionSize = 0;
        m_OpenVideoSectionSize = 0;

        Open( pFileName );

        if (m_pOutput != NULL)
        {
            AddData("DUMP", pInfo, sizeof(coredump_file_header));
            coredump_file_version versionInfo;
            AddData(pVersionTag, &versionInfo, sizeof(versionInfo));
        }
    }
    return (m_pOutput != NULL);
}

//==============================================================================

coredump_writer::coredump_writer( coredump_file_header* pInfo, char* pFileName )
:   m_pOutput(NULL)
,   m_OpenMemorySection( INVALID_MEMORY_SECTION )
,   m_FileOffset( 0 )
#ifndef WIN32
,   m_FileBufferOffset( 0 )
,   m_FileWriteOffset( 0 )
#endif
{
    Init(pInfo, pFileName);
}

//==============================================================================

coredump_writer::~coredump_writer( )
{
    if( m_pOutput != NULL )
    {
         Close();
    }
}

//========================================================================================================
// Platform-independent file IO
//========================================================================================================

void coredump_writer::Open( char* pFileName )
{
#if WIN32
    fopen_s( &m_pOutput, pFileName, "wb" );
#else
    nn::Result Ret = nn::fs::OpenFile( &m_Output, pFileName, nn::fs::OpenMode_Write|nn::fs::OpenMode_AllowAppend );
    if( Ret.IsSuccess() )
    {
        m_pOutput = &m_Output;
        m_FileOffset = 0;
        m_FileWriteOffset = 0;
    }
#endif
}

//========================================================================================================

void coredump_writer::Close()
{
    if( m_pOutput != NULL )
    {
#if WIN32
        fclose( m_pOutput );
#else
        Flush();
        nn::fs::CloseFile( m_Output );
        m_FileOffset = 0;
        m_FileWriteOffset = 0;
#endif
        m_OpenMemorySection = INVALID_MEMORY_SECTION;
        m_pOutput = NULL;
    }
}

//========================================================================================================

result coredump_writer::Flush()
{
    result Ret = RESULT_COREDUMP_OK;
#ifndef WIN32
    if (m_FileBufferOffset != 0)
    {
        nn::fs::WriteOption writeOption = { nn::fs::WriteOptionFlag_Flush };
        nn::Result Res = nn::fs::WriteFile((nn::fs::FileHandle)m_Output, m_FileWriteOffset, m_FileBuffer, m_FileBufferOffset, writeOption);
        if (Res.IsSuccess() == true)
        {
            m_FileWriteOffset += m_FileBufferOffset;
        }
        else
        {
            REPORT("coredump_writer::Write failure:  Error = 0x%x, Module %d Description %d\n", Res.GetInnerValueForDebug(), Res.GetModule(), Res.GetDescription());
            Ret = RESULT_COREDUMP_IO_ERROR;
        }
        m_FileBufferOffset = 0;
    }
#endif
    return Ret;
}

//========================================================================================================

result coredump_writer::Write( s64 Offset, void* pData, u64 SizeOfData )
{
    if( m_pOutput != NULL )
    {
        result Ret = RESULT_COREDUMP_OK;
#if WIN32
        fseek( m_pOutput, (long)Offset, SEEK_SET );
        fwrite( pData, 1, (size_t)SizeOfData, m_pOutput );
        fseek( m_pOutput, 0, SEEK_END );
#else
        if (Offset < m_FileWriteOffset)
        {
            nn::fs::WriteOption writeOption = { nn::fs::WriteOptionFlag_Flush };
            nn::Result Res = nn::fs::WriteFile((nn::fs::FileHandle)m_Output, Offset, pData, SizeOfData, writeOption);
            if (Res.IsFailure())
            {
                REPORT("coredump_writer::Write failure:  Error = 0x%x, Module %d Description %d\n", Res.GetInnerValueForDebug(), Res.GetModule(), Res.GetDescription());
                Ret = RESULT_COREDUMP_IO_ERROR;
            }
        }
        else
        {
            // This is somewhere within our current buffer.
            s64 Diff = Offset - m_FileWriteOffset;
            char* pWriteTo = m_FileBuffer + Diff;
            memcpy( pWriteTo, pData, SizeOfData );
        }
#endif
        return Ret;
    }

    return RESULT_COREDUMP_UNABLE_TO_OPEN_OUTPUT;
}

//========================================================================================================

result coredump_writer::Write( void* pData, u64 SizeOfData )
{
    if( m_pOutput != NULL )
    {
        result Ret = RESULT_COREDUMP_OK;
        if (SizeOfData > 0)
        {
#if WIN32
            fwrite(pData, 1, (size_t)SizeOfData, m_pOutput);
            m_FileOffset += SizeOfData;
#else
#if USE_UNBUFFERED_WRITE
            nn::fs::WriteOption writeOption = { nn::fs::WriteOptionFlag_Flush };
            nn::Result Ret = nn::fs::WriteFile((nn::fs::FileHandle)m_Output, m_FileOffset, pData, SizeOfData, writeOption);
            if (Ret.IsSuccess() == true)
            {
                m_FileOffset += SizeOfData;
                m_FileWriteOffset = m_FileOffset;
            }
            else
            {
                REPORT("coredump_writer::Write failure:  Error = 0x%x, Module %d Description %d\n", Ret.GetInnerValueForDebug(), Ret.GetModule(), Ret.GetDescription());
            }
#else
            s64 AmountToWrite = SizeOfData;
            char* pReadFrom = (char*)pData;
            do
            {
                char* pWriteTo = m_FileBuffer + m_FileBufferOffset;
                s64 fileBufferSizeLeft = SIZE_OF_FILE_BUFFER - m_FileBufferOffset;
                s64 fileBufferSizeToWrite = fileBufferSizeLeft > AmountToWrite ? AmountToWrite : fileBufferSizeLeft;
                if (fileBufferSizeToWrite <= 0)
                {
                    REPORT("coredump_writer::Write ERROR:  fileBufferSizeToWrite %lld, AmountToWrite %lld, fileBufferSizeLeft %lld, m_FileBufferOffset %lld\n", fileBufferSizeToWrite, AmountToWrite, fileBufferSizeLeft, m_FileBufferOffset);
                }

                memcpy(pWriteTo, pReadFrom, fileBufferSizeToWrite);
                m_FileBufferOffset += fileBufferSizeToWrite;

                m_FileOffset += fileBufferSizeToWrite;
                if (m_FileBufferOffset >= SIZE_OF_FILE_BUFFER)
                {
                    result FlushRet = Flush();
                    if (FlushRet != RESULT_COREDUMP_OK)
                    {
                        Ret = FlushRet;
                        break;
                    }
                }

                pReadFrom += fileBufferSizeToWrite;
                AmountToWrite -= fileBufferSizeToWrite;
                if (AmountToWrite < 0)
                {
                    REPORT("coredump_writer::Write ERROR:  AmountToWrite < 0:  fileBufferSizeToWrite %lld, AmountToWrite %lld, fileBufferSizeLeft %lld, m_FileBufferOffset %lld\n", fileBufferSizeToWrite, AmountToWrite, fileBufferSizeLeft, m_FileBufferOffset);
                }
            }
            while (AmountToWrite > 0);
#endif
#endif
        }
        return Ret;
    }

    return RESULT_COREDUMP_UNABLE_TO_OPEN_OUTPUT;
}

//==============================================================================

//==============================================================================

result coredump_writer::CreateMemorySection( u64 Address, MemoryType Type, u64 Size, AllocateFunction Alloc, DeallocateFunction Dealloc )
{
    //Do we already have a memory section open?
    if( m_OpenMemorySection != INVALID_MEMORY_SECTION )
    {
        return RESULT_COREDUMP_MEMORY_SECTION_ALREADY_OPEN;
    }

    //Save this marker so we'll recognize out-of-order writing.
    m_OpenMemorySection = Address;

    //==============================================================================
    // Because we're compressing our data, we can't be sure how much space we'll
    // need.  Remember where we started, and we can come back later and write
    // the correct size.
    //==============================================================================
    m_OpenMemorySectionFileOffset = m_FileOffset;

#if USE_ZLIB
    coredump_file_data_header Header( pZLibCompressedMemoryTypeTag, Size + sizeof(coredump_memory_header) );
#else
    coredump_file_data_header Header( pCompressedMemoryTypeTag, Size + sizeof(coredump_memory_header) );
#endif

    //Write the data header
    if( Write( &Header, sizeof(Header) ) == RESULT_COREDUMP_OK )
    {
        //==============================================================================
        // Made it here, so we're ready to write.  Init our compression stream object.
        //==============================================================================
#if USE_ZLIB
        memset( &m_MemoryCompressor, 0, sizeof(m_MemoryCompressor) );
        m_MemoryCompressor.zalloc = Alloc;
        m_MemoryCompressor.zfree = Dealloc;
        m_MemoryCompressor.opaque = (voidpf)0;
        int ZRet = deflateInit( &m_MemoryCompressor, Z_DEFAULT_COMPRESSION );
        if( ZRet != Z_OK)
        {
            REPORT( "ZLIB ERROR:  deflateInit returned %d\n", ZRet );
        }
#else
        LZ4_resetStream( &m_CompressionStream );
#endif
        //Write the memory data header
        coredump_memory_header MemoryHeader( Address, Type, Size );
        return Write( &MemoryHeader, sizeof(MemoryHeader) );
    }

    return RESULT_COREDUMP_UNABLE_TO_OPEN_OUTPUT;
}

//==============================================================================

u64 coredump_writer::CompressionBufferSizeRequired( u64 Size )
{
    u64 Required = Size;
#if USE_LZ4
    return LZ4_COMPRESSBOUND( Size );
#endif
    return Required;
}

//==============================================================================

result coredump_writer::CompressMemoryData( void* pSourceData, u64 SizeOfSourceData, void* pCompressionBuffer, u64 SizeOfCompressionBuffer, u64* pSizeOfCompressedData )
{
    result Ret = RESULT_COREDUMP_OK;
#if USE_ZLIB
    m_MemoryCompressor.next_in  = (z_const unsigned char *)pSourceData;
    m_MemoryCompressor.next_out = (z_const unsigned char *)pCompressionBuffer;
    m_MemoryCompressor.avail_in = m_MemoryCompressor.avail_out = (uInt)SizeOfSourceData;

//REPORT( "[coredump_writer::CompressMemoryData] Source = 0x%llx, CompressedBuffer = 0x%llx, Size = %d\n", m_MemoryCompressor.next_in, m_MemoryCompressor.next_out, m_MemoryCompressor.avail_in );
    int ZRet = deflate( &m_MemoryCompressor, Z_FULL_FLUSH );
    if( ZRet == Z_OK)
    {
        *pSizeOfCompressedData = ( SizeOfSourceData - m_MemoryCompressor.avail_out );
        Ret = RESULT_COREDUMP_OK;
    }
    else
    {
        REPORT( "ZLIB ERROR: deflate returned %d\n", ZRet );
        Ret = (coredump::result)ZRet;
    }
#else
    int CompressedSize = LZ4_compress_fast_continue( &m_CompressionStream, (const char*)pSourceData, (char*)pCompressionBuffer, (int)SizeOfSourceData, (int)SizeOfCompressionBuffer, 1 );
    if( CompressedSize > 0 )
    {
        *pSizeOfCompressedData = CompressedSize;
        Ret = RESULT_COREDUMP_OK;
    }
    else
    {
        REPORT( "LZ4 ERROR: LZ4_compress returned %d, Size of data = %lld\n", CompressedSize, SizeOfSourceData );
        Ret = (coredump::result)CompressedSize;
    }

#endif

    return Ret;
}

//==============================================================================

result coredump_writer::AddMemoryData( void* pData, u64 Size, void* pCompressionBuffer, u64 SizeOfCompressionBuffer )
{
    // Need to have an open memory section to write this.
    if( m_OpenMemorySection == INVALID_MEMORY_SECTION )
    {
        return RESULT_COREDUMP_MEMORY_SECTION_NOT_OPEN;
    }
    // Compress this...
    u64 CompressedSize = 0;
    result Ret = CompressMemoryData( pData, Size, pCompressionBuffer, SizeOfCompressionBuffer, &CompressedSize );
    if( Ret == RESULT_COREDUMP_OK )
    {
#if USE_LZ4
        //==============================================================================
        // For LZ4, we need the compressed chunk size for every write.  Write that
        // first before we write the compressed data so that the dump file reader will
        // know how big this compressed chunk is.
        //==============================================================================
        u32 WriteSize = (u32)CompressedSize;
        Ret = Write( &WriteSize, sizeof(WriteSize) );
#endif
        // Write it.
        Ret = Write( pCompressionBuffer, CompressedSize );
    }

    return Ret;
}

//==============================================================================

result coredump_writer::CloseMemorySection( u64 Address )
{
    if( m_OpenMemorySection == INVALID_MEMORY_SECTION )
    {
        return RESULT_COREDUMP_MEMORY_SECTION_NOT_OPEN;
    }

    //If we have one open, but it's not this one -
    if( m_OpenMemorySection != Address )
    {
        return RESULT_COREDUMP_MEMORY_SECTION_NOT_OPEN;
    }

    // "Close" this memory section.
    m_OpenMemorySection = INVALID_MEMORY_SECTION;

#if USE_ZLIB
    //====================================================
    // Clean up our compressor.
    deflateEnd( &m_MemoryCompressor );
#endif

    //==============================================================
    // When we started this memory chunk, we wrote the uncompressed
    // size.  Now we need to go back and write the correct size.
    //==============================================================

    // Here's how much we wrote, including our file data header:
    s64 TotalChunkSize = m_FileOffset - m_OpenMemorySectionFileOffset;

    // Here's our actual size:
    s64 ActualDataSize = TotalChunkSize - ( sizeof(coredump_file_data_header) );

    // Create a dummy header so we can measure our offset.
    coredump_file_data_header DummyHeader( pCompressedMemoryTypeTag, 0);

    //Here's where the size is added
    s64 WriteOffset = m_OpenMemorySectionFileOffset + sizeof(DummyHeader.m_Tag);

    // Write the size
    return Write( WriteOffset, &ActualDataSize, sizeof(u64) );
}

result coredump_writer::CreateUncompressedMemorySection( u64 Address, MemoryType Type, u64 Size )
{
    //Do we already have a memory section open?
    if( m_OpenMemorySection != INVALID_MEMORY_SECTION )
    {
        return RESULT_COREDUMP_MEMORY_SECTION_ALREADY_OPEN;
    }

    //Save this marker so we'll recognized out-of-order writing.
    m_OpenMemorySection = Address;

    coredump_file_data_header Header( pMemoryTypeTag, Size + sizeof(coredump_memory_header) );

    //Write the data header
    if( Write( &Header, sizeof(Header) ) == RESULT_COREDUMP_OK )
    {
        //Write the memory data header
        coredump_memory_header MemoryHeader( Address, Type, Size );
        return Write( &MemoryHeader, sizeof(MemoryHeader) );
    }

    return RESULT_COREDUMP_UNABLE_TO_OPEN_OUTPUT;
}

//==============================================================================

result coredump_writer::AddMemoryData( void* pData, u64 Size )
{
    // Need to have an open memory section to write this.
    if( m_OpenMemorySection == INVALID_MEMORY_SECTION )
    {
        return RESULT_COREDUMP_MEMORY_SECTION_NOT_OPEN;
    }

    return Write( pData, Size );
}

//==============================================================================

result coredump_writer::CloseUncompressedMemorySection( u64 Address )
{
    if( m_OpenMemorySection == INVALID_MEMORY_SECTION )
    {
        return RESULT_COREDUMP_MEMORY_SECTION_NOT_OPEN;
    }

    //If we have one open, but it's not this one -
    if( m_OpenMemorySection != Address )
    {
        return RESULT_COREDUMP_MEMORY_SECTION_NOT_OPEN;
    }

    // "Close" this memory section.
    m_OpenMemorySection = INVALID_MEMORY_SECTION;

    return RESULT_COREDUMP_OK;
}

//============================================================================================

//============================================================================================

result coredump_writer::AddTTY( char* pData, u64 SizeOfData )
{
    return AddData( pTTYTag, pData, SizeOfData );
}

//============================================================================================

result coredump_writer::AddStackFrames( u64 ThreadId, u64 NumberOfFrames, u64* pFrames )
{
    coredump_stack_frames_header FrameHdr( ThreadId, NumberOfFrames );
    u64 SizeOfFrames = sizeof(u64) * NumberOfFrames;
    u64 SizeOfData = sizeof( FrameHdr ) + SizeOfFrames;

    coredump_file_data_header Header(pStackFramesTag, SizeOfData);
    result Ret = Write(&Header, sizeof(Header));
    if (Ret == RESULT_COREDUMP_OK)
    {
        Ret = Write( &FrameHdr, sizeof( FrameHdr ) );
        if (Ret == RESULT_COREDUMP_OK)
        {
            Ret = Write( pFrames, SizeOfFrames );
        }
    }

    return Ret;
}

//============================================================================================

result coredump_writer::AddModule( char* pModuleName, char* pModuleId, u64 LoadAddress, u64 Size )
{
    coredump_module_info Module( pModuleName, pModuleId, LoadAddress, Size );
    return AddData( pModuleTypeTag, (void*)&Module, sizeof(coredump_module_info) );
}

//============================================================================================

result coredump_writer::AddThread( coredump_thread_info* pThreadInfo, u64 TPIDR )
{
    u64 SizeOfThreadDataHeader = sizeof(coredump_thread_info);

    // We're not going to write the pointers themselves, so remove their size from this equation.
    SizeOfThreadDataHeader -= sizeof(u64*) * 4;

    // But we ARE going to write the data they point at.
    u64 SizeOfGPRegisters           = pThreadInfo->m_NumberOfGPRegisters * sizeof(u64);
    u64 SizeOfGPControlRegisters    = pThreadInfo->m_NumberOfGPControlRegisters * sizeof(u64);
    u64 SizeOfFPRegisters           = pThreadInfo->m_NumberOfFPRegisters * sizeof(u64);
    u64 SizeOfFPControlRegisters    = pThreadInfo->m_NumberOfFPControlRegisters * sizeof(u64);

    //Here's our size total -
    u64 SizeOfData = SizeOfThreadDataHeader + SizeOfGPRegisters + SizeOfGPControlRegisters + SizeOfFPRegisters + SizeOfFPControlRegisters;

    //Write the data header
    coredump_file_data_header Header( pThreadTypeTag, SizeOfData );

    result Ret = Write( &Header, sizeof(Header) ) ;
    if( Ret == RESULT_COREDUMP_OK )
    {
        //Write the thread data header
        pThreadInfo->m_Name[ sizeof(pThreadInfo->m_Name) - 1] = 0;
        Ret = Write( pThreadInfo, SizeOfThreadDataHeader );
        if( Ret == RESULT_COREDUMP_OK )
        {
            //Now write the registers themselves
            Ret = Write( pThreadInfo->m_pGPRegisters, SizeOfGPRegisters );
            if( Ret == RESULT_COREDUMP_OK )
            {
                Ret = Write( pThreadInfo->m_pGPControlRegisters, SizeOfGPControlRegisters );
                if( Ret == RESULT_COREDUMP_OK )
                {
                    Ret = Write( pThreadInfo->m_pFPRegisters, SizeOfFPRegisters );
                    if( Ret == RESULT_COREDUMP_OK )
                    {
                        Ret = Write( pThreadInfo->m_pFPControlRegisters, SizeOfFPControlRegisters );

                        if( Ret == RESULT_COREDUMP_OK )
                        {
                            coredump_thread_local_storage storage( pThreadInfo->m_ThreadId, TPIDR );
                            Ret = AddData( pThreadLocalStorageTag, &storage, sizeof( storage ) );
                        }
                    }
                }
            }
        }
    }

    return Ret;
}

//============================================================================================

result coredump_writer::AddRegisterDefinitions( void* pData, u64 SizeOfData )
{
    return AddData( pRegisterDefinitionsTypeTag, pData, SizeOfData );
}

//==============================================================================
// Screenshot adding API.
result coredump_writer::CreateScreenshot( ScreenshotType FileType, u64 Size )
{
    if( Size == 0 )
    {
        return RESULT_COREDUMP_INVALID_INPUT;
    }

    if( m_OpenImageSectionSize > 0 )
    {
        return RESULT_COREDUMP_IMAGE_ALREADY_OPEN;
    }

    const char* pTag = pRawImageTag;
    switch( FileType )
    {
        default:
            return RESULT_COREDUMP_IMAGE_FORMAT_UNRECOGNIZED;
            break;

        case ScreenshotType_Raw:
            pTag = pRawImageTag;
            break;

        case ScreenshotType_JPG:
            pTag = pJPGImageTag;
            break;

        case ScreenshotType_PNG:
            pTag = pPNGImageTag;
            break;

        case ScreenshotType_BMP:
            pTag = pBMPImageTag;
            break;
    }

    // Save this size for later.
    m_OpenImageSectionSize = Size;

    coredump_file_data_header Header( pTag, Size );

    //Write the data header
    return Write( &Header, sizeof( Header ) );
}

//============================================================================================

result coredump_writer::AddScreenshotData( void* pData, u64 Size )
{
    if( m_OpenImageSectionSize == 0 )
    {
        return RESULT_COREDUMP_IMAGE_NOT_OPEN;
    }

    result Ret = Write( pData, Size );

    if( Ret == RESULT_COREDUMP_OK )
    {
        m_OpenImageSectionSize -= Size;
    }
    return Ret;
}

//============================================================================================

result coredump_writer::CloseScreenshot()
{
    result Ret = RESULT_COREDUMP_OK;
    if( m_OpenImageSectionSize > 0 )
    {
        //=======================================================
        // Because we wrote our header with this size,
        // insure file integrity by padding with zeroes.
        //=======================================================
        while( m_OpenImageSectionSize > 0 )
        {
            char WriteValue = 0;
            if( Write( &WriteValue, sizeof( WriteValue ) ) != RESULT_COREDUMP_OK )
            {
                break;
            }
            m_OpenImageSectionSize -= 1;
        }
        m_OpenImageSectionSize = 0;
        Ret = RESULT_COREDUMP_IMAGE_DATA_INCOMPLETE;
    }
    return Ret;
}

//==============================================================================
// Video adding API.

result coredump_writer::CreateVideo( VideoType FileType, u64 Size )
{
    if( Size == 0 )
    {
        return RESULT_COREDUMP_INVALID_INPUT;
    }

    if( m_OpenVideoSectionSize > 0 )
    {
        return RESULT_COREDUMP_IMAGE_ALREADY_OPEN;
    }

    const char* pTag = pRawVideoTag;
    switch( FileType )
    {
    default:
        return RESULT_COREDUMP_IMAGE_FORMAT_UNRECOGNIZED;
        break;

    case VideoType_Raw:
        pTag = pRawVideoTag;
        break;

    case VideoType_MP4:
        pTag = pMP4VideoTag;
        break;
    }

    // Save this size for later.
    m_OpenVideoSectionSize = Size;

    coredump_file_data_header Header( pTag, Size );

    //Write the data header
    return Write( &Header, sizeof( Header ) );
}

//==============================================================================

result coredump_writer::AddVideoData( void* pData, u64 Size )
{
    if( m_OpenVideoSectionSize == 0 )
    {
        return RESULT_COREDUMP_IMAGE_NOT_OPEN;
    }

    result Ret = Write( pData, Size );

    if( Ret == RESULT_COREDUMP_OK )
    {
        m_OpenVideoSectionSize -= Size;
    }
    return Ret;
}

//==============================================================================

result coredump_writer::CloseVideo()
{
    result Ret = RESULT_COREDUMP_OK;
    if( m_OpenVideoSectionSize > 0 )
    {
        //=======================================================
        // Because we wrote our header with this size,
        // insure file integrity by padding with zeroes.
        //=======================================================
        while( m_OpenVideoSectionSize > 0 )
        {
            char WriteValue = 0;
            if( Write( &WriteValue, sizeof( WriteValue ) ) != RESULT_COREDUMP_OK )
            {
                break;
            }
            m_OpenVideoSectionSize -= 1;
        }
        m_OpenVideoSectionSize = 0;
        Ret = RESULT_COREDUMP_IMAGE_DATA_INCOMPLETE;
    }
    return Ret;
}

//==============================================================================

result coredump_writer::AddApplicationId( u64 Id )
{
    return AddData( pApplicationIdTag, &Id, sizeof( Id ) );
}

//============================================================================================

result coredump_writer::AddData( const char* pTag, void* pData, u64 SizeOfData )
{
    if( m_pOutput != NULL )
    {
        coredump_file_data_header Header( pTag, SizeOfData );
        if( Write( &Header, sizeof(Header) ) == RESULT_COREDUMP_OK )
        {
            return Write( pData, SizeOfData );
        }
    }

    return RESULT_COREDUMP_UNABLE_TO_OPEN_OUTPUT;
}

//============================================================================================


result coredump_writer::AddFileData( char* pFileName, const char* pTag, char* pCoreDumpFileName )
{
#if WIN32
    FILE* pOutput;
    if( fopen_s( &pOutput, pCoreDumpFileName, "ab" ) != 0 )
#else
    nn::fs::FileHandle Output;
    nn::Result Res = nn::fs::OpenFile( &Output, pCoreDumpFileName, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend );
    if( Res.IsFailure() )
#endif
    {
        return RESULT_COREDUMP_UNABLE_TO_OPEN_OUTPUT;
    }

#if WIN32
    FILE* pInput;
    if( fopen_s( &pInput, pFileName, "rb" ) != 0 )
#else
    nn::fs::FileHandle Input;
    Res = nn::fs::OpenFile( &Input, pFileName, nn::fs::OpenMode_Read );
    if( Res.IsFailure() )
#endif
    {
#if WIN32
        fclose( pOutput );
#else
        nn::fs::CloseFile( Output );
#endif
        return RESULT_COREDUMP_UNABLE_TO_OPEN_INPUT;
    }

    // Calculate our data size
    u64 SizeOfData = 0;

#if WIN32
    fseek( pInput, 0, SEEK_END );
    SizeOfData = ftell( pInput );
    fseek( pInput, 0, SEEK_SET );
    fseek( pOutput, 0, SEEK_END );
#else
    int64_t sizeOfFile = 0;
    nn::fs::GetFileSize( &sizeOfFile, Input );
    SizeOfData = (u64)sizeOfFile;

    // Get our offset
    s64 FileWriteOffset = 0;
    nn::fs::GetFileSize( &FileWriteOffset, Output );
    int64_t FileReadOffset = 0;
#endif

    // Write our header
    coredump_file_data_header Header( pTag, SizeOfData );
#if WIN32
    fwrite( &Header, 1, (size_t)sizeof( Header ), pOutput );
#else
    nn::fs::WriteOption writeOption = { nn::fs::WriteOptionFlag_Flush };
    Res = nn::fs::WriteFile( Output, FileWriteOffset, &Header, (size_t)sizeof( Header ), writeOption );
    FileWriteOffset += sizeof( Header );
#endif

    // Now the file data
    enum
    {
        SIZE_OF_TEMP_COPY_BUFFER = ((64 * 1024) * 1)
    };

    result Ret = RESULT_COREDUMP_OK;
    char* pTempBuffer = new char[SIZE_OF_TEMP_COPY_BUFFER];
    while( SizeOfData > 0 )
    {
        size_t AmountToWrite = SizeOfData > SIZE_OF_TEMP_COPY_BUFFER ? SIZE_OF_TEMP_COPY_BUFFER : SizeOfData;
#if WIN32
        size_t AmountRead = fread( pTempBuffer, 1, AmountToWrite, pInput );
#else
        size_t AmountRead = 0;
        Res = nn::fs::ReadFile( Input, FileReadOffset, pTempBuffer, AmountToWrite );
        if( Res.IsSuccess() )
        {
            FileReadOffset += AmountToWrite;
            AmountRead = AmountToWrite;
        }
#endif
        if( AmountRead > 0 )
        {
#if WIN32
            size_t AmountWritten = fwrite( pTempBuffer, 1, AmountRead, pOutput );
#else
            size_t AmountWritten = 0;
            nn::fs::WriteOption writeOption = { nn::fs::WriteOptionFlag_Flush };
            Res = nn::fs::WriteFile( Output, FileWriteOffset, pTempBuffer, AmountRead, writeOption );
            if( Res.IsSuccess() )
            {
                AmountWritten = AmountRead;
                FileWriteOffset += AmountRead;
            }
            else
            {
                Ret = RESULT_COREDUMP_IO_ERROR;
                break;
            }
#endif
            SizeOfData -= AmountWritten;
        }
        else
        {
            Ret = RESULT_COREDUMP_IO_ERROR;
            break;
        }
    }
    delete[] pTempBuffer;
#if WIN32
    fclose( pInput );
    fclose( pOutput );
#else
    nn::fs::CloseFile( Input );
    nn::fs::CloseFile( Output );
#endif
    return Ret;
}

//==============================================================================

result coredump_writer::AddBMPImage( char* pImageFileName, char* pCoreDumpFileName )
{
    return AddFileData( pImageFileName, pBMPImageTag, pCoreDumpFileName );
}

//==============================================================================

result coredump_writer::AddMP4Video( char* pVideoFileName, char* pCoreDumpFileName )
{
    return AddFileData( pVideoFileName, pMP4VideoTag, pCoreDumpFileName );
}

//==============================================================================
} //namespace coredump
//==============================================================================
