﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <stdint.h>
#include <cstdio>
#include <cstring>
#include <cctype>

//==============================================================================
namespace coredump {
//==============================================================================

typedef          uint16_t   u16;        // unsigned 16-bit integer
typedef          int16_t    s16;        //   signed 16-bit integer
typedef          uint32_t   u32;        // unsigned 32-bit integer
typedef          int32_t    s32;        //   signed 32-bit integer
typedef          uint64_t   u64;        // unsigned 64-bit integer
typedef          int64_t    s64;        //   signed 64-bit integer

//==============================================================================
// Memory types
enum MemoryType
{
    MemoryType_None              = 0,
    MemoryType_Read              = (1 <<  2),   // Read.
    MemoryType_Write             = (1 <<  1),   // Write.
    MemoryType_Execute           = (1 <<  0),   // Execute.
    MemoryType_ReadWrite         = ( MemoryType_Read | MemoryType_Write ),
    MemoryType_ReadExecute       = ( MemoryType_Read | MemoryType_Execute ),
    MemoryType_All               = ( MemoryType_Read | MemoryType_Write | MemoryType_Execute ),
};

//==============================================================================
// Screenshot types
enum ScreenshotType
{
    ScreenshotType_Raw = 0,
    ScreenshotType_BMP,
    ScreenshotType_JPG,
    ScreenshotType_PNG,
};

//==============================================================================
// Video types
enum VideoType
{
    VideoType_Raw = 0,
    VideoType_MP4,
};

//==============================================================================
// Result types
enum result : int32_t
{
    RESULT_COREDUMP_OK              = 0,
    RESULT_COREDUMP_NOT_FOUND,
    RESULT_COREDUMP_NOT_VALID,
    RESULT_COREDUMP_OUT_OF_MEMORY,
    RESULT_COREDUMP_UNABLE_TO_OPEN_OUTPUT,
    RESULT_COREDUMP_MEMORY_SECTION_ALREADY_OPEN,
    RESULT_COREDUMP_MEMORY_SECTION_NOT_OPEN,
    RESULT_COREDUMP_MEMORY_SECTION_INVALID,
    RESULT_COREDUMP_INTERNAL_ERROR,
    RESULT_COREDUMP_UNRECOGNIZED_VERSION,
    RESULT_COREDUMP_IO_ERROR,
    RESULT_COREDUMP_UNABLE_TO_OPEN_INPUT,
    RESULT_COREDUMP_IMAGE_FORMAT_UNRECOGNIZED,
    RESULT_COREDUMP_IMAGE_ALREADY_OPEN,
    RESULT_COREDUMP_IMAGE_NOT_OPEN,
    RESULT_COREDUMP_IMAGE_DATA_INCOMPLETE,
    RESULT_COREDUMP_INVALID_INPUT,
    RESULT_COREDUMP_UNABLE_TO_DEBUG,
};

//========================================================================================================
// Tag types
extern const char* pHeaderTypeTag;
extern const char* pRegisterDefinitionsTypeTag;
extern const char* pModuleTypeTag;
extern const char* pThreadTypeTag;
extern const char* pMemoryTypeTag;
extern const char* pCompressedMemoryTypeTag;
extern const char* pZLibCompressedMemoryTypeTag;
extern const char* pTTYTag;
extern const char* pVersionTag;
extern const char* pApplicationIdTag;
extern const char* pStackFramesTag;
extern const char* pBMPImageTag;
extern const char* pRawImageTag;
extern const char* pJPGImageTag;
extern const char* pPNGImageTag;
extern const char* pMP4VideoTag;
extern const char* pRawVideoTag;
extern const char* pThreadLocalStorageTag;

#ifdef WIN32
#define COREDUMP_STRNCPY( Dest, Source, Count ) strncpy_s( Dest, Source, Count )
#else
#define COREDUMP_STRNCPY( Dest, Source, Count ) std::strncpy( Dest, Source, Count )
#endif


//========================================================================================================
// File struct types

//========================================================================================================
// Base type:  8-byte tag followed by a 64-bit size value
struct coredump_file_data_header
{
    coredump_file_data_header()
    {
        memset( this, 0, sizeof(coredump_file_data_header) );
    }

    explicit coredump_file_data_header( const char* pTag )
    {
        memset( this, 0, sizeof( coredump_file_data_header ) );
        size_t TagLength = strlen( pTag );
        size_t WriteLength = TagLength > sizeof( m_Tag ) ? sizeof( m_Tag ) : TagLength;
        memcpy( m_Tag, pTag, WriteLength );
        m_Size = 0;
    }

    coredump_file_data_header( const char* pTag, u64 Size )
    {
        memset( this, 0, sizeof( coredump_file_data_header ) );
        size_t TagLength = strlen( pTag );
        size_t WriteLength = TagLength > sizeof( m_Tag ) ? sizeof( m_Tag ) : TagLength;
        memcpy( m_Tag, pTag, WriteLength );
        m_Size = Size;
    }

    bool IsTag( const char* pTag )
    {
        size_t Length = strlen(pTag);
        for(size_t Index = 0; Index < Length; Index += 1 )
        {
            if( pTag[Index] != m_Tag[Index] )
            {
                return false;
            }
        }
        return true;
    }

    char m_Tag[8];
    u64  m_Size;
};

//========================================================================================================
// File header
struct coredump_file_header
{
    coredump_file_header()
    {
        memset( this, 0, sizeof(coredump_file_header) );
    }

    bool IsArchitecture( char* pType )
    {
        return ( strcmp( pType, m_Architecture ) == 0 );
    }

    u64     m_ProcessId;
    char    m_Architecture[16];
    u64     m_ExceptionNumber;
    char    m_ProcessName[1024];
    char    m_Args[1024];
    char    m_OSVersion[64];
    u64     m_LoadAddr;
    u64     m_Size;
};

//========================================================================================================
// Version chunk, as per Siglo-65108

enum Version : int32_t
{
    Version_Major   = 1,
    Version_Minor   = 0,
    Version_Build   = 0
};

struct coredump_file_version
{
    coredump_file_version()
    {
        memset( this, 0, sizeof(coredump_file_version) );
        m_Major = Version_Major;
        m_Minor = Version_Minor;
        m_Build = Version_Build;
    }
    u32     m_Major;
    u32     m_Minor;
    u32     m_Build;
};

//========================================================================================================

struct coredump_module_info
{
    coredump_module_info( char* pModuleName, char* pModuleId, u64 LoadAddress, u64 Size )
    {
        // Don't save the module name if it's invalid
        if( isModuleName( pModuleName ))
        {
            memcpy( m_ModuleName, pModuleName, sizeof(m_ModuleName) );
        }
        else
        {
#if WIN32
            strcpy_s( m_ModuleName, "ID=");
#else
            strcpy( m_ModuleName, "ID=");
#endif
            char* pWriteId = m_ModuleName + 3;
            memcpy( pWriteId, pModuleId, sizeof(m_ModuleId) );
        }
        memcpy( m_ModuleId, pModuleId, sizeof(m_ModuleId) );
        m_LoadAddress   = LoadAddress;
        m_Size          = Size;
    }

    coredump_module_info()
    {
        memset( this, 0, sizeof(coredump_module_info) );
    }

    bool isModuleName( char* pModuleName )
    {
        bool Ret = false;
        if( pModuleName != NULL )
        {
            // Insure it's a valid name
            int32_t Length = (int32_t)strlen( pModuleName );
            Ret = Length > 0;
            if( Ret )
            {
                //=======================================================================
                // Check for the false-positive case of '', which is what is embedded
                // when the user selects "No" for Embed Assembly Name
                //=======================================================================
                if( pModuleName[0] == '\'' && pModuleName[1] == '\'' )
                {
                    Ret = false;
                }
            }
        }
        return Ret;
    }
    char    m_ModuleName[1024]; // File name and path.  Or at least name.
    char    m_ModuleId[32];     // Module ID.  Used for cases where we don't have the module path.
    u64     m_LoadAddress;      // Where in memory it's loaded.
    u64     m_Size;             // Size in memory
};

//========================================================================================================
//
struct coredump_thread_info
{
    coredump_thread_info()
    {
        memset( this, 0, sizeof(coredump_thread_info) );
    }

    //===================================================================
    // NOTICE the minor waste of space here (ie., an s16 for a boolean).
    // Done this way to defeat padding problems created by the compiler
    //===================================================================
    u64     m_ThreadId;
    s16     m_CurrentThread;    // 1 or 0.
    s16     m_Status;           // "R" (running) | "S" (stopped) | "W" (waiting) | "M" (moribund.  Dead or dying.)
    s16     m_Priority;         // Priority of this thread.
    s16     m_Core;             // The lower 8 bits are the core, the upper 8 bits are the affinity mask
    u64     m_IP;
    u64     m_SP;
    char    m_Name[32];
    u64     m_NumberOfGPRegisters;
    u64     m_NumberOfGPControlRegisters;
    u64     m_NumberOfFPRegisters;
    u64     m_NumberOfFPControlRegisters;
    u64*    m_pGPRegisters;
    u64*    m_pGPControlRegisters;
    u64*    m_pFPRegisters;
    u64*    m_pFPControlRegisters;
};

//========================================================================================================
// Memory
struct coredump_memory_header
{
    coredump_memory_header()
    {
        m_Address       = 0;
        m_MemoryType    = 0;
        m_Size          = 0;
    }

    coredump_memory_header( u64 Address, u64 MemoryType, u64 Size )
    {
        m_Address       = Address;
        m_MemoryType    = MemoryType;
        m_Size          = Size;
    }

    //==============================================================================

    bool Contains( u64 Address )
    {
        return ( (Address >= m_Address) && ( Address < (m_Size + m_Address) ) );
    }

    //==============================================================================

    u32 Read( char* pData, u64 Address, void* pBuffer, u32 nBytes )
    {
        //Calculate the proper offset:
        u64 Offset = Address - m_Address;

        //Validate we can actually read this much.
        u32 ReadSize = (u32)nBytes;
        if( (m_Size - Offset) < ReadSize )
        {
            ReadSize = (u32)(m_Size - Offset);
        }
        char* pReadFrom = (char*)((u64)pData + Offset);

        memcpy( pBuffer, pReadFrom, ReadSize );
        return ReadSize;
    }

    u64     m_Address;
    u64     m_MemoryType;
    u64     m_Size;
};

//========================================================================================================
// Stack frame

struct coredump_stack_frames_header
{
    coredump_stack_frames_header()
    {
        m_ThreadId = 0;
        m_NumberOfFrames = 0;
    }

    coredump_stack_frames_header( u64 ThreadId, u64 NumberOfFrames )
    {
        m_ThreadId = ThreadId;
        m_NumberOfFrames = NumberOfFrames;
    }

    //==============================================================================

    u64     m_ThreadId;
    u64     m_NumberOfFrames;
};

//========================================================================================================
// Thread local storage

struct coredump_thread_local_storage
{
    coredump_thread_local_storage()
    {
        m_ThreadId = 0;
        m_TPIDR = 0;
    }

    coredump_thread_local_storage( u64 ThreadId, u64 TPIDR )
    {
        m_ThreadId = ThreadId;
        m_TPIDR = TPIDR;
    }

    //==============================================================================

    u64     m_ThreadId;
    u64     m_TPIDR;
};

//==============================================================================
} //namespace coredump
//==============================================================================

