﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SnapShotDumper.h"
#include "SnapShotDumper_System.h"
#include "SnapShotDumper_Output.h"
#include <cctype>
#include <nn.h>
#include <nn/os.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/fs.h>
#include <nn/svc/svc_Types.h>
#include <nn/init.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/factory/settings_SerialNumber.h>
#include <nn/time.h>
#include <nn/util/util_StringUtil.h>
#include <nn/htc.h>

//==============================================================================
// Various settings for this instance
namespace settings
{
    enum
    {
        OUTPUT_FILE_NAME_SIZE = 0x1000,
    };

    static bool         IsQuickDump = false;
    static bool         OutputLog = false;
    static bool         OutputDump = true;
    static char         OutputFileName[OUTPUT_FILE_NAME_SIZE];
    static nn::Bit64    ProcessId = 0;
}

//============================================================================================

extern "C" void nninitStartup()
{
}

//============================================================================================

void ReportUsage()
{
    NN_LOG( "usage:  SnapShotdumper <processId> [<output file name>] [-quick] [-log 0|1] [-dump -1|0|1]\n" );
}

//============================================================================================

bool StringsEqual( const char* pStr1, const char* pStr2 )
{
    int Length = strlen( pStr1 );
    if ( strlen( pStr2 ) != Length )
    {
        return false;
    }

    for( int Index = 0; Index < Length; Index += 1 )
    {
        if( tolower(pStr1[Index]) !=  (unsigned int) tolower(pStr2[Index]) )
        {
            return false;
        }
    }

    return true;
}

//============================================================================================

static bool FormatOutputFileName( char* OutputFileName )
{
    bool Ret = false;
    if( OutputFileName != NULL && strlen( OutputFileName ) > 0 )
    {
        //Does it have a valid extension?
        char* pExtension = strrchr ( OutputFileName, '.' );
        if( pExtension != NULL && StringsEqual( pExtension, SNAP_SHOT_DUMPER_EXTENSION ) )
        {
            *pExtension = '\0'; //Remove extension once
        }
        Ret = true;
    }

    return Ret;
}

//============================================================================================

// the value of src can be destroyed.
size_t ParseEnvironmentVariable(char *buffer, size_t bufferSize, char *src)
{
    nn::htc::Initialize();
    int size = 0;
    while (*src != '\0' && bufferSize > size + 1)
    {
        if (*src == '%')
        {
            src++;
            const char *envName = src;
            while (*src != '\0' && *src != '%')
                src++;
            *src = '\0';
            src++;

            size_t envSize;
            nn::Result result = nn::htc::GetEnvironmentVariable(&envSize, buffer + size, bufferSize - size, envName);
            if (result.IsFailure() || envSize == 0)
            {
                NN_LOG("[Dump Error] Failed to read environment variable %s.\n", envName);
            }
            else
            {
                size += envSize - 1;
            }
        }
        else
        {
            *(buffer + size) = *src;
            src++;
            size++;
        }
    }
    buffer[size] = '\0';
    size++;
    nn::htc::Finalize();
    return size;
}

//============================================================================================

bool CreateDefaultFileName(char *buffer, size_t bufferSize)
{
    const int SettingsValueSize = 256;
    char settingsValue[SettingsValueSize];
    nn::settings::fwdbg::GetSettingsItemValue(settingsValue, SettingsValueSize, "snap_shot_dump", "output_dir");
    size_t size = ParseEnvironmentVariable(buffer, bufferSize, settingsValue);

    const int FileNameLength = 1 + 14 + 1 + 14; // /SN_YYYYMMDDhhmmss
    NN_SDK_ASSERT(size + FileNameLength <= bufferSize, "Too long path\n");
    if (size <= 0 || size + FileNameLength > bufferSize)
    {
        return false;
    }

    nn::time::PosixTime timeStamp;
    nn::time::StandardUserSystemClock::GetCurrentTime(&timeStamp);
    nn::time::CalendarTime time;
    nn::time::ToCalendarTime(&time, nullptr, timeStamp);

    nn::settings::factory::SerialNumber serialNumber;
    serialNumber.string[0] = '\0';

    const nn::Result result = nn::settings::factory::GetSerialNumber(&serialNumber);
    if (result.IsFailure())
    {
        NN_LOG("[SnapShotDumper] Failed to get Serial Number of the device.\n");
    }

    nn::util::TSNPrintf(buffer + size - 1, bufferSize - size + 1, "\\%s_%04d%02d%02d%02d%02d%02d",
        serialNumber.string,
        time.year, time.month, time.day, time.hour, time.minute, time.second);
    return true;
}

//==============================================================================

struct RecognizedArg
{
    const char* m_pName;
    int         m_NumArguments;
    void( *m_Parse )(char*);

    //==============================================================================

    bool Is( char* pFromCommandLine )
    {
        char* pTestArg = FindArg( pFromCommandLine );
        return StringsEqual( pTestArg, m_pName );
    }

    //==============================================================================

    static char* FindArg( char* pFromCommandLine )
    {
        if( *pFromCommandLine == '-' || *pFromCommandLine == '\\' )
        {
            return pFromCommandLine + 1;
        }
        return pFromCommandLine;
    }

    //==============================================================================

    static void handleArg_Quick( char* pParam )
    {
        SNAP_SHOT_DUMPER_LOG( "Found quick flag\n" );
        settings::IsQuickDump = true;
    }

    //==============================================================================

    static void handleArg_Log( char* pParam )
    {
        //==============================================================================
        // If SSD receives an argument "-log number", it should output log accordingly.
        //==============================================================================

        SNAP_SHOT_DUMPER_LOG( "Found log flag:  %s\n", pParam );

        int Param = std::atoi( pParam );

        //=============================================================================================================
        // If the number is 0 (or no - log option), no log is to be output.  This is the same as the current behavior.
        // If the number is 1, it should output log about all the threads.
        //=============================================================================================================
        if( Param == 1 )
        {
            settings::OutputLog = true;
        }
    }

    //==============================================================================

    static void handleArg_Dump( char* pParam )
    {
        //==============================================================================
        // If SSD receives "-dump number", it should switch the granularity of nxdmp.
        //==============================================================================
        SNAP_SHOT_DUMPER_LOG( "Found dump flag:  %s\n", pParam );
        int Param = std::atoi( pParam );

        //==============================================================================
        // If -1, no nxdmp.
        // If 0, quick dump.
        // If 1, full dump.
        // 0 and 1 come from the definitions of DumpType_* enum in tm.h.
        //==============================================================================

        switch( Param )
        {
        case -1: // If -1, no nxdmp.
            settings::OutputDump = false;
            break;

        case 0: // If 0, quick dump.
            settings::IsQuickDump = true;
            break;

        case 1:     // If 1, full dump.
        default:    // Also our default
            break;
        }
    }

    //==============================================================================
};

//==============================================================================

struct RecognizedArg s_AllRecognizedArgs[] =
{
    { "quick",  0, RecognizedArg::handleArg_Quick },
    { "log",    1, RecognizedArg::handleArg_Log },
    { "dump",   1, RecognizedArg::handleArg_Dump },
};

//============================================================================================

enum
{
    NUMBER_OF_RECOGNIZED_ARGS = (sizeof( s_AllRecognizedArgs ) / sizeof( RecognizedArg ))
};

//==========================================================================================================================================

bool GetCommandLineParams( )
{
    int NumArgs = nn::os::GetHostArgc();
    char** pArgs = nn::os::GetHostArgv();
    for(int ArgIndex = 0; ArgIndex < NumArgs; ArgIndex += 1 )
    {
        SNAP_SHOT_DUMPER_TRACE( "SnapSnotDumper", "GetCommandLineParams:  arg %d:  %s", ArgIndex, pArgs[ArgIndex] );
    }

    if ( NumArgs < 2 )
    {
        return false;
    }

    //============================================================
    // Get our required inputs.
    //============================================================
    char* pProcessIdString = pArgs[1];
    settings::ProcessId = static_cast<nn::Bit64>( std::atoi( pProcessIdString ) );

    int ArgIndex = 2;

    if (NumArgs > 2 && pArgs[2][0] != '-')
    {
        strcpy( settings::OutputFileName, pArgs[2] );
        ArgIndex++;
    }
    else
    {
        if (!CreateDefaultFileName( settings::OutputFileName, sizeof( settings::OutputFileName ) ))
        {
            return false;
        }
    }

    // See if we have any more arguments
    for( ; ArgIndex < NumArgs; ArgIndex++ )
    {
        for( int Index = 0; Index < NUMBER_OF_RECOGNIZED_ARGS; Index += 1 )
        {
            RecognizedArg* pRecognizedArg = &s_AllRecognizedArgs[Index];
            if( pRecognizedArg->Is( pArgs[ArgIndex] ) )
            {
                char* pParam = NULL;
                if( pRecognizedArg->m_NumArguments )
                {
                    ArgIndex += 1;
                    pParam = pArgs[ArgIndex];
                }
                pRecognizedArg->m_Parse( pParam );
            }
        }
    }

    return FormatOutputFileName( settings::OutputFileName );
}

//============================================================================================

extern "C" void nnMain()
{
    nn::fs::InitializeWithMultiSessionForTargetTool();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());
    if( GetCommandLineParams() == false )
    {
        ReportUsage();
    }
    else
    {
        SNAP_SHOT_DUMPER_LOG( "Dumping process %lld, output = %s", ProcessId, sOutputFileName );

        //============================================================================================

        nn::Result SysResult = SnapShotDumper::System::Init();
        if ( SysResult.IsSuccess() )
        {
            if( settings::OutputDump == true )
            {
                nn::Result FsResult = SnapShotDumper::System::FsInit( settings::OutputFileName, sizeof( settings::OutputFileName ) );
                if( FsResult.IsFailure() )
                {
                    NN_LOG( "[SnapShotDumper] Failed to initialize FS (%08X). Is SD card inserted?\n", FsResult.GetInnerValueForDebug() );
                    SNAP_SHOT_DUMPER_LOG( "Unable to init system:  %lld", FsResult.GetInnerValueForDebug() );
                    return;
                }
            }
            SNAP_SHOT_DUMPER_TRACE( "SnapSnotDumper", "System inited.  Attaching to %lld", ProcessId );
            nn::svc::Handle ProcessHandle;
            nn::Result Result =  nn::dbg::DebugActiveProcess( &ProcessHandle, settings::ProcessId );
            if( Result.IsSuccess() )
            {
                SnapShotDumper::Output* pOutput = new SnapShotDumper::Output( settings::ProcessId, ProcessHandle, settings::IsQuickDump );
                NN_ABORT_UNLESS_RESULT_SUCCESS( SnapShotDumper::System::CompileProcessDetails( pOutput ) );
                if( settings::OutputDump == true )
                {
                    pOutput->Write( settings::OutputFileName );
                }

                if( settings::OutputLog == true )
                {
                    pOutput->Log();
                }

                delete pOutput;
                nn::dbg::CloseHandle( ProcessHandle );
            }
            else
            {
                SNAP_SHOT_DUMPER_LOG( "Unable to attach to %lld:  0x%08llX", ProcessId, Result.GetInnerValueForDebug() );
            }

            SNAP_SHOT_DUMPER_LOG( "Completed" );
            if( settings::OutputDump == true )
            {
                NN_LOG("[SnapShotDumper] Dump completed\n");
                SnapShotDumper::System::Close();
            }
        }
        else
        {
            SNAP_SHOT_DUMPER_LOG( "Unable to init system:  %lld", SysResult.GetInnerValueForDebug()  );
        }
    }

    //============================================================================================
}

//============================================================================================
