﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <string>
#include <utility>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/settings/system/settings_Usb.h>

#include "SettingsManager_NameScope.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Usb.h"
#include "SettingsManager_Utility.h"

namespace {

//!< USB 設定のキー
const char* const SettingNameUsbSettings = "usb_settings";

//!< 現在値のキー
const char* const KeyCurrent = "current";

//!< 初期値のキー
const char* const KeyDefault = "default";

//!< USB 3.0 が有効か否かを表す値のキー
const char* const KeyUsb30EnableFlag = "usb30_enable_flag";

//!< USB 3.0 が有効か否かを表す値をエクスポートします。
bool ExportUsb30EnableFlag(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyUsb30EnableFlag);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                ::nn::settings::system::IsUsb30Enabled()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(false));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(pDictNode->AppendMember(KeyUsb30EnableFlag, ::std::move(node)));

    return true;
}

//!< USB 3.0 が有効か否かを表す値をインポートします。
bool ImportUsb30EnableFlag(const Node& dictNode) NN_NOEXCEPT
{
    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyUsb30EnableFlag));

    NameScope nameScope(KeyUsb30EnableFlag);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto usb30EnableFlag = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&usb30EnableFlag));

    ::nn::settings::system::SetUsb30Enabled(usb30EnableFlag);

    return true;
}

} // namespace

bool IsSettingNameUsbSettings(const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNameUsbSettings);
}

bool ExportUsbSettings(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(SettingNameUsbSettings);

        COMMAND_DO(ExportUsb30EnableFlag(&node));
    }

    COMMAND_DO(pNode->AppendMember(SettingNameUsbSettings, ::std::move(node)));

    return true;
}

bool ImportUsbSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNameUsbSettings);

    COMMAND_DO(ImportUsb30EnableFlag(node));

    return true;
}
