﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <string>
#include <utility>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/settings/system/settings_Pcv.h>

#include "SettingsManager_NameScope.h"
#include "SettingsManager_Pcv.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Utility.h"

namespace {

//!< PCV 設定のキー
const char* const SettingNamePcvSettings = "pcv_settings";

//!< 現在値のキー
const char* const KeyCurrent = "current";

//!< 初期値のキー
const char* const KeyDefault = "default";

//!< 外部 RTC のリセットが発生したか否かを表す値のキー
const char* const KeyExternalRtcResetFlag = "external_rtc_reset_flag";

//!< 外部 RTC のリセットが発生したか否かを表す値をエクスポートします。
bool ExportExternalRtcResetFlag(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyExternalRtcResetFlag);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                ::nn::settings::system::IsExternalRtcReset()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(false));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyExternalRtcResetFlag, ::std::move(node)));

    return true;
}

//!< 外部 RTC のリセットが発生したか否かを表す値をインポートします。
bool ImportExternalRtcResetFlag(const Node& dictNode) NN_NOEXCEPT
{
    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyExternalRtcResetFlag));

    NameScope nameScope(KeyExternalRtcResetFlag);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto rtcResetFlag = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&rtcResetFlag));

    ::nn::settings::system::SetExternalRtcReset(rtcResetFlag);

    return true;
}

} // namespace

bool IsSettingNamePcvSettings(const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNamePcvSettings);
}

bool ExportPcvSettings(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(SettingNamePcvSettings);

        COMMAND_DO(ExportExternalRtcResetFlag(&node));
    }

    COMMAND_DO(pNode->AppendMember(SettingNamePcvSettings, ::std::move(node)));

    return true;
}

bool ImportPcvSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNamePcvSettings);

    COMMAND_DO(ImportExternalRtcResetFlag(node));

    return true;
}
