﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <string>
#include <utility>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/settings/system/settings_Nfc.h>

#include "SettingsManager_NameScope.h"
#include "SettingsManager_Nfc.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Utility.h"

namespace {

//!< NFC 設定のキー
const char* const SettingNameNfcSettings = "nfc_settings";

//!< 現在値のキー
const char* const KeyCurrent = "current";

//!< 初期値のキー
const char* const KeyDefault = "default";

//!< NFC が有効か否かを表す値のキー
const char* const KeyNfcEnableFlag = "nfc_enable_flag";

//!< NFC が有効か否かを表す値をエクスポートします。
bool ExportNfcEnableFlag(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyNfcEnableFlag);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(::nn::settings::system::IsNfcEnabled()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(true));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(pDictNode->AppendMember(KeyNfcEnableFlag, ::std::move(node)));

    return true;
}

//!< NFC が有効か否かを表す値をインポートします。
bool ImportNfcEnableFlag(const Node& dictNode) NN_NOEXCEPT
{
    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyNfcEnableFlag));

    NameScope nameScope(KeyNfcEnableFlag);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto nfcEnableFlag = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&nfcEnableFlag));

    ::nn::settings::system::SetNfcEnabled(nfcEnableFlag);

    return true;
}

} // namespace

bool IsSettingNameNfcSettings(const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNameNfcSettings);
}

bool ExportNfcSettings(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(SettingNameNfcSettings);

        COMMAND_DO(ExportNfcEnableFlag(&node));
    }

    COMMAND_DO(pNode->AppendMember(SettingNameNfcSettings, ::std::move(node)));

    return true;
}

bool ImportNfcSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNameNfcSettings);

    COMMAND_DO(ImportNfcEnableFlag(node));

    return true;
}
