﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <memory>
#include <string>
#include <type_traits>
#include <vector>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/settings/system/settings_Network.h>
#include <nn/util/util_StringUtil.h>

#include "SettingsManager_ErrorCode.h"
#include "SettingsManager_NameScope.h"
#include "SettingsManager_Network.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Utility.h"

namespace {

//!< 設定情報ヘッダのキー
const char* const SettingNameNetworkSettings = "network_settings";

//!< ネットワーク設定の名前のキー
const char* const KeyNetworkSettingsName = "name";

//!< ネットワーク設定の ID のキー
const char* const KeyNetworkSettingsId = "id";

//!< ネットワーク設定を自動接続に利用するか否かを表すフラグのキー
const char* const KeyNetworkSettingsAutoConnection = "auto_connection";

//!< ネットワーク設定が大容量通信を許可するか否かを表すフラグのキー
const char* const KeyNetworkSettingsLargeCapacity = "large_capacity";

//!< ネットワーク設定の NIC 種別のキー
const char* const KeyNetworkSettingsNicType = "nic_type";

//!< ネットワーク設定の SSID 設定のキー
const char* const KeyNetworkSettingsSsidSettings = "ssid_config";

//!< ネットワーク設定のセキュリティ設定のキー
const char* const KeyNetworkSettingsSecuritySettings = "security";

//!< ネットワーク設定の IP 設定のキー
const char* const KeyNetworkSettingsIpSettings = "ip_settings";

//!< ネットワーク設定の Proxy 設定のキー
const char* const KeyNetworkSettingsProxySettings = "proxy";

//!< ネットワーク設定の MTU のキー
const char* const KeyNetworkSettingsMtu = "mtu";

//!< SSID 設定の SSID のキー (16 進数表現)
const char* const KeySsidSettingsHex = "hex";

//!< SSID 設定の SSID のキー (文字列表現)
const char* const KeySsidSettingsName = "name";

//!< SSID 設定がステルス状態か否かを表すフラグのキー
const char* const KeySsidSettingsNonBroadcast = "non_broadcast";

//!< セキュリティ設定の認証方式のキー
const char* const KeySecuritySettingsAuthentication = "authentication";

//!< セキュリティ設定の暗号化方式のキー
const char* const KeySecuritySettingsEncryption = "encryption";

//!< セキュリティ設定の Key Material のキー
const char* const KeySecuritySettingsKeyMaterial = "key_material";

//!< IP 設定の IP アドレスの自動取得の有効化フラグのキー
const char* const KeyIpSettingsAutoIp = "auto_ip";

//!< IP 設定の DNS サーバアドレスの自動取得の有効化フラグのキー
const char* const KeyIpSettingsAutoDns = "auto_dns";

//!< IP 設定の IP アドレスのキー
const char* const KeyIpSettingsAddress = "address";

//!< IP 設定のサブネットマスクのキー
const char* const KeyIpSettingsSubnetMask = "subnet_mask";

//!< IP 設定のデフォルトゲートウェイのキー
const char* const KeyIpSettingsDefaultGateway = "default_gateway";

//!< IP 設定の優先 DNS サーバのアドレスのキー
const char* const KeyIpSettingsPreferredDns = "preferred_dns";

//!< IP 設定の代替 DNS サーバのアドレスのキー
const char* const KeyIpSettingsAlternateDns = "alternate_dns";

//!< Proxy 設定の有効化フラグのキー
const char* const KeyProxySettingsEnables = "enables";

//!< Proxy 設定のユーザ認証の有効化フラグのキー
const char* const KeyProxySettingsAuthenticates = "authenticates";

//!< Proxy 設定のホスト名のキー
const char* const KeyProxySettingsHostName = "hostname";

//!< Proxy 設定のポート番号のキー
const char* const KeyProxySettingsPort = "port";

//!< Proxy 設定のユーザ認証のユーザ名のキー
const char* const KeyProxySettingsUserName = "username";

//!< Proxy 設定のユーザ認証のパスワードのキー
const char* const KeyProxySettingsPassword = "password";

//!< オープン認証を表す認証方式
const char* const AuthenticationModeOpen = "Open";

//!< 共有キー認証を表す認証方式
const char* const AuthenticationModeShared = "Shared";

//!< WPA-PSK 認証を表す認証方式
const char* const AuthenticationModeWpaPsk = "WpaPsk";

//!< WPA2-PSK 認証を表す認証方式
const char* const AuthenticationModeWpa2Psk = "Wpa2Psk";

//!< 暗号化無しを表す暗号化方式
const char* const EncryptionModeNone = "None";

//!< WEP 暗号化方式を表す暗号化方式
const char* const EncryptionModeWep = "Wep";

//!< AES 暗号化方式を表す暗号化方式
const char* const EncryptionModeAes = "Aes";

//!< IEEE 802.11 を表す NIC 種別
const char* const NicTypeIeee80211 = "Ieee80211";

//!< Ethernet を表す NIC 種別
const char* const NicTypeEthernet = "Ethernet";

//!< IP アドレスをパースします。
template<typename T>
bool ParseIpAddress(T (&outBuffer)[4], const ::std::string& address) NN_NOEXCEPT
{
    ::std::vector<::std::string> numbers = Split(address, ".");

    if (numbers.size() != 4)
    {
        return false;
    }

    for (const ::std::string& number : numbers)
    {
        if (number.length() == 0)
        {
            return false;
        }
    }

    T buffer[4] = {};

    for (int i = 0; i < 4; ++i)
    {
        for (const char& value : numbers[i])
        {
            if (value < '0' || '9' < value)
            {
                return false;
            }
            else
            {
                buffer[i] = buffer[i] * 10 + (value - '0');
            }
        }
    }

    for (int i = 0; i < 4; ++i)
    {
        outBuffer[i] = buffer[i];
    }

    return true;
}

//!< IP アドレスの文字列表現を返します。
::std::string GetStringOfAddress(const ::nn::Bit8 (&address)[4]) NN_NOEXCEPT
{
    int values[4] = {};

    for (int i = 0; i < 4; ++i)
    {
        values[i] = static_cast<int>(address[i]);
    }

    return Join(".", ::std::begin(values), ::std::end(values));
}

//!< ネットワーク設定の名前をエクスポートします。
bool ExportNetworkSettingsName(
    Node* pDictNode,
    const ::nn::settings::system::NetworkSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateStringNode();

    {
        NameScope nameScope(KeyNetworkSettingsName);

        COMMAND_DO(node.SetValue(settings.name));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyNetworkSettingsName, ::std::move(node)));

    return true;
}

//!< ネットワーク設定の名前をインポートします。
bool ImportNetworkSettingsName(
    ::nn::settings::system::NetworkSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyNetworkSettingsName));

    NameScope nameScope(KeyNetworkSettingsName);

    ::std::string name;
    COMMAND_DO(pNode->GetValue(&name));

    const int length = ::nn::settings::system::NetworkSettingsNameLengthMax;

    if (::nn::util::Strlcpy(pSettings->name, name.c_str(), length) >= length)
    {
        PrintErrorCode(ErrorCode::NodeValueTooLong, nameScope.Get(), name);

        return false;
    }

    return true;
}

//!< ネットワーク設定の ID をエクスポートします。
bool ExportNetworkSettingsId(
    Node* pDictNode,
    const ::nn::settings::system::NetworkSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    ::nn::util::Uuid uuid = {};

    const size_t length = ::std::extent<decltype(settings.id)>::value;

    ::std::copy(&settings.id[0], &settings.id[length], uuid.data);

    ::std::string value;

    EncodeUuid(&value, uuid);

    auto node = Node::CreateStringNode();

    {
        NameScope nameScope(KeyNetworkSettingsId);

        COMMAND_DO(node.SetValue(value));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyNetworkSettingsId, ::std::move(node)));

    return true;
}

//!< ネットワーク設定の ID をインポートします。
bool ImportNetworkSettingsId(
    ::nn::settings::system::NetworkSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyNetworkSettingsId));

    NameScope nameScope(KeyNetworkSettingsId);

    ::std::string value;
    COMMAND_DO(pNode->GetValue(&value));

    ::nn::util::Uuid uuid = {};

    if (!DecodeUuid(&uuid, value))
    {
        PrintErrorCode(ErrorCode::NodeValueInvalid, nameScope.Get(), value);

        return false;
    }

    const size_t length = ::std::extent<decltype(uuid.data)>::value;

    ::std::copy(&uuid.data[0], &uuid.data[length], pSettings->id);

    return true;
}

//!< ネットワーク設定を自動接続に利用するか否かを表すフラグをエクスポートします。
bool ExportNetworkSettingsAutoConnection(
    Node* pDictNode,
    const ::nn::settings::system::NetworkSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateBooleanNode();

    {
        NameScope nameScope(KeyNetworkSettingsAutoConnection);

        COMMAND_DO(
            node.SetValue(
                settings.flags.Test<
                    ::nn::settings::system::ProfileFlag::AutoConnectable>()));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeyNetworkSettingsAutoConnection, ::std::move(node)));

    return true;
}

//!< ネットワーク設定を自動接続に利用するか否かを表すフラグをインポートします。
bool ImportNetworkSettingsAutoConnection(
    ::nn::settings::system::NetworkSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyNetworkSettingsAutoConnection));

    NameScope nameScope(KeyNetworkSettingsAutoConnection);

    auto isAutoConnectable = bool();
    COMMAND_DO(pNode->GetValue(&isAutoConnectable));

    pSettings->flags.Set<
        ::nn::settings::system::ProfileFlag::AutoConnectable>(
            isAutoConnectable);

    return true;
}

//!< ネットワーク設定が大容量通信を許可するか否かを表すフラグをエクスポートします。
bool ExportNetworkSettingsLargeCapacity(
    Node* pDictNode,
    const ::nn::settings::system::NetworkSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateBooleanNode();

    {
        NameScope nameScope(KeyNetworkSettingsLargeCapacity);

        COMMAND_DO(
            node.SetValue(
                settings.flags.Test<
                    ::nn::settings::system::ProfileFlag::LargeCapacity>()));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeyNetworkSettingsLargeCapacity, ::std::move(node)));

    return true;
}

//!< ネットワーク設定が大容量通信を許可するか否かを表すフラグをインポートします。
bool ImportNetworkSettingsLargeCapacity(
    ::nn::settings::system::NetworkSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyNetworkSettingsLargeCapacity));

    NameScope nameScope(KeyNetworkSettingsLargeCapacity);

    auto isLargeCapacity = bool();
    COMMAND_DO(pNode->GetValue(&isLargeCapacity));

    pSettings->flags.Set<
        ::nn::settings::system::ProfileFlag::LargeCapacity>(
            isLargeCapacity);

    return true;
}

//!< ネットワーク設定の NIC 種別をエクスポートします。
bool ExportNetworkSettingsNicType(
    Node* pDictNode,
    const ::nn::settings::system::NetworkSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateStringNode();

    {
        NameScope nameScope(KeyNetworkSettingsNicType);

        switch (settings.nicType)
        {
        case ::nn::settings::system::NicType_Ieee80211:
            COMMAND_DO(node.SetValue(NicTypeIeee80211));
            break;

        case ::nn::settings::system::NicType_Ethernet:
            COMMAND_DO(node.SetValue(NicTypeEthernet));
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyNetworkSettingsNicType, ::std::move(node)));

    return true;
}

//!< ネットワーク設定の NIC 種別をインポートします。
bool ImportNetworkSettingsNicType(
    ::nn::settings::system::NetworkSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyNetworkSettingsNicType));

    NameScope nameScope(KeyNetworkSettingsNicType);

    ::std::string nicType;
    COMMAND_DO(pNode->GetValue(&nicType));

    if (nicType == NicTypeIeee80211)
    {
        pSettings->nicType =
            static_cast<int32_t>(::nn::settings::system::NicType_Ieee80211);

        return true;
    }

    if (nicType == NicTypeEthernet)
    {
        pSettings->nicType =
            static_cast<int32_t>(::nn::settings::system::NicType_Ethernet);

        return true;
    }

    PrintErrorCode(ErrorCode::NodeValueInvalid, nameScope.Get(), nicType);

    return false;
}

//!< SSID 設定の SSID をエクスポートします。
bool ExportSsidSettingsSsid(
    Node* pDictNode,
    const ::nn::settings::system::SsidSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    const auto length = ::std::min(::std::max(
        0, settings.ssidLength), static_cast<int>(sizeof(settings.ssid)));

    bool printable = true;

    for (int i = 0; i < length; ++i)
    {
        auto value = static_cast<char>(settings.ssid[i]);

        if (value < ' ' || '~' < value || value == '&' ||
                                          value == '<' ||
                                          value == '>')
        {
            printable = false;

            break;
        }
    }

    auto node = Node::CreateStringNode();

    if (printable)
    {
        ::std::string name(
            reinterpret_cast<const char*>(settings.ssid),
            static_cast<size_t>(length));

        {
            NameScope nameScope(KeySsidSettingsName);

            COMMAND_DO(node.SetValue(name));
        }

        COMMAND_DO(
            pDictNode->AppendMember(KeySsidSettingsName, ::std::move(node)));
    }
    else
    {
        char ssid[sizeof(settings.ssid) * 2 + 1] = {};

        for (int i = 0; i < length; ++i)
        {
            ::nn::util::SNPrintf(&ssid[i * 2], 3, "%02X", settings.ssid[i]);
        }

        {
            NameScope nameScope(KeySsidSettingsHex);

            COMMAND_DO(node.SetValue(ssid));
        }

        COMMAND_DO(
            pDictNode->AppendMember(KeySsidSettingsHex, ::std::move(node)));
    }

    return true;
}

//!< SSID 設定の SSID をインポートします。
bool ImportSsidSettingsSsid(
    ::nn::settings::system::SsidSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::vector<::std::string> keys;
    COMMAND_DO(dictNode.GetKeys(&keys));

    for (::nn::Bit8& value : pSettings->ssid)
    {
        value = static_cast<::nn::Bit8>(0);
    }

    for (const ::std::string& key : keys)
    {
        ::std::unique_ptr<const Node> pNode;
        COMMAND_DO(dictNode.GetMember(&pNode, key));

        NameScope nameScope(key);

        if (key == KeySsidSettingsHex)
        {
            ::std::string hex;
            COMMAND_DO(pNode->GetValue(&hex));

            const auto length = static_cast<int>(hex.length());

            for (int i = 0; i < length; i += 2)
            {
                if (i + 2 <= ::nn::settings::system::SsidLengthMax * 2)
                {
                    int64_t value = 0;

                    if (DecodeInteger(&value, hex.substr(i, 2), 16))
                    {
                        pSettings->ssid[i / 2] = static_cast<::nn::Bit8>(value);

                        continue;
                    }
                }

                PrintErrorCode(
                    ErrorCode::NodeValueInvalid, nameScope.Get(), hex);

                return false;
            }

            pSettings->ssidLength = static_cast<int32_t>(length / 2);

            return true;
        }

        if (key == KeySsidSettingsName)
        {
            ::std::string name;
            COMMAND_DO(pNode->GetValue(&name));

            const auto length = static_cast<int>(name.length());

            if (length <= ::nn::settings::system::SsidLengthMax)
            {
                ::std::copy(
                    name.begin(), name.end(),
                    reinterpret_cast<char*>(pSettings->ssid));

                pSettings->ssidLength = static_cast<int32_t>(length);

                return true;
            }

            PrintErrorCode(ErrorCode::NodeValueTooLong, nameScope.Get(), name);

            return false;
        }
    }

    PrintErrorCode(
        ErrorCode::NodeKeyMissing, NameScope(KeySsidSettingsName).Get());

    return false;
}

//!< SSID 設定がステルス状態か否かを表すフラグをエクスポートします。
bool ExportSsidSettingsNonBroadcast(
    Node* pDictNode,
    const ::nn::settings::system::SsidSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateBooleanNode();

    {
        NameScope nameScope(KeySsidSettingsNonBroadcast);

        COMMAND_DO(
            node.SetValue(
                settings.flags.Test<
                    ::nn::settings::system::SsidFlag::IsCloaked>()));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeySsidSettingsNonBroadcast, ::std::move(node)));

    return true;
}

//!< SSID 設定がステルス状態か否かを表すフラグをインポートします。
bool ImportSsidSettingsNonBroadcast(
    ::nn::settings::system::SsidSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeySsidSettingsNonBroadcast));

    NameScope nameScope(KeySsidSettingsNonBroadcast);

    auto isCloaked = bool();
    COMMAND_DO(pNode->GetValue(&isCloaked));

    pSettings->flags.Set<
        ::nn::settings::system::SsidFlag::IsCloaked>(isCloaked);

    return true;
}

//!< ネットワーク設定の SSID 設定をエクスポートします。
bool ExportNetworkSettingsSsidSettings(
    Node* pDictNode,
    const ::nn::settings::system::NetworkSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyNetworkSettingsSsidSettings);

        COMMAND_DO(
            ExportSsidSettingsSsid(&node, settings.ssidSettings));

        COMMAND_DO(
            ExportSsidSettingsNonBroadcast(&node, settings.ssidSettings));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeyNetworkSettingsSsidSettings, ::std::move(node)));

    return true;
}

//!< ネットワーク設定の SSID 設定をインポートします。
bool ImportNetworkSettingsSsidSettings(
    ::nn::settings::system::NetworkSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyNetworkSettingsSsidSettings));

    NameScope nameScope(KeyNetworkSettingsSsidSettings);

    COMMAND_DO(
        ImportSsidSettingsSsid(&pSettings->ssidSettings, *pNode));

    COMMAND_DO(
        ImportSsidSettingsNonBroadcast(&pSettings->ssidSettings, *pNode));

    return true;
}

//!< セキュリティ設定の認証方式をエクスポートします。
bool ExportSecuritySettingsAuthentication(
    Node* pDictNode,
    const ::nn::settings::system::SecuritySettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateStringNode();

    {
        NameScope nameScope(KeySecuritySettingsAuthentication);

        switch (settings.authenticationMode)
        {
        case ::nn::settings::system::AuthenticationType_Open:
            COMMAND_DO(node.SetValue(AuthenticationModeOpen));
            break;

        case ::nn::settings::system::AuthenticationType_Shared:
            COMMAND_DO(node.SetValue(AuthenticationModeShared));
            break;

        case ::nn::settings::system::AuthenticationType_WpaPsk:
            COMMAND_DO(node.SetValue(AuthenticationModeWpaPsk));
            break;

        case ::nn::settings::system::AuthenticationType_Wpa2Psk:
            COMMAND_DO(node.SetValue(AuthenticationModeWpa2Psk));
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeySecuritySettingsAuthentication, ::std::move(node)));

    return true;
}

//!< セキュリティ設定の認証方式をインポートします。
bool ImportSecuritySettingsAuthentication(
    ::nn::settings::system::SecuritySettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeySecuritySettingsAuthentication));

    NameScope nameScope(KeySecuritySettingsAuthentication);

    ::std::string authenticationMode;
    COMMAND_DO(pNode->GetValue(&authenticationMode));

    if (authenticationMode == AuthenticationModeOpen)
    {
        pSettings->authenticationMode = static_cast<int32_t>(
            ::nn::settings::system::AuthenticationType_Open);

        return true;
    }

    if (authenticationMode == AuthenticationModeShared)
    {
        pSettings->authenticationMode = static_cast<int32_t>(
            ::nn::settings::system::AuthenticationType_Shared);

        return true;
    }

    if (authenticationMode == AuthenticationModeWpaPsk)
    {
        pSettings->authenticationMode = static_cast<int32_t>(
            ::nn::settings::system::AuthenticationType_WpaPsk);

        return true;
    }

    if (authenticationMode == AuthenticationModeWpa2Psk)
    {
        pSettings->authenticationMode = static_cast<int32_t>(
            ::nn::settings::system::AuthenticationType_Wpa2Psk);

        return true;
    }

    PrintErrorCode(
        ErrorCode::NodeValueInvalid, nameScope.Get(), authenticationMode);

    return false;
}

//!< セキュリティ設定の暗号化方式をエクスポートします。
bool ExportSecuritySettingsEncryption(
    Node* pDictNode,
    const ::nn::settings::system::SecuritySettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateStringNode();

    {
        NameScope nameScope(KeySecuritySettingsEncryption);

        switch (settings.encryptionMode)
        {
        case ::nn::settings::system::EncryptionMode_None:
            COMMAND_DO(node.SetValue(EncryptionModeNone));
            break;

        case ::nn::settings::system::EncryptionMode_Wep:
            COMMAND_DO(node.SetValue(EncryptionModeWep));
            break;

        case ::nn::settings::system::EncryptionMode_Aes:
            COMMAND_DO(node.SetValue(EncryptionModeAes));
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeySecuritySettingsEncryption, ::std::move(node)));

    return true;
}

//!< セキュリティ設定の暗号化方式をインポートします。
bool ImportSecuritySettingsEncryption(
    ::nn::settings::system::SecuritySettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeySecuritySettingsEncryption));

    NameScope nameScope(KeySecuritySettingsEncryption);

    ::std::string encryptionMode;
    COMMAND_DO(pNode->GetValue(&encryptionMode));

    if (encryptionMode == EncryptionModeNone)
    {
        pSettings->encryptionMode =
            static_cast<int32_t>(::nn::settings::system::EncryptionMode_None);

        return true;
    }

    if (encryptionMode == EncryptionModeWep)
    {
        pSettings->encryptionMode =
            static_cast<int32_t>(::nn::settings::system::EncryptionMode_Wep);

        return true;
    }

    if (encryptionMode == EncryptionModeAes)
    {
        pSettings->encryptionMode =
            static_cast<int32_t>(::nn::settings::system::EncryptionMode_Aes);

        return true;
    }

    PrintErrorCode(
        ErrorCode::NodeValueInvalid, nameScope.Get(), encryptionMode);

    return false;
}

//!< セキュリティ設定の Key Material をエクスポートします。
bool ExportSecuritySettingsKeyMaterial(
    Node* pDictNode,
    const ::nn::settings::system::SecuritySettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    const auto length = ::std::min(::std::max(
        0,
        settings.keyMaterialLength),
        static_cast<int>(NN_ARRAY_SIZE(settings.keyMaterial)));

    auto node = Node::CreateStringNode();

    {
        NameScope nameScope(KeySecuritySettingsKeyMaterial);

        COMMAND_DO(
            node.SetValue(
                ::std::string(
                    settings.keyMaterial, static_cast<size_t>(length))));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeySecuritySettingsKeyMaterial, ::std::move(node)));

    return true;
}

//!< セキュリティ設定の Key Material をインポートします。
bool ImportSecuritySettingsKeyMaterial(
    ::nn::settings::system::SecuritySettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeySecuritySettingsKeyMaterial));

    NameScope nameScope(KeySecuritySettingsKeyMaterial);

    ::std::string keyMaterial;
    COMMAND_DO(pNode->GetValue(&keyMaterial));

    const auto length = keyMaterial.length();

    if (length <= NN_ARRAY_SIZE(pSettings->keyMaterial))
    {
        for (char& value : pSettings->keyMaterial)
        {
            value = '\0';
        }

        ::std::copy(
            keyMaterial.begin(), keyMaterial.end(), pSettings->keyMaterial);

        pSettings->keyMaterialLength = static_cast<int32_t>(length);

        return true;
    }

    PrintErrorCode(ErrorCode::NodeValueTooLong, nameScope.Get(), keyMaterial);

    return false;
}

//!< ネットワーク設定のセキュリティ設定をエクスポートします。
bool ExportNetworkSettingsSecuritySettings(
    Node* pDictNode,
    const ::nn::settings::system::NetworkSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyNetworkSettingsSecuritySettings);

        COMMAND_DO(
            ExportSecuritySettingsAuthentication(
                &node, settings.securitySettings));

        COMMAND_DO(
            ExportSecuritySettingsEncryption(&node, settings.securitySettings));

        COMMAND_DO(
            ExportSecuritySettingsKeyMaterial(
                &node, settings.securitySettings));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeyNetworkSettingsSecuritySettings, ::std::move(node)));

    return true;
}

//!< ネットワーク設定のセキュリティ設定をインポートします。
bool ImportNetworkSettingsSecuritySettings(
    ::nn::settings::system::NetworkSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyNetworkSettingsSecuritySettings));

    NameScope nameScope(KeyNetworkSettingsSecuritySettings);

    COMMAND_DO(
        ImportSecuritySettingsAuthentication(
            &pSettings->securitySettings, *pNode));

    COMMAND_DO(
        ImportSecuritySettingsEncryption(
            &pSettings->securitySettings, *pNode));

    COMMAND_DO(
        ImportSecuritySettingsKeyMaterial(
            &pSettings->securitySettings, *pNode));

    return true;
}

//!< IP 設定の IP アドレスの自動取得の有効化フラグをエクスポートします。
bool ExportIpSettingsAutoIp(
    Node* pDictNode,
    const ::nn::settings::system::IpSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateBooleanNode();

    {
        NameScope nameScope(KeyIpSettingsAutoIp);

        COMMAND_DO(
            node.SetValue(
                settings.flags.Test<
                    ::nn::settings::system::IpFlag::EnablesAutoIp>()));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyIpSettingsAutoIp, ::std::move(node)));

    return true;
}

//!< IP 設定の IP アドレスの自動取得の有効化フラグをインポートします。
bool ImportIpSettingsAutoIp(::nn::settings::system::IpSettings* pSettings,
                            const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyIpSettingsAutoIp));

    NameScope nameScope(KeyIpSettingsAutoIp);

    auto enablesAutoIp = bool();
    COMMAND_DO(pNode->GetValue(&enablesAutoIp));

    pSettings->flags.Set<
        ::nn::settings::system::IpFlag::EnablesAutoIp>(enablesAutoIp);

    return true;
}

//!< IP 設定の DNS サーバアドレスの自動取得の有効化フラグをエクスポートします。
bool ExportIpSettingsAutoDns(
    Node* pDictNode,
    const ::nn::settings::system::IpSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateBooleanNode();

    {
        NameScope nameScope(KeyIpSettingsAutoDns);

        COMMAND_DO(
            node.SetValue(
                settings.flags.Test<
                    ::nn::settings::system::IpFlag::EnablesAutoDns>()));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyIpSettingsAutoDns, ::std::move(node)));

    return true;
}

//!< IP 設定の DNS サーバアドレスの自動取得の有効化フラグをインポートします。
bool ImportIpSettingsAutoDns(::nn::settings::system::IpSettings* pSettings,
                             const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyIpSettingsAutoDns));

    NameScope nameScope(KeyIpSettingsAutoDns);

    auto enablesAutoDns = bool();
    COMMAND_DO(pNode->GetValue(&enablesAutoDns));

    pSettings->flags.Set<
        ::nn::settings::system::IpFlag::EnablesAutoDns>(enablesAutoDns);

    return true;
}

//!< IP 設定の IP アドレスをエクスポートします。
bool ExportIpSettingsAddress(
    Node* pDictNode,
    const ::nn::settings::system::IpSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateStringNode();

    {
        NameScope nameScope(KeyIpSettingsAddress);

        COMMAND_DO(node.SetValue(GetStringOfAddress(settings.ipAddress)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyIpSettingsAddress, ::std::move(node)));

    return true;
}

//!< IP 設定の IP アドレスをインポートします。
bool ImportIpSettingsAddress(::nn::settings::system::IpSettings* pSettings,
                             const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyIpSettingsAddress));

    NameScope nameScope(KeyIpSettingsAddress);

    ::std::string address;
    COMMAND_DO(pNode->GetValue(&address));

    if (ParseIpAddress(pSettings->ipAddress, address))
    {
        return true;
    }

    PrintErrorCode(ErrorCode::NodeValueInvalid, nameScope.Get(), address);

    return false;
}

//!< IP 設定のサブネットマスクをエクスポートします。
bool ExportIpSettingsSubnetMask(
    Node* pDictNode,
    const ::nn::settings::system::IpSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateStringNode();

    {
        NameScope nameScope(KeyIpSettingsSubnetMask);

        COMMAND_DO(node.SetValue(GetStringOfAddress(settings.subnetMask)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyIpSettingsSubnetMask, ::std::move(node)));

    return true;
}

//!< IP 設定のサブネットマスクをインポートします。
bool ImportIpSettingsSubnetMask(::nn::settings::system::IpSettings* pSettings,
                                const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyIpSettingsSubnetMask));

    NameScope nameScope(KeyIpSettingsSubnetMask);

    ::std::string subnetMask;
    COMMAND_DO(pNode->GetValue(&subnetMask));

    if (ParseIpAddress(pSettings->subnetMask, subnetMask))
    {
        return true;
    }

    PrintErrorCode(ErrorCode::NodeValueInvalid, nameScope.Get(), subnetMask);

    return false;
}

//!< IP 設定のデフォルトゲートウェイをエクスポートします。
bool ExportIpSettingsDefaultGateway(
    Node* pDictNode,
    const ::nn::settings::system::IpSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateStringNode();

    {
        NameScope nameScope(KeyIpSettingsDefaultGateway);

        COMMAND_DO(
            node.SetValue(GetStringOfAddress(settings.defaultGateway)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeyIpSettingsDefaultGateway, ::std::move(node)));

    return true;
}

//!< IP 設定のデフォルトゲートウェイをインポートします。
bool ImportIpSettingsDefaultGateway(
    ::nn::settings::system::IpSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyIpSettingsDefaultGateway));

    NameScope nameScope(KeyIpSettingsDefaultGateway);

    ::std::string defaultGateway;
    COMMAND_DO(pNode->GetValue(&defaultGateway));

    if (ParseIpAddress(pSettings->defaultGateway, defaultGateway))
    {
        return true;
    }

    PrintErrorCode(
        ErrorCode::NodeValueInvalid, nameScope.Get(), defaultGateway);

    return false;
}

//!< IP 設定の優先 DNS サーバのアドレスをエクスポートします。
bool ExportIpSettingsPreferredDns(
    Node* pDictNode,
    const ::nn::settings::system::IpSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateStringNode();

    {
        NameScope nameScope(KeyIpSettingsPreferredDns);

        COMMAND_DO(node.SetValue(GetStringOfAddress(settings.preferredDns)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyIpSettingsPreferredDns, ::std::move(node)));

    return true;
}

//!< IP 設定の優先 DNS サーバのアドレスをインポートします。
bool ImportIpSettingsPreferredDns(
    ::nn::settings::system::IpSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyIpSettingsPreferredDns));

    NameScope nameScope(KeyIpSettingsPreferredDns);

    ::std::string preferredDns;
    COMMAND_DO(pNode->GetValue(&preferredDns));

    if (ParseIpAddress(pSettings->preferredDns, preferredDns))
    {
        return true;
    }

    PrintErrorCode(ErrorCode::NodeValueInvalid, nameScope.Get(), preferredDns);

    return false;
}

//!< IP 設定の代替 DNS サーバのアドレスをエクスポートします。
bool ExportIpSettingsAlternateDns(
    Node* pDictNode,
    const ::nn::settings::system::IpSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateStringNode();

    {
        NameScope nameScope(KeyIpSettingsAlternateDns);

        COMMAND_DO(node.SetValue(GetStringOfAddress(settings.alternateDns)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyIpSettingsAlternateDns, ::std::move(node)));

    return true;
}

//!< IP 設定の代替 DNS サーバのアドレスをインポートします。
bool ImportIpSettingsAlternateDns(
    ::nn::settings::system::IpSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyIpSettingsAlternateDns));

    NameScope nameScope(KeyIpSettingsAlternateDns);

    ::std::string alternateDns;
    COMMAND_DO(pNode->GetValue(&alternateDns));

    if (ParseIpAddress(pSettings->alternateDns, alternateDns))
    {
        return true;
    }

    PrintErrorCode(ErrorCode::NodeValueInvalid, nameScope.Get(), alternateDns);

    return false;
}

//!< ネットワーク設定の IP 設定をエクスポートします。
bool ExportNetworkSettingsIpSettings(
    Node* pDictNode,
    const ::nn::settings::system::NetworkSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyNetworkSettingsIpSettings);

        COMMAND_DO(ExportIpSettingsAutoIp(&node, settings.ipSettings));

        COMMAND_DO(ExportIpSettingsAddress(&node, settings.ipSettings));

        COMMAND_DO(ExportIpSettingsSubnetMask(&node, settings.ipSettings));

        COMMAND_DO(ExportIpSettingsDefaultGateway(&node, settings.ipSettings));

        COMMAND_DO(ExportIpSettingsAutoDns(&node, settings.ipSettings));

        COMMAND_DO(ExportIpSettingsPreferredDns(&node, settings.ipSettings));

        COMMAND_DO(ExportIpSettingsAlternateDns(&node, settings.ipSettings));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeyNetworkSettingsIpSettings, ::std::move(node)));

    return true;
}

//!< ネットワーク設定の IP 設定をインポートします。
bool ImportNetworkSettingsIpSettings(
    ::nn::settings::system::NetworkSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyNetworkSettingsIpSettings));

    NameScope nameScope(KeyNetworkSettingsIpSettings);

    COMMAND_DO(ImportIpSettingsAutoIp(&pSettings->ipSettings, *pNode));

    COMMAND_DO(ImportIpSettingsAddress(&pSettings->ipSettings, *pNode));

    COMMAND_DO(ImportIpSettingsSubnetMask(&pSettings->ipSettings, *pNode));

    COMMAND_DO(ImportIpSettingsDefaultGateway(&pSettings->ipSettings, *pNode));

    COMMAND_DO(ImportIpSettingsAutoDns(&pSettings->ipSettings, *pNode));

    COMMAND_DO(ImportIpSettingsPreferredDns(&pSettings->ipSettings, *pNode));

    COMMAND_DO(ImportIpSettingsAlternateDns(&pSettings->ipSettings, *pNode));

    return true;
}

//!< Proxy 設定の有効化フラグをエクスポートします。
bool ExportProxySettingsEnables(
    Node* pDictNode,
    const ::nn::settings::system::ProxySettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateBooleanNode();

    {
        NameScope nameScope(KeyProxySettingsEnables);

        COMMAND_DO(
            node.SetValue(
                settings.flags.Test<
                    ::nn::settings::system::ProxyFlag::Enables>()));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyProxySettingsEnables, ::std::move(node)));

    return true;
}

//!< Proxy 設定の有効化フラグをインポートします。
bool ImportProxySettingsEnables(
    ::nn::settings::system::ProxySettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyProxySettingsEnables));

    NameScope nameScope(KeyProxySettingsEnables);

    auto enables = bool();
    COMMAND_DO(pNode->GetValue(&enables));

    pSettings->flags.Set<::nn::settings::system::ProxyFlag::Enables>(enables);

    return true;
}

//!< Proxy 設定のユーザ認証の有効化フラグをエクスポートします。
bool ExportProxySettingsAuthenticates(
    Node* pDictNode,
    const ::nn::settings::system::ProxySettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateBooleanNode();

    {
        NameScope nameScope(KeyProxySettingsAuthenticates);

        COMMAND_DO(
            node.SetValue(
                settings.flags.Test<
                    ::nn::settings::system::ProxyFlag::Authenticates>()));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeyProxySettingsAuthenticates, ::std::move(node)));

    return true;
}

//!< Proxy 設定のユーザ認証の有効化フラグをインポートします。
bool ImportProxySettingsAuthenticates(
    ::nn::settings::system::ProxySettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyProxySettingsAuthenticates));

    NameScope nameScope(KeyProxySettingsAuthenticates);

    auto authenticates = bool();
    COMMAND_DO(pNode->GetValue(&authenticates));

    pSettings->flags.Set<
        ::nn::settings::system::ProxyFlag::Authenticates>(authenticates);

    return true;
}

//!< Proxy 設定のホスト名をエクスポートします。
bool ExportProxySettingsHostName(
    Node* pDictNode,
    const ::nn::settings::system::ProxySettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateStringNode();

    {
        NameScope nameScope(KeyProxySettingsHostName);

        const int length = ::nn::util::Strnlen(
            settings.hostName,
            static_cast<int>(NN_ARRAY_SIZE(settings.hostName) - 1));

        COMMAND_DO(
            node.SetValue(
                ::std::string(settings.hostName, static_cast<size_t>(length))));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyProxySettingsHostName, ::std::move(node)));

    return true;
}

//!< Proxy 設定のホスト名をインポートします。
bool ImportProxySettingsHostName(
    ::nn::settings::system::ProxySettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyProxySettingsHostName));

    NameScope nameScope(KeyProxySettingsHostName);

    ::std::string hostName;
    COMMAND_DO(pNode->GetValue(&hostName));

    if (hostName.size() < NN_ARRAY_SIZE(pSettings->hostName))
    {
        for (char& value : pSettings->hostName)
        {
            value = '\0';
        }

        ::std::copy(hostName.begin(), hostName.end(), pSettings->hostName);

        return true;
    }

    PrintErrorCode(ErrorCode::NodeValueTooLong, nameScope.Get(), hostName);

    return false;
}

//!< Proxy 設定のポート番号をエクスポートします。
bool ExportProxySettingsPort(
    Node* pDictNode,
    const ::nn::settings::system::ProxySettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateIntegerNode();

    {
        NameScope nameScope(KeyProxySettingsPort);

        COMMAND_DO(node.SetValue(static_cast<int32_t>(settings.portNumber)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyProxySettingsPort, ::std::move(node)));

    return true;
}

//!< Proxy 設定のポート番号をインポートします。
bool ImportProxySettingsPort(::nn::settings::system::ProxySettings* pSettings,
                             const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyProxySettingsPort));

    NameScope nameScope(KeyProxySettingsPort);

    auto port = int32_t();
    COMMAND_DO(pNode->GetValue(&port));

    pSettings->portNumber = static_cast<uint16_t>(port);

    return true;
}

//!< Proxy 設定のユーザ認証のユーザ名をエクスポートします。
bool ExportProxySettingsUserName(
    Node* pDictNode,
    const ::nn::settings::system::ProxySettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateStringNode();

    {
        NameScope nameScope(KeyProxySettingsUserName);

        const int length = ::nn::util::Strnlen(
            settings.userName,
            static_cast<int>(NN_ARRAY_SIZE(settings.userName) - 1));

        COMMAND_DO(
            node.SetValue(
                ::std::string(settings.userName, static_cast<size_t>(length))));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyProxySettingsUserName, ::std::move(node)));

    return true;
}

//!< Proxy 設定のユーザ認証のユーザ名をインポートします。
bool ImportProxySettingsUserName(
    ::nn::settings::system::ProxySettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyProxySettingsUserName));

    NameScope nameScope(KeyProxySettingsUserName);

    ::std::string userName;
    COMMAND_DO(pNode->GetValue(&userName));

    if (userName.size() < NN_ARRAY_SIZE(pSettings->userName))
    {
        for (char& value : pSettings->userName)
        {
            value = '\0';
        }

        ::std::copy(userName.begin(), userName.end(), pSettings->userName);

        return true;
    }

    PrintErrorCode(ErrorCode::NodeValueTooLong, nameScope.Get(), userName);

    return false;
}

//!< Proxy 設定のユーザ認証のパスワードをエクスポートします。
bool ExportProxySettingsPassword(
    Node* pDictNode,
    const ::nn::settings::system::ProxySettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateStringNode();

    {
        NameScope nameScope(KeyProxySettingsPassword);

        const int length = ::nn::util::Strnlen(
            settings.password,
            static_cast<int>(NN_ARRAY_SIZE(settings.password) - 1));

        COMMAND_DO(
            node.SetValue(
                ::std::string(settings.password, static_cast<size_t>(length))));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyProxySettingsPassword, ::std::move(node)));

    return true;
}

//!< Proxy 設定のユーザ認証のパスワードをインポートします。
bool ImportProxySettingsPassword(
    ::nn::settings::system::ProxySettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyProxySettingsPassword));

    NameScope nameScope(KeyProxySettingsPassword);

    ::std::string password;
    COMMAND_DO(pNode->GetValue(&password));

    if (password.size() < NN_ARRAY_SIZE(pSettings->password))
    {
        for (char& value : pSettings->password)
        {
            value = '\0';
        }

        ::std::copy(password.begin(), password.end(), pSettings->password);

        return true;
    }

    PrintErrorCode(ErrorCode::NodeValueTooLong, nameScope.Get(), password);

    return false;
}

//!< ネットワーク設定の Proxy 設定をエクスポートします。
bool ExportNetworkSettingsProxySettings(
    Node* pDictNode,
    const ::nn::settings::system::NetworkSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyNetworkSettingsProxySettings);

        COMMAND_DO(ExportProxySettingsEnables(&node, settings.proxySettings));

        COMMAND_DO(ExportProxySettingsHostName(&node, settings.proxySettings));

        COMMAND_DO(ExportProxySettingsPort(&node, settings.proxySettings));

        COMMAND_DO(
            ExportProxySettingsAuthenticates(&node, settings.proxySettings));

        COMMAND_DO(ExportProxySettingsUserName(&node, settings.proxySettings));

        COMMAND_DO(ExportProxySettingsPassword(&node, settings.proxySettings));
    }

    COMMAND_DO(
        pDictNode->AppendMember(
            KeyNetworkSettingsProxySettings, ::std::move(node)));

    return true;
}

//!< ネットワーク設定の Proxy 設定をインポートします。
bool ImportNetworkSettingsProxySettings(
    ::nn::settings::system::NetworkSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyNetworkSettingsProxySettings));

    NameScope nameScope(KeyNetworkSettingsProxySettings);

    COMMAND_DO(
        ImportProxySettingsEnables(&pSettings->proxySettings, *pNode));

    COMMAND_DO(
        ImportProxySettingsAuthenticates(&pSettings->proxySettings, *pNode));

    COMMAND_DO(
        ImportProxySettingsHostName(&pSettings->proxySettings, *pNode));

    COMMAND_DO(
        ImportProxySettingsPort(&pSettings->proxySettings, *pNode));

    COMMAND_DO(
        ImportProxySettingsUserName(&pSettings->proxySettings, *pNode));

    COMMAND_DO(
        ImportProxySettingsPassword(&pSettings->proxySettings, *pNode));

    return true;
}

//!< ネットワーク設定の MTU をエクスポートします。
bool ExportNetworkSettingsMtu(
    Node* pDictNode,
    const ::nn::settings::system::NetworkSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateIntegerNode();

    {
        NameScope nameScope(KeyNetworkSettingsMtu);

        COMMAND_DO(node.SetValue(static_cast<int32_t>(settings.mtu)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyNetworkSettingsMtu, ::std::move(node)));

    return true;
}

//!< ネットワーク設定の MTU をインポートします。
bool ImportNetworkSettingsMtu(
    ::nn::settings::system::NetworkSettings* pSettings,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyNetworkSettingsMtu));

    NameScope nameScope(KeyNetworkSettingsMtu);

    auto mtu = int32_t();
    COMMAND_DO(pNode->GetValue(&mtu));

    pSettings->mtu = static_cast<uint16_t>(mtu);

    return true;
}

} // namespace

bool IsSettingNameNetworkSettings(const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNameNetworkSettings);
}

bool ExportNetworkSettings(Node* pNode) NN_NOEXCEPT
{
    const int EntryCountMax = ::nn::settings::system::NetworkSettingsCountMax;

    ::std::unique_ptr<::nn::settings::system::NetworkSettings[]> entries(
        new ::nn::settings::system::NetworkSettings[EntryCountMax]());

    const int count = ::nn::settings::system::GetNetworkSettings(
        entries.get(), EntryCountMax);

    auto arrayNode = Node::CreateArrayNode();

    for (int i = 0; i < count; ++i)
    {
        NameScope nameScope(SettingNameNetworkSettings);

        const ::nn::settings::system::NetworkSettings& settings = entries[i];

        auto dictNode = Node::CreateObjectNode();

        COMMAND_DO(ExportNetworkSettingsName(&dictNode, settings));

        COMMAND_DO(ExportNetworkSettingsId(&dictNode, settings));

        COMMAND_DO(ExportNetworkSettingsAutoConnection(&dictNode, settings));

        COMMAND_DO(ExportNetworkSettingsLargeCapacity(&dictNode, settings));

        COMMAND_DO(ExportNetworkSettingsNicType(&dictNode, settings));

        COMMAND_DO(ExportNetworkSettingsSsidSettings(&dictNode, settings));

        COMMAND_DO(ExportNetworkSettingsSecuritySettings(&dictNode, settings));

        COMMAND_DO(ExportNetworkSettingsIpSettings(&dictNode, settings));

        COMMAND_DO(ExportNetworkSettingsProxySettings(&dictNode, settings));

        COMMAND_DO(ExportNetworkSettingsMtu(&dictNode, settings));

        COMMAND_DO(arrayNode.AppendElement(::std::move(dictNode)));
    }

    COMMAND_DO(
        pNode->AppendMember(
            SettingNameNetworkSettings, ::std::move(arrayNode)));

    return true;
}

bool ImportNetworkSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNameNetworkSettings);

    const int EntryCountMax = ::nn::settings::system::NetworkSettingsCountMax;

    ::std::unique_ptr<::nn::settings::system::NetworkSettings[]> entries(
        new ::nn::settings::system::NetworkSettings[EntryCountMax]());

    int count = 0;

    auto elementCount = size_t();

    COMMAND_DO(node.GetElementCount(&elementCount));

    for (size_t i = 0; i < elementCount; ++i)
    {
        if (count >= EntryCountMax)
        {
            break;
        }

        ::nn::settings::system::NetworkSettings& settings = entries[count];

        ::std::unique_ptr<const Node> pNode;
        COMMAND_DO(node.GetElement(&pNode, i));

        COMMAND_DO(ImportNetworkSettingsName(&settings, *pNode));

        COMMAND_DO(ImportNetworkSettingsId(&settings, *pNode));

        COMMAND_DO(ImportNetworkSettingsAutoConnection(&settings, *pNode));

        COMMAND_DO(ImportNetworkSettingsLargeCapacity(&settings, *pNode));

        COMMAND_DO(ImportNetworkSettingsNicType(&settings, *pNode));

        COMMAND_DO(ImportNetworkSettingsSsidSettings(&settings, *pNode));

        COMMAND_DO(ImportNetworkSettingsSecuritySettings(&settings, *pNode));

        COMMAND_DO(ImportNetworkSettingsIpSettings(&settings, *pNode));

        COMMAND_DO(ImportNetworkSettingsProxySettings(&settings, *pNode));

        COMMAND_DO(ImportNetworkSettingsMtu(&settings, *pNode));

        ++count;
    }

    ::nn::settings::system::SetNetworkSettings(entries.get(), count);

    return true;
}
