﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <string>
#include <utility>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/settings/system/settings_Hid.h>

#include "SettingsManager_Hid.h"
#include "SettingsManager_NameScope.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Utility.h"

namespace {

//!< HID 設定のキー
const char* const SettingNameHidSettings = "hid_settings";

//!< 現在値のキー
const char* const KeyCurrent = "current";

//!< 初期値のキー
const char* const KeyDefault = "default";

//!< Pro Controller 有線 USB 通信機能が有効か否かを表す値のキー
const char* const KeyUsbFullKeyEnableFlag = "usb_full_key_enable_flag";

//!< 振動のマスターボリュームを表す値のキー
const char* const KeyVibrationMasterVolume = "vibration_master_volume";

//!< Pro Controller 有線 USB 通信機能が有効か否かを表す値をエクスポートします。
bool ExportUsbFullKeyEnableFlag(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyUsbFullKeyEnableFlag);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                ::nn::settings::system::IsUsbFullKeyEnabled()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(false));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyUsbFullKeyEnableFlag, ::std::move(node)));

    return true;
}

//!< Pro Controller 有線 USB 通信機能が有効か否かを表す値をインポートします。
bool ImportUsbFullKeyEnableFlag(const Node& dictNode) NN_NOEXCEPT
{
    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyUsbFullKeyEnableFlag));

    NameScope nameScope(KeyUsbFullKeyEnableFlag);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto usbFullKeyEnableFlag = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&usbFullKeyEnableFlag));

    ::nn::settings::system::SetUsbFullKeyEnabled(usbFullKeyEnableFlag);

    return true;
}

//!< 振動のマスターボリュームをエクスポートします。
bool ExportVibrationMasterVolume(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyVibrationMasterVolume);

        auto currentValueNode = Node::CreateDoubleNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                static_cast<double>(
                    ::nn::settings::system::GetVibrationMasterVolume())));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateDoubleNode();

        COMMAND_DO(defaultValueNode.SetValue(1.0));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyVibrationMasterVolume, ::std::move(node)));

    return true;
}

//!< 振動のマスターボリュームをインポートします。
bool ImportVibrationMasterVolume(const Node& dictNode) NN_NOEXCEPT
{
    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyVibrationMasterVolume));

    NameScope nameScope(KeyVibrationMasterVolume);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto volume = double();
    COMMAND_DO(pCurrentValueNode->GetValue(&volume));

    ::nn::settings::system::SetVibrationMasterVolume(
        static_cast<float>(volume));

    return true;
}

} // namespace

bool IsSettingNameHidSettings(const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNameHidSettings);
}

bool ExportHidSettings(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(SettingNameHidSettings);

        COMMAND_DO(ExportUsbFullKeyEnableFlag(&node));

        COMMAND_DO(ExportVibrationMasterVolume(&node));
    }

    COMMAND_DO(pNode->AppendMember(SettingNameHidSettings, ::std::move(node)));

    return true;
}

bool ImportHidSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNameHidSettings);

    COMMAND_DO(ImportUsbFullKeyEnableFlag(node));

    COMMAND_DO(ImportVibrationMasterVolume(node));

    return true;
}
