﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <map>
#include <memory>
#include <string>
#include <utility>
#include <vector>
#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs/fs_Result.h>
#include <nn/util/util_ScopeExit.h>

#include "detail/settings_KeyValueStore.h"
#include "detail/settings_SystemData-os.win.h"
#include "detail/settings_SystemSaveData-os.win.h"
#include "SettingsManager_ErrorCode.h"
#include "SettingsManager_FirmwareDebugSettings.h"
#include "SettingsManager_NameScope.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Utility.h"

namespace {

//!< システムデータのマウント名
const char SystemDataMountName[] = "FwdbgSettingsD";

//!< マップのキーのセパレータ
const char MapKeySeparator = '!';

//!< 現在値のキー
const char* const KeyCurrent = "current";

//!< 初期値のキー
const char* const KeyDefault = "default";

//!< 設定項目の型を表す列挙体
enum class Type : uint8_t
{
    Unknown,    //!< 不明な型
    String,     //!< 文字列型
    Bool,       //!< 論理値型
    Int,        //!< 整数型
};

//!< ファームウェアデバッグ設定をエクスポートします。
bool ExportFirmwareDebugSettings(
    Node* pNode, uint8_t type,
    const void* currentValue, size_t currentSize,
    const void* defaultValue, size_t defaultSize) NN_NOEXCEPT;

//!< ファームウェアデバッグ設定をエクスポートします。
bool ExportFirmwareDebugSettings(
    Node* pNode, const ::std::vector<::std::string>& names, bool needsFilter
    ) NN_NOEXCEPT;

//!< ファームウェアデバッグ設定をインポートします。
bool ImportFirmwareDebugSettings(
    const char* mapKey, uint8_t type,
    const void* currentValue, size_t currentSize,
    const void* defaultValue, size_t defaultSize) NN_NOEXCEPT;

} // namespace

bool ExportFirmwareDebugSettings(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    COMMAND_DO(
        ExportFirmwareDebugSettings(
            pNode, ::std::vector<::std::string>(), false));

    return true;
}

bool ExportFirmwareDebugSettings(
    Node* pNode, const ::std::vector<::std::string>& names) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    COMMAND_DO(ExportFirmwareDebugSettings(pNode, names, true));

    return true;
}

bool ImportFirmwareDebugSettings(
    const ::std::string& name, const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(name);

    ::std::vector<::std::string> settingNames;

    COMMAND_DO(node.GetKeys(&settingNames));

    for (const ::std::string& settingName : settingNames)
    {
        ::std::unique_ptr<const Node> pValueNode;
        COMMAND_DO(node.GetMember(&pValueNode, settingName));

        NameScope subNameScope(settingName);

        ::std::unique_ptr<const Node> pCurrentValueNode;
        ::std::unique_ptr<const Node> pDefaultValueNode;
        COMMAND_DO(pValueNode->GetMember(&pCurrentValueNode, KeyCurrent));
        COMMAND_DO(pValueNode->GetMember(&pDefaultValueNode, KeyDefault));

        const ::std::string mapKey = (name + MapKeySeparator) + settingName;

        if (pDefaultValueNode->IsStringNode())
        {
            ::std::string currentValue;
            ::std::string defaultValue;
            COMMAND_DO(pCurrentValueNode->GetValue(&currentValue));
            COMMAND_DO(pDefaultValueNode->GetValue(&defaultValue));
            COMMAND_DO(
                ImportFirmwareDebugSettings(
                    mapKey.c_str(), static_cast<uint8_t>(Type::String),
                    currentValue.c_str(), currentValue.size() + 1,
                    defaultValue.c_str(), defaultValue.size() + 1));
        }
        else if (pDefaultValueNode->IsBooleanNode())
        {
            auto currentValue = bool();
            auto defaultValue = bool();
            COMMAND_DO(pCurrentValueNode->GetValue(&currentValue));
            COMMAND_DO(pDefaultValueNode->GetValue(&defaultValue));
            COMMAND_DO(
                ImportFirmwareDebugSettings(
                    mapKey.c_str(), static_cast<uint8_t>(Type::Bool),
                    &currentValue, sizeof(currentValue),
                    &defaultValue, sizeof(defaultValue)));
        }
        else if (pDefaultValueNode->IsIntegerNode())
        {
            auto currentValue = int32_t();
            auto defaultValue = int32_t();
            COMMAND_DO(pCurrentValueNode->GetValue(&currentValue));
            COMMAND_DO(pDefaultValueNode->GetValue(&defaultValue));
            COMMAND_DO(
                ImportFirmwareDebugSettings(
                    mapKey.c_str(), static_cast<uint8_t>(Type::Int),
                    &currentValue, sizeof(currentValue),
                    &defaultValue, sizeof(defaultValue)));
        }
        else
        {
            PrintErrorCode(
                ErrorCode::NodeTypeUnsupported,
                subNameScope.Get(), pDefaultValueNode->GetTypeName());

            return false;
        }
    }

    return true;
}

bool FlushFirmwareDebugSettings() NN_NOEXCEPT
{
    ::nn::settings::detail::SystemData systemData;
    systemData.SetMountName(SystemDataMountName);

    ::nn::settings::detail::SystemSaveData systemSaveData(
        systemData.GetSystemDataFile());

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::settings::detail::SaveKeyValueStoreAllForDebug(&systemSaveData));

    return true;
}

namespace {

bool ExportFirmwareDebugSettings(
    Node* pNode, uint8_t type,
    const void* currentValue, size_t currentSize,
    const void* defaultValue, size_t defaultSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);
    NN_SDK_REQUIRES_NOT_NULL(currentValue);
    NN_SDK_REQUIRES_NOT_NULL(defaultValue);

    Node currentValueNode;
    Node defaultValueNode;

    switch (static_cast<Type>(type))
    {
    case Type::String:
        currentValueNode = Node::CreateStringNode();
        defaultValueNode = Node::CreateStringNode();
        COMMAND_DO(
            currentValueNode.SetValue(
                ::std::string(
                    reinterpret_cast<const char*>(currentValue),
                    currentSize - 1)));
        COMMAND_DO(
            defaultValueNode.SetValue(
                ::std::string(
                    reinterpret_cast<const char*>(defaultValue),
                    defaultSize - 1)));
        break;

    case Type::Bool:
        currentValueNode = Node::CreateBooleanNode();
        defaultValueNode = Node::CreateBooleanNode();
        COMMAND_DO(
            currentValueNode.SetValue(
                *reinterpret_cast<const bool*>(currentValue)));
        COMMAND_DO(
            defaultValueNode.SetValue(
                *reinterpret_cast<const bool*>(defaultValue)));
        break;

    case Type::Int:
        currentValueNode = Node::CreateIntegerNode();
        defaultValueNode = Node::CreateIntegerNode();
        COMMAND_DO(
            currentValueNode.SetValue(
                *reinterpret_cast<const int32_t*>(currentValue)));
        COMMAND_DO(
            defaultValueNode.SetValue(
                *reinterpret_cast<const int32_t*>(defaultValue)));
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }

    COMMAND_DO(pNode->AppendMember(KeyCurrent, ::std::move(currentValueNode)));
    COMMAND_DO(pNode->AppendMember(KeyDefault, ::std::move(defaultValueNode)));

    return true;
}

bool ExportFirmwareDebugSettings(
    Node* pNode, const ::std::vector<::std::string>& names, bool needsFilter
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    auto itemCount = uint64_t();

    const ::nn::Result result =
        ::nn::settings::detail::GetKeyValueStoreItemCountForDebug(&itemCount);

    if (::nn::fs::ResultTargetNotFound::Includes(result))
    {
        return true;
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    ::std::unique_ptr<
        ::nn::settings::detail::KeyValueStoreItemForDebug[]> items(
            new ::nn::settings::detail::KeyValueStoreItemForDebug[
                static_cast<size_t>(itemCount)]());

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::settings::detail::GetKeyValueStoreItemForDebug(
            &itemCount, items.get(), static_cast<size_t>(itemCount)));

    typedef ::std::map<::std::string, Node*> Map;

    Map map;

    NN_UTIL_SCOPE_EXIT
    {
        for (Map::value_type pair : map)
        {
            delete pair.second;
        }
    };

    for (uint64_t i = 0; i < itemCount; ++i)
    {
        const auto& item = items[static_cast<size_t>(i)];

        const char* separator = ::std::strchr(item.mapKey, MapKeySeparator);

        ::std::string name(item.mapKey, separator - item.mapKey);

        Node* pRootNode = nullptr;

        const Map::iterator iter = map.find(name);

        if (iter != map.end())
        {
            pRootNode = iter->second;
        }
        else
        {
            pRootNode = new Node(::std::move(Node::CreateObjectNode()));

            map[name] = pRootNode;
        }

        NameScope nameScope(name);

        auto dictNode = Node::CreateObjectNode();

        ::std::string key(separator + 1);

        {
            NameScope subNameScope(key);

            COMMAND_DO(
                ExportFirmwareDebugSettings(
                    &dictNode, item.type,
                    item.currentValue, item.currentSize,
                    item.defaultValue, item.defaultSize));
        }

        COMMAND_DO(pRootNode->AppendMember(key, ::std::move(dictNode)));
    }

    for (Map::value_type pair : map)
    {
        const ::std::string& name = pair.first;

        if (!needsFilter || Contains(names, name)) {
            COMMAND_DO(pNode->AppendMember(name, ::std::move(*pair.second)));
        }
    }

    return true;
}

bool ImportFirmwareDebugSettings(
    const char* mapKey, uint8_t type,
    const void* currentValue, size_t currentSize,
    const void* defaultValue, size_t defaultSize) NN_NOEXCEPT
{
    ::nn::settings::detail::KeyValueStoreItemForDebug item = {};
    item.mapKey = mapKey;
    item.type = type;
    item.currentValue = currentValue;
    item.currentSize = currentSize;
    item.defaultValue = defaultValue;
    item.defaultSize = defaultSize;

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::settings::detail::AddKeyValueStoreItemForDebug(&item, 1));

    return true;
}

} // namespace
