﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <string>
#include <utility>
#include <vector>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SystemData.h>
#include <nn/ncm/ncm_SystemContentMetaId.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nn/util/util_ScopeExit.h>

#include "SettingsManager_FileSystem.h"
#include "SettingsManager_FirmwareDebugSettings.h"
#include "SettingsManager_NameScope.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Utility.h"

namespace {

//!< システムデータの識別子
const ::nn::ncm::SystemDataId SystemDataId = { 0x0100000000000818 };

//!< システムデータのマウント名
const char SystemDataMountName[] = "SystemData";

//!< システムデータのファイル名
const char SystemDataFileName[] = "file";

//!< 現在値のキー
const char* const KeyCurrent = "current";

//!< 初期値のキー
const char* const KeyDefault = "default";

//!< 設定項目の型を表す列挙体
enum class Type : uint8_t
{
    Unknown,    //!< 不明な型
    String,     //!< 文字列型
    Bool,       //!< 論理値型
    Int,        //!< 整数型
};

//!< システムデータ上のファイルを扱うクラスです。
class SystemDataFile final : public IFile
{
    NN_DISALLOW_COPY(SystemDataFile);
    NN_DISALLOW_MOVE(SystemDataFile);

private:
    //!< ファイルがオープンされているか否かを表す値
    bool m_IsOpen;

    //!< ファイルパス
    ::std::string m_Path;

    //!< ファイルハンドル
    ::nn::fs::FileHandle m_Handle;

public:
    SystemDataFile() NN_NOEXCEPT;

    virtual ~SystemDataFile() NN_NOEXCEPT NN_OVERRIDE;

    virtual const ::std::string& GetPath() const NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::fs::FileHandle GetHandle() const NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result Open(
        const ::std::string& path, int mode) NN_NOEXCEPT NN_OVERRIDE;

private:
    //!< ファイルをクローズします。
    void Close() NN_NOEXCEPT;
};

//!< ファイルの読み取りを扱うクラスです。
class FileReader final
{
    NN_DISALLOW_COPY(FileReader);
    NN_DISALLOW_MOVE(FileReader);

    //!< ファイル
    const IFile& m_File;

    //!< オフセット
    int64_t m_Offset;

public:
    explicit FileReader(const IFile& file) NN_NOEXCEPT;

    //!< 現在の読み取り開始位置を返します。
    int64_t GetOffset() const NN_NOEXCEPT;

    //!< ファイルの内容を取得します。
    template<typename T>
    bool Read(T* pOutValue) NN_NOEXCEPT
    {
        return this->Read(reinterpret_cast<char*>(pOutValue), sizeof(T));
    }

    //!< ファイルの内容を取得します。
    bool Read(char outBuffer[], size_t count) NN_NOEXCEPT;
};

//!< ファームウェアデバッグ設定をエクスポートします。
bool ExportFirmwareDebugSettings(
    Node* pNode, uint8_t type,
    char* currentValue, size_t currentSize,
    char* defaultValue, size_t defaultSize) NN_NOEXCEPT;

//!< ファームウェアデバッグ設定をエクスポートします。
bool ExportFirmwareDebugSettings(
    Node* pNode, const ::std::vector<::std::string>& names, bool needsFilter
    ) NN_NOEXCEPT;

} // namespace

bool ExportFirmwareDebugSettings(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    COMMAND_DO(
        ExportFirmwareDebugSettings(
            pNode, ::std::vector<::std::string>(), false));

    return true;
}

bool ExportFirmwareDebugSettings(
    Node* pNode, const ::std::vector<::std::string>& names) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    COMMAND_DO(ExportFirmwareDebugSettings(pNode, names, true));

    return true;
}

bool ImportFirmwareDebugSettings(
    const ::std::string& name, const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(name);

    ::std::vector<::std::string> settingNames;
    COMMAND_DO(node.GetKeys(&settingNames));

    for (const ::std::string& settingName : settingNames)
    {
        ::std::unique_ptr<const Node> pValueNode;
        COMMAND_DO(node.GetMember(&pValueNode, settingName));

        NameScope nameScope(settingName);

        ::std::unique_ptr<const Node> pCurrentValueNode;
        COMMAND_DO(pValueNode->GetMember(&pCurrentValueNode, KeyCurrent));

        if (pCurrentValueNode->IsStringNode())
        {
            ::std::string value;
            COMMAND_DO(pCurrentValueNode->GetValue(&value));
            ::nn::settings::fwdbg::SetSettingsItemValue(
                name.c_str(),
                settingName.c_str(), value.c_str(), value.size() + 1);
        }
        else if (pCurrentValueNode->IsBooleanNode())
        {
            auto value = bool();
            COMMAND_DO(pCurrentValueNode->GetValue(&value));
            ::nn::settings::fwdbg::SetSettingsItemValue(
                name.c_str(),
                settingName.c_str(), &value, sizeof(value));
        }
        else if (pCurrentValueNode->IsIntegerNode())
        {
            auto value = int32_t();
            COMMAND_DO(pCurrentValueNode->GetValue(&value));
            ::nn::settings::fwdbg::SetSettingsItemValue(
                name.c_str(),
                settingName.c_str(), &value, sizeof(value));
        }
    }

    return true;
}

bool FlushFirmwareDebugSettings() NN_NOEXCEPT
{
    return true;
}

namespace {

SystemDataFile::SystemDataFile() NN_NOEXCEPT
    : m_IsOpen(false)
    , m_Path()
    , m_Handle()
{
    // 何もしない
}

SystemDataFile::~SystemDataFile() NN_NOEXCEPT
{
    this->Close();
}

const ::std::string& SystemDataFile::GetPath() const NN_NOEXCEPT
{
    return m_Path;
}

::nn::fs::FileHandle SystemDataFile::GetHandle() const NN_NOEXCEPT
{
    return m_Handle;
}

::nn::Result SystemDataFile::Open(
    const ::std::string& path, int mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(mode & ::nn::fs::OpenMode_Write, 0);

    // オープン済みだった場合に備えてクローズ
    this->Close();

    auto needsRollback = true;

    NN_RESULT_DO(::nn::fs::MountSystemData(SystemDataMountName, SystemDataId));

    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            ::nn::fs::Unmount(SystemDataMountName);
        }
    };

    const ::std::string filePath =
        ::std::string(SystemDataMountName) + ":/" + path;

    NN_RESULT_DO(::nn::fs::OpenFile(&m_Handle, filePath.c_str(), mode));

    needsRollback = false;

    m_IsOpen = true;

    m_Path = filePath;

    NN_RESULT_SUCCESS;
}

void SystemDataFile::Close() NN_NOEXCEPT
{
    if (m_IsOpen)
    {
        ::nn::fs::CloseFile(m_Handle);

        ::nn::fs::Unmount(SystemDataMountName);

        m_IsOpen = false;

        m_Path.clear();

        m_Handle = ::nn::fs::FileHandle();
    }
}

FileReader::FileReader(const IFile& file) NN_NOEXCEPT
    : m_File(file)
    , m_Offset(0)
{
    // 何もしない
}

int64_t FileReader::GetOffset() const NN_NOEXCEPT
{
    return m_Offset;
}

bool FileReader::Read(char outBuffer[], size_t count) NN_NOEXCEPT
{
    auto size = size_t();

    COMMAND_DO(ReadFile(&size, outBuffer, count, m_File, m_Offset));

    NN_SDK_ASSERT_EQUAL(size, count);

    m_Offset += static_cast<int64_t>(size);

    return true;
}

bool ExportFirmwareDebugSettings(
    Node* pNode, uint8_t type,
    char* currentValue, size_t currentSize,
    char* defaultValue, size_t defaultSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);
    NN_SDK_REQUIRES_NOT_NULL(currentValue);
    NN_SDK_REQUIRES_NOT_NULL(defaultValue);

    switch (static_cast<Type>(type))
    {
    case Type::String:
        {
            auto currentValueNode = Node::CreateStringNode();
            COMMAND_DO(
                currentValueNode.SetValue(
                    ::std::string(currentValue, currentSize - 1)));
            COMMAND_DO(
                pNode->AppendMember(KeyCurrent, ::std::move(currentValueNode)));
            auto defaultValueNode = Node::CreateStringNode();
            COMMAND_DO(
                defaultValueNode.SetValue(
                    ::std::string(defaultValue, defaultSize - 1)));
            COMMAND_DO(
                pNode->AppendMember(KeyDefault, ::std::move(defaultValueNode)));
            break;
        }
    case Type::Bool:
        {
            auto currentValueNode = Node::CreateBooleanNode();
            COMMAND_DO(
                currentValueNode.SetValue(
                    *reinterpret_cast<bool*>(currentValue)));
            COMMAND_DO(
                pNode->AppendMember(KeyCurrent, ::std::move(currentValueNode)));
            auto defaultValueNode = Node::CreateBooleanNode();
            COMMAND_DO(
                defaultValueNode.SetValue(
                    *reinterpret_cast<bool*>(defaultValue)));
            COMMAND_DO(
                pNode->AppendMember(KeyDefault, ::std::move(defaultValueNode)));
            break;
        }
    case Type::Int:
        {
            auto currentValueNode = Node::CreateIntegerNode();
            COMMAND_DO(
                currentValueNode.SetValue(
                    *reinterpret_cast<int32_t*>(currentValue)));
            COMMAND_DO(
                pNode->AppendMember(KeyCurrent, ::std::move(currentValueNode)));
            auto defaultValueNode = Node::CreateIntegerNode();
            COMMAND_DO(
                defaultValueNode.SetValue(
                    *reinterpret_cast<int32_t*>(defaultValue)));
            COMMAND_DO(
                pNode->AppendMember(KeyDefault, ::std::move(defaultValueNode)));
            break;
        }
    default: NN_UNEXPECTED_DEFAULT;
    }

    return true;
}

bool ExportFirmwareDebugSettings(
    Node* pNode, const ::std::vector<::std::string>& names, bool needsFilter
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    SystemDataFile file;
    COMMAND_DO(OpenFile(&file, SystemDataFileName, ::nn::fs::OpenMode_Read));

    FileReader reader(file);

    auto totalSize = uint32_t();
    COMMAND_DO(reader.Read(&totalSize));

    auto dictNode = Node::CreateObjectNode();
    ::std::string lastName = "";

    while (reader.GetOffset() < totalSize)
    {
        auto mapKeySize = uint32_t();
        COMMAND_DO(reader.Read(&mapKeySize));

        ::std::unique_ptr<char[]> mapKey(new char[mapKeySize]());
        COMMAND_DO(reader.Read(mapKey.get(), mapKeySize));

        const size_t index =
            ::std::string(mapKey.get(), mapKeySize - 1).find("!");
        const ::std::string name(mapKey.get(), index);
        const ::std::string key(&(mapKey.get()[index + 1]));

        if (name != lastName)
        {
            if (!lastName.empty())
            {
                if (!needsFilter || Contains(names, lastName))
                {
                    COMMAND_DO(
                        pNode->AppendMember(lastName, ::std::move(dictNode)));
                }

                dictNode = Node::CreateObjectNode();
            }

            lastName = name;
        }

        NameScope nameScope(lastName);

        auto type = uint8_t();
        COMMAND_DO(reader.Read(&type));

        auto mapValueSize = uint32_t();
        COMMAND_DO(reader.Read(&mapValueSize));

        ::std::unique_ptr<char[]> mapValue(new char[mapValueSize]());
        COMMAND_DO(reader.Read(mapValue.get(), mapValueSize));

        const size_t bufferSize =
            ::nn::settings::fwdbg::GetSettingsItemValueSize(
                name.c_str(), key.c_str());
        ::std::unique_ptr<char[]> buffer(new char[bufferSize]());
        ::nn::settings::fwdbg::GetSettingsItemValue(
            buffer.get(), bufferSize, name.c_str(), key.c_str());

        auto valueNode = Node::CreateObjectNode();

        {
            NameScope nameScope(key);

            COMMAND_DO(
                ExportFirmwareDebugSettings(
                    &valueNode, type,
                    buffer.get(), bufferSize, mapValue.get(), mapValueSize));
        }

        COMMAND_DO(dictNode.AppendMember(key, ::std::move(valueNode)));
    }

    if (!lastName.empty())
    {
        if (!needsFilter || Contains(names, lastName))
        {
            COMMAND_DO(pNode->AppendMember(lastName, ::std::move(dictNode)));
        }
    }

    return true;
}

} // namespace
