﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <memory>
#include <string>
#include <utility>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/settings/system/settings_Eula.h>
#include <nn/settings/system/settings_Region.h>

#include "SettingsManager_ErrorCode.h"
#include "SettingsManager_Eula.h"
#include "SettingsManager_NameScope.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Utility.h"

namespace {

//!< 設定情報ヘッダのキー
const char* const SettingNameEulaVersions = "eula_versions";

//!< バージョンのキー
const char* const KeyVersion = "version";

//!< リージョンコードのキー
const char* const KeyRegionCode = "region_code";

//!< ネットワーク時刻のキー
const char* const KeyNetworkClockTimePoint = "network_clock_time_point";

//!< 単調増加クロックのキー
const char* const KeySteadyClockTimePoint = "steady_clock_time_point";

//!< 単調増加クロックのソース ID のキー
const char* const KeySteadyClockSourceId = "steady_clock_source_id";

//!< 日本リージョンを表すリージョンコード
const char* const RegionCodeJapan = "Japan";

//!< 米国リージョンを表すリージョンコード
const char* const RegionCodeUsa = "Usa";

//!< 欧州リージョンを表すリージョンコード
const char* const RegionCodeEurope = "Europe";

//!< 豪州リージョンを表すリージョンコード
const char* const RegionCodeAustralia = "Australia";

//!< バージョンをエクスポートします。
bool ExportVersion(
    Node* pDictNode,
    const ::nn::settings::system::EulaVersion& eula) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateStringNode();

    {
        NameScope nameScope(KeyVersion);

        ::std::string value;

        COMMAND_DO(EncodeInteger(&value, static_cast<int64_t>(eula.version)));

        COMMAND_DO(node.SetValue(value));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyVersion, ::std::move(node)));

    return true;
}

//!< バージョンをインポートします。
bool ImportVersion(
    ::nn::settings::system::EulaVersion* pOutValue,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyVersion));

    NameScope nameScope(KeyVersion);

    ::std::string value;
    COMMAND_DO(pNode->GetValue(&value));

    auto version = int64_t();
    if (!DecodeInteger(&version, value))
    {
        PrintErrorCode(ErrorCode::NodeValueInvalid, nameScope.Get(), value);

        return false;
    }

    pOutValue->version = static_cast<uint32_t>(version);

    return true;
}

//!< リージョンコードをエクスポートします。
bool ExportRegionCode(
    Node* pDictNode,
    const ::nn::settings::system::EulaVersion& eula) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateStringNode();

    {
        NameScope nameScope(KeyRegionCode);

        switch (eula.regionCode)
        {
        case ::nn::settings::system::RegionCode_Japan:
            COMMAND_DO(node.SetValue(RegionCodeJapan));
            break;

        case ::nn::settings::system::RegionCode_Usa:
            COMMAND_DO(node.SetValue(RegionCodeUsa));
            break;

        case ::nn::settings::system::RegionCode_Europe:
            COMMAND_DO(node.SetValue(RegionCodeEurope));
            break;

        case ::nn::settings::system::RegionCode_Australia:
            COMMAND_DO(node.SetValue(RegionCodeAustralia));
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    COMMAND_DO(pDictNode->AppendMember(KeyRegionCode, ::std::move(node)));

    return true;
}

//!< リージョンコードをインポートします。
bool ImportRegionCode(
    ::nn::settings::system::EulaVersion* pOutValue,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyRegionCode));

    NameScope nameScope(KeyRegionCode);

    ::std::string regionCode;
    COMMAND_DO(pNode->GetValue(&regionCode));

    if (regionCode == RegionCodeJapan)
    {
        pOutValue->regionCode =
            static_cast<int32_t>(::nn::settings::system::RegionCode_Japan);

        return true;
    }

    if (regionCode == RegionCodeUsa)
    {
        pOutValue->regionCode =
            static_cast<int32_t>(::nn::settings::system::RegionCode_Usa);

        return true;
    }

    if (regionCode == RegionCodeEurope)
    {
        pOutValue->regionCode =
            static_cast<int32_t>(::nn::settings::system::RegionCode_Europe);

        return true;
    }

    if (regionCode == RegionCodeAustralia)
    {
        pOutValue->regionCode =
            static_cast<int32_t>(::nn::settings::system::RegionCode_Australia);

        return true;
    }

    PrintErrorCode(ErrorCode::NodeValueInvalid, nameScope.Get(), regionCode);

    return false;
}

//!< ネットワーク時刻をエクスポートします。
bool ExportNetworkClockTimePoint(
    Node* pDictNode,
    const ::nn::settings::system::EulaVersion& eula) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateStringNode();

    {
        NameScope nameScope(KeyNetworkClockTimePoint);

        ::std::string value;

        COMMAND_DO(EncodeInteger(&value, eula.networkSystemClock.value));

        COMMAND_DO(node.SetValue(value));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyNetworkClockTimePoint, ::std::move(node)));

    return true;
}

//!< ネットワーク時刻をインポートします。
bool ImportNetworkClockTimePoint(
    ::nn::settings::system::EulaVersion* pOutValue,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyNetworkClockTimePoint));

    NameScope nameScope(KeyNetworkClockTimePoint);

    ::std::string value;
    COMMAND_DO(pNode->GetValue(&value));

    auto clock = int64_t();
    if (!DecodeInteger(&clock, value))
    {
        PrintErrorCode(ErrorCode::NodeValueInvalid, nameScope.Get(), value);

        return false;
    }

    pOutValue->networkSystemClock.value = clock;

    return true;
}

//!< 単調増加クロックをエクスポートします。
bool ExportSteadyClockTimePoint(
    Node* pDictNode,
    const ::nn::settings::system::EulaVersion& eula) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateStringNode();

    {
        NameScope nameScope(KeySteadyClockTimePoint);

        ::std::string value;

        COMMAND_DO(EncodeInteger(&value, eula.steadyClock.value));

        COMMAND_DO(node.SetValue(value));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeySteadyClockTimePoint, ::std::move(node)));

    return true;
}

//!< 単調増加クロックをインポートします。
bool ImportSteadyClockTimePoint(
    ::nn::settings::system::EulaVersion* pOutValue,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeySteadyClockTimePoint));

    NameScope nameScope(KeySteadyClockTimePoint);

    ::std::string value;
    COMMAND_DO(pNode->GetValue(&value));

    auto clock = int64_t();
    if (!DecodeInteger(&clock, value))
    {
        PrintErrorCode(ErrorCode::NodeValueInvalid, nameScope.Get(), value);

        return false;
    }

    pOutValue->steadyClock.value = clock;

    return true;
}

//!< 単調増加クロックのソース ID をエクスポートします。
bool ExportSteadyClockSourceId(
    Node* pDictNode,
    const ::nn::settings::system::EulaVersion& eula) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateStringNode();

    {
        NameScope nameScope(KeySteadyClockSourceId);

        ::std::string value;

        COMMAND_DO(EncodeUuid(&value, eula.steadyClock.sourceId));

        COMMAND_DO(node.SetValue(value));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeySteadyClockSourceId, ::std::move(node)));

    return true;
}

//!< 単調増加クロックをインポートします。
bool ImportSteadyClockSourceId(
    ::nn::settings::system::EulaVersion* pOutValue,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeySteadyClockSourceId));

    NameScope nameScope(KeySteadyClockSourceId);

    ::std::string value;
    COMMAND_DO(pNode->GetValue(&value));

    if (!DecodeUuid(&pOutValue->steadyClock.sourceId, value))
    {
        PrintErrorCode(ErrorCode::NodeValueInvalid, nameScope.Get(), value);

        return false;
    }

    return true;
}

} // namespace

//!< ネットワーク設定の設定名か否かを表す値を返します。
bool IsSettingNameEulaVersions(const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNameEulaVersions);
}

//!< ネットワーク設定をエクスポートします。
bool ExportEulaVersions(Node* pNode) NN_NOEXCEPT
{
    const int EntryCountMax = ::nn::settings::system::EulaVersionCountMax;

    ::std::unique_ptr<::nn::settings::system::EulaVersion[]> entries(
        new ::nn::settings::system::EulaVersion[EntryCountMax]());

    const int count = ::nn::settings::system::GetEulaVersions(
        entries.get(), EntryCountMax);

    auto arrayNode = Node::CreateArrayNode();

    for (int i = 0; i < count; ++i)
    {
        NameScope nameScope(SettingNameEulaVersions);

        const ::nn::settings::system::EulaVersion& eula = entries[i];

        auto dictNode = Node::CreateObjectNode();

        COMMAND_DO(ExportVersion(&dictNode, eula));

        COMMAND_DO(ExportRegionCode(&dictNode, eula));

        switch (eula.clockType)
        {
        case ::nn::settings::system::EulaVersionClockType_NetworkSystemClock:
            COMMAND_DO(ExportNetworkClockTimePoint(&dictNode, eula));

            break;

        case ::nn::settings::system::EulaVersionClockType_SteadyClock:
            COMMAND_DO(ExportSteadyClockTimePoint(&dictNode, eula));

            COMMAND_DO(ExportSteadyClockSourceId(&dictNode, eula));

            break;

        default: NN_UNEXPECTED_DEFAULT;
        }

        COMMAND_DO(arrayNode.AppendElement(::std::move(dictNode)));
    }

    COMMAND_DO(
        pNode->AppendMember(SettingNameEulaVersions, ::std::move(arrayNode)));

    return true;
}

//!< ネットワーク設定をインポートします。
bool ImportEulaVersions(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNameEulaVersions);

    const int EntryCountMax = ::nn::settings::system::EulaVersionCountMax;

    ::std::unique_ptr<::nn::settings::system::EulaVersion[]> entries(
        new ::nn::settings::system::EulaVersion[EntryCountMax]());

    int count = 0;

    auto elementCount = size_t();

    COMMAND_DO(node.GetElementCount(&elementCount));

    for (size_t i = 0; i < elementCount; ++i)
    {
        if (count >= EntryCountMax)
        {
            break;
        }

        ::nn::settings::system::EulaVersion& eula = entries[count];

        ::std::unique_ptr<const Node> pNode;
        COMMAND_DO(node.GetElement(&pNode, i));

        ::std::vector<::std::string> keys;
        COMMAND_DO(pNode->GetKeys(&keys));

        bool usesNetworkClock = (::std::find(
            keys.begin(), keys.end(), KeyNetworkClockTimePoint) != keys.end());

        COMMAND_DO(ImportVersion(&eula, *pNode));

        COMMAND_DO(ImportRegionCode(&eula, *pNode));

        if (usesNetworkClock)
        {
            COMMAND_DO(ImportNetworkClockTimePoint(&eula, *pNode));

            eula.clockType =static_cast<int32_t>(
                ::nn::settings::system::
                    EulaVersionClockType_NetworkSystemClock);
        }
        else
        {
            COMMAND_DO(ImportSteadyClockTimePoint(&eula, *pNode));

            COMMAND_DO(ImportSteadyClockSourceId(&eula, *pNode));

            eula.clockType =static_cast<int32_t>(
                ::nn::settings::system::
                    EulaVersionClockType_SteadyClock);
        }

        ++count;
    }

    ::nn::settings::system::SetEulaVersions(entries.get(), count);

    return true;
}
