﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <string>
#include <utility>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/settings/system/settings_Capture.h>

#include "SettingsManager_Capture.h"
#include "SettingsManager_ErrorCode.h"
#include "SettingsManager_NameScope.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Utility.h"

namespace {

//!< キャプチャ設定のキー
const char* const SettingNameCaptureSettings = "capture_settings";

//!< 現在値のキー
const char* const KeyCurrent = "current";

//!< 初期値のキー
const char* const KeyDefault = "default";

//!< 選択肢のキー
const char* const KeyChoices = "choices";

//!< 撮影した画面写真を保存するストレージのキー
const char* const KeyPrimaryAlbumStorage = "primary_album_storage";

//!< NAND を表すストレージ設定
const char* const PrimaryAlbumStorageNand = "Nand";

//!< SD カードを表すストレージ設定
const char* const PrimaryAlbumStorageSdCard = "SdCard";

//!< 撮影した画面写真を保存するストレージをエクスポートします。
bool ExportPrimaryAlbumStorage(Node* pDictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeyPrimaryAlbumStorage);

        auto currentValueNode = Node::CreateStringNode();

        switch (::nn::settings::system::GetPrimaryAlbumStorage())
        {
        case ::nn::settings::system::PrimaryAlbumStorage_Nand:
            COMMAND_DO(currentValueNode.SetValue(PrimaryAlbumStorageNand));
            break;

        case ::nn::settings::system::PrimaryAlbumStorage_SdCard:
            COMMAND_DO(currentValueNode.SetValue(PrimaryAlbumStorageSdCard));
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateStringNode();

        COMMAND_DO(defaultValueNode.SetValue(PrimaryAlbumStorageSdCard));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));

        auto choicesValueNode = Node::CreateArrayNode();

        const char* const choices[] =
        {
            PrimaryAlbumStorageNand,
            PrimaryAlbumStorageSdCard,
        };

        for (const char* const choice : choices)
        {
            auto choiceValueNode = Node::CreateStringNode();

            COMMAND_DO(choiceValueNode.SetValue(choice));

            COMMAND_DO(
                choicesValueNode.AppendElement(::std::move(choiceValueNode)));
        }

        COMMAND_DO(
            node.AppendMember(KeyChoices, ::std::move(choicesValueNode)));
    }

    COMMAND_DO(
        pDictNode->AppendMember(KeyPrimaryAlbumStorage, ::std::move(node)));

    return true;
}

//!< 撮影した画面写真を保存するストレージをインポートします。
bool ImportPrimaryAlbumStorage(const Node& dictNode) NN_NOEXCEPT
{
    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeyPrimaryAlbumStorage));

    NameScope nameScope(KeyPrimaryAlbumStorage);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    ::std::string storage;
    COMMAND_DO(pCurrentValueNode->GetValue(&storage));

    if (storage == PrimaryAlbumStorageNand)
    {
        ::nn::settings::system::SetPrimaryAlbumStorage(
            ::nn::settings::system::PrimaryAlbumStorage_Nand);

        return true;
    }

    if (storage == PrimaryAlbumStorageSdCard)
    {
        ::nn::settings::system::SetPrimaryAlbumStorage(
            ::nn::settings::system::PrimaryAlbumStorage_SdCard);

        return true;
    }

    PrintErrorCode(ErrorCode::NodeValueInvalid, nameScope.Get(), storage);

    return false;
}

} // namespace

bool IsSettingNameCaptureSettings(const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNameCaptureSettings);
}

bool ExportCaptureSettings(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(SettingNameCaptureSettings);

        COMMAND_DO(ExportPrimaryAlbumStorage(&node));
    }

    COMMAND_DO(
        pNode->AppendMember(SettingNameCaptureSettings, ::std::move(node)));

    return true;
}

bool ImportCaptureSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNameCaptureSettings);

    COMMAND_DO(ImportPrimaryAlbumStorage(node));

    return true;
}
