﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <string>
#include <utility>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/settings/system/settings_Account.h>

#include "SettingsManager_Account.h"
#include "SettingsManager_NameScope.h"
#include "SettingsManager_RapidJson.h"
#include "SettingsManager_Utility.h"

namespace {

//!< アカウント設定のキー
const char* const SettingNameAccountSettings = "account_settings";

//!< 現在値のキー
const char* const KeyCurrent = "current";

//!< 初期値のキー
const char* const KeyDefault = "default";

//!< アカウントが一つの場合にアカウント選択を省略するか否かを表す値のキー
const char* const KeySkipFlag = "skips_if_single_user";

//!< アカウントが一つの場合にアカウント選択を省略するか否かを表す値をエクスポートします。
bool ExportSkipFlag(
    Node* pDictNode,
    const ::nn::settings::system::AccountSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDictNode);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(KeySkipFlag);

        auto currentValueNode = Node::CreateBooleanNode();

        COMMAND_DO(
            currentValueNode.SetValue(
                settings.userSelectorSettings.flags.Test<
                    ::nn::settings::system::UserSelectorFlag::SkipsIfSingleUser
                    >()));

        COMMAND_DO(
            node.AppendMember(KeyCurrent, ::std::move(currentValueNode)));

        auto defaultValueNode = Node::CreateBooleanNode();

        COMMAND_DO(defaultValueNode.SetValue(false));

        COMMAND_DO(
            node.AppendMember(KeyDefault, ::std::move(defaultValueNode)));
    }

    COMMAND_DO(pDictNode->AppendMember(KeySkipFlag, ::std::move(node)));

    return true;
}

//!< アカウントが一つの場合にアカウント選択を省略するか否かを表す値をインポートします。
bool ImportSkipFlag(
    ::nn::settings::system::AccountSettings* pOutValue,
    const Node& dictNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::unique_ptr<const Node> pNode;
    COMMAND_DO(dictNode.GetMember(&pNode, KeySkipFlag));

    NameScope nameScope(KeySkipFlag);

    ::std::unique_ptr<const Node> pCurrentValueNode;
    COMMAND_DO(pNode->GetMember(&pCurrentValueNode, KeyCurrent));

    auto skipFlag = bool();
    COMMAND_DO(pCurrentValueNode->GetValue(&skipFlag));

    pOutValue->userSelectorSettings.flags.Set<
        ::nn::settings::system::UserSelectorFlag::SkipsIfSingleUser>(skipFlag);

    return true;
}

} // namespace

bool IsSettingNameAccountSettings(const ::std::string& value) NN_NOEXCEPT
{
    return (value == SettingNameAccountSettings);
}

bool ExportAccountSettings(Node* pNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNode);

    ::nn::settings::system::AccountSettings settings = {};

    ::nn::settings::system::GetAccountSettings(&settings);

    auto node = Node::CreateObjectNode();

    {
        NameScope nameScope(SettingNameAccountSettings);

        COMMAND_DO(ExportSkipFlag(&node, settings));
    }

    COMMAND_DO(
        pNode->AppendMember(SettingNameAccountSettings, ::std::move(node)));

    return true;
}

bool ImportAccountSettings(const Node& node) NN_NOEXCEPT
{
    NameScope nameScope(SettingNameAccountSettings);

    ::nn::settings::system::AccountSettings settings = {};

    COMMAND_DO(ImportSkipFlag(&settings, node));

    ::nn::settings::system::SetAccountSettings(settings);

    return true;
}
