﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/socket.h>

namespace devmenuUtil { namespace network {

/**
 * @brief   Ping の処理結果です。
 *
 * @details
 */
enum PingResult
{
    PingResult_Success             = 0, //!< 成功
    PingResult_TimedOut            = 1, //!< タイムアウトが発生した
    PingResult_InitializationError = 2, //!< 初期化エラー
    PingResult_SendError           = 3, //!< 通信エラー（送信時）
    PingResult_ReceiveError        = 4, //!< 通信エラー（受信時）
    PingResult_InvalidResponseSize = 5, //!< 不正なレスポンスサイズ
    PingResult_InvalidIpHeader     = 6, //!< 不正な IP ヘッダ
    PingResult_InvalidIcmpHeader   = 7, //!< 不正な ICMP ヘッダ
    PingResult_UnhandledIcmpType   = 8, //!< ハンドルしていない ICMP タイプ
};

/**
 * @brief   ホスト名です。
 *
 * @details
 */
struct HostName
{
    char value[512];
};

}}

namespace devmenuUtil { namespace network {

/**
 * @brief   ホスト名または IP アドレス文字列から IP アドレスを取得します。
 *
 * @param[out]  pOut    IP アドレス
 * @param[in]   pName   ホスト名または IP アドレス文字列
 *
 * @return  成功したかどうか
 *
 * @pre
 *  - pOut != nullptr
 *  - pName != nullptr
 *
 * @details
 */
bool GetIpAddress(nn::socket::InAddr* pOut, const char* pName) NN_NOEXCEPT;

/**
 * @brief   IP アドレスからホスト名を取得します。
 *
 * @param[out]  pOut    ホスト名
 * @param[in]   addr    IP アドレス
 *
 * @return  成功したかどうか
 *
 * @pre
 *  - pOut != nullptr
 *
 * @details
 */
bool GetHostName(HostName* pOut, const nn::socket::InAddr& addr) NN_NOEXCEPT;

/**
 * @brief   Ping を打ちます。
 *
 * @param[out]  pOutIsReached       宛先に届いたかどうか
 * @param[out]  pOutReplySourceAddr 応答元の IP アドレス
 * @param[in]   targetAddr          宛先の IP アドレス
 * @param[in]   numHops             ホップ数
 * @param[in]   receiveTimeout      受信タイムアウト
 *
 * @return  処理結果
 *
 * @pre
 *  - pOutIsReached != nullptr
 *  - pOutReplySourceAddr != nullptr
 *  - numHops > 0
 *
 * @details
 */
PingResult Ping(bool* pOutIsReached, nn::socket::InAddr* pOutReplySourceAddr,
    const nn::socket::InAddr& targetAddr, int32_t numHops, nn::TimeSpan receiveTimeout) NN_NOEXCEPT;

/**
 * @brief   PingResult の文字列を取得します。
 *
 * @param[in]   result  PingResult
 *
 * @return  文字列
 *
 * @details
 */
const char* GetPingResultString(PingResult result) NN_NOEXCEPT;

}}
