﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "DevMenuCommand_CommandTypes.h"

#include <utility>
#include <string>
#include <type_traits>

namespace nn { namespace devmenucommand {

template <typename T, typename Enabled = void>
struct DefaultSerializer
{
    static ProcessResult Deserialize(T* pOut, const std::string& s, const std::string& valueName) NN_NOEXCEPT
    {
        NN_UNUSED(pOut);
        NN_UNUSED(s);
        return MakeProcessResultFailure("internal error: Serializer not set for " + valueName);
    }
    static std::string Serialize(const T& v) NN_NOEXCEPT
    {
        return "### invalid value ###";
    }
    static std::string GetArgumentHelpMessage() NN_NOEXCEPT
    {
        return "";
    }
};

struct BoolDefaultSerializer
{
    static ProcessResult Deserialize(bool* pOut, const std::string& s, const std::string& valueName) NN_NOEXCEPT;
    static std::string Serialize(const bool& v) NN_NOEXCEPT;
    static std::string GetArgumentHelpMessage() NN_NOEXCEPT;
};

template <>
struct DefaultSerializer<bool>
    : public BoolDefaultSerializer {};

template <typename Int>
struct IntDefaultSerializer
{
    static ProcessResult Deserialize(Int* pOut, const std::string& s, const std::string& valueName) NN_NOEXCEPT;
    static std::string Serialize(const Int& v) NN_NOEXCEPT;
    static std::string GetArgumentHelpMessage() NN_NOEXCEPT;
};

template <typename Int>
struct DefaultSerializer<Int, typename std::enable_if<std::is_integral<Int>::value>::type>
    : public IntDefaultSerializer<Int> {};

template <>
struct DefaultSerializer<std::string>
{
    static ProcessResult Deserialize(std::string* pOut, const std::string& s, const std::string&) NN_NOEXCEPT
    {
        *pOut = s;
        return ProcessResultSuccess;
    }
    static std::string Serialize(const std::string& v) NN_NOEXCEPT
    {
        return v;
    }
    static std::string GetArgumentHelpMessage() NN_NOEXCEPT
    {
        return "<string>";
    }
};

}}
