﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenuCommand_ValueSerializers.h"

#include <nn/nn_Common.h>
#include <cctype>
#include <nn/util/util_IntUtil.h>
#include <nn/nn_SdkAssert.h>
#include <cstdlib>
#include "DevMenuCommand_CommandImplUtility.h"

namespace nn { namespace devmenucommand {

ProcessResult BoolDefaultSerializer::Deserialize(bool* pOut, const std::string& s, const std::string& valueName) NN_NOEXCEPT
{
    if (s.size() == 0)
    {
        return MakeProcessResultFailure(valueName + " needs argument: true | false");
    }
    auto isTrue = false
        || s == "true"
        || s == "TRUE"
        || s == "True"
        || s == "T"
        || s == "enabled"
        || s == "enable"
    ;
    auto isFalse = false
        || s == "false"
        || s == "FALSE"
        || s == "False"
        || s == "F"
        || s == "disabled"
        || s == "disable"
    ;
    if (!(isTrue || isFalse))
    {
        return MakeProcessResultFailure(valueName + ": invalid bool argument '" + s + "'");
    }
    if (isTrue)
    {
        *pOut = true;
    }
    else
    {
        NN_SDK_ASSERT(isFalse);
        *pOut = false;
    }
    return ProcessResultSuccess;
}

std::string BoolDefaultSerializer::Serialize(const bool& v) NN_NOEXCEPT
{
    return v ? "true" : "false";
}

std::string BoolDefaultSerializer::GetArgumentHelpMessage() NN_NOEXCEPT
{
    return "<true|false>";
}

template <typename Int>
ProcessResult IntDefaultSerializer<Int>::Deserialize(Int* pOut, const std::string& s, const std::string& valueName) NN_NOEXCEPT
{
    auto&& t = [&]() -> std::pair<std::string, int>
    {
        static const std::pair<const char*, int> Table[] =
        {
            {"Gi", 1024 * 1024 * 1024},
            {"Mi", 1024 * 1024},
            {"Ki", 1024},
            {"G", 1000 * 1000 * 1000},
            {"M", 1000 * 1000},
            {"K", 1000},
        };
        for (auto&& e: Table)
        {
            if (auto pPrefix = RemoveSuffix(s, e.first))
            {
                return {std::move(*pPrefix), e.second};
            }
        }
        return {s, 1};
    }();
    auto&& target = t.first;
    auto&& multiple = t.second;
    char* pEnd;
    std::intmax_t longValue = std::strtol(target.c_str(), &pEnd, 0);
    if (!(target.size() > 0 && *pEnd == '\0'))
    {
        return MakeProcessResultFailure(valueName + ": invalid integer argument '" + s + "'");
    }
    if (!util::IsIntValueRepresentable<Int>(longValue))
    {
        return MakeProcessResultFailure(valueName + ": out of range '" + s + "'");
    }
    longValue *= multiple;
    if (!util::IsIntValueRepresentable<Int>(longValue))
    {
        return MakeProcessResultFailure(valueName + ": out of range '" + s + "'");
    }

    *pOut = static_cast<Int>(longValue);
    return ProcessResultSuccess;
}

template <typename Int>
std::string IntDefaultSerializer<Int>::Serialize(const Int& v) NN_NOEXCEPT
{
    return std::to_string(v);
}

template <typename Int>
std::string IntDefaultSerializer<Int>::GetArgumentHelpMessage() NN_NOEXCEPT
{
    return "<integer>";
}

template struct IntDefaultSerializer<char>;
template struct IntDefaultSerializer<unsigned char>;
template struct IntDefaultSerializer<signed char>;
template struct IntDefaultSerializer<int>;
template struct IntDefaultSerializer<long>;
template struct IntDefaultSerializer<long long>;
template struct IntDefaultSerializer<unsigned int>;
template struct IntDefaultSerializer<unsigned long>;
template struct IntDefaultSerializer<unsigned long long>;

}}
