﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <string>
#include <functional>
#include <vector>
#include <utility>
#include <nn/util/util_Optional.h>
#include "../DevMenuCommand_Option.h"

namespace nn { namespace devmenucommand {

enum class Visibility
{
    Public,
    Hidden,
    Private,
    HiddenPrivate,
    None,
};

enum class ReflectionTime
{
    Instant,
    NeedsRelaunchApplication,
    NeedsRebootSystem,
};

struct DeprecatedT
{
    bool _value;
};

struct SubCommand
{
    std::string name;
    std::function<Result(bool* outValue, const Option&)> function;
    Visibility visibility;
    bool isDeprecated;
    std::string argumentHelpMessage;
};

template <typename T>
struct ProcessResultT
{
    util::optional<T> p;
    std::string message;

    ProcessResultT() = default;

    NN_IMPLICIT ProcessResultT(const T& x)
        : p(x)
    {
    }

    static ProcessResultT<T> MakeSuccess(const T& x, const std::string& message = "")
    {
        ProcessResultT<T> ret;
        ret.p = x;
        ret.message = message;
        return ret;
    }

    static ProcessResultT<T> MakeFailure(const std::string& message)
    {
        ProcessResultT<T> ret;
        ret.message = message;
        return ret;
    }

    NN_EXPLICIT_OPERATOR bool() const
    {
        return p;
    }

    bool operator!() const
    {
        return !static_cast<bool>(*this);
    }

    template <typename U>
    NN_IMPLICIT operator ProcessResultT<U>() const
    {
        NN_ASSERT(!p);
        return ProcessResultT<U>::MakeFailure(message);
    }

    T& operator*()
    {
        return *p;
    }

    const T& operator*() const
    {
        return *p;
    }

};

namespace detail {

struct MonoState
{
};

}

using ProcessResult = ProcessResultT<detail::MonoState>;

const ProcessResult ProcessResultSuccess{detail::MonoState{}};

inline ProcessResult MakeProcessResultFailure(const std::string& message = "error") NN_NOEXCEPT
{
    return ProcessResult::MakeFailure(message);
}

inline bool NeedsHelpMessage(const SubCommand& x, bool forPrivate) NN_NOEXCEPT
{
    if (x.isDeprecated)
    {
        return false;
    }
    switch (x.visibility)
    {
        case Visibility::Public: return true;
        case Visibility::Hidden: return forPrivate;
        case Visibility::Private: return forPrivate;
        case Visibility::HiddenPrivate: return false;
        case Visibility::None: return false;
        default: NN_UNEXPECTED_DEFAULT;
    }
}

inline bool IsAvailable(const SubCommand& x, bool forPrivate) NN_NOEXCEPT
{
    switch (x.visibility)
    {
        case Visibility::Public: return true;
        case Visibility::Hidden: return true;
        case Visibility::Private: return forPrivate;
        case Visibility::HiddenPrivate: return forPrivate;
        case Visibility::None: return false;
        default: NN_UNEXPECTED_DEFAULT;
    }
}

class ICommandBuilder
{
public:
    virtual std::vector<SubCommand> MakeSubCommands() const NN_NOEXCEPT = 0;
    virtual ~ICommandBuilder() = default;
};

template <typename T>
inline std::vector<SubCommand> AggregateSubCommands(T&& builders) NN_NOEXCEPT
{
    std::vector<SubCommand> ret;
    for (auto&& pBuilder : builders)
    {
        for (auto&& e : pBuilder->MakeSubCommands())
        {
            ret.push_back(e);
        }
    }
    return ret;
}

namespace abbreviation {

/**
    @brief 公開されており使用できることを表す。
*/
const auto Public = Visibility::Public;

/**
    @brief 使用することはできるが、ヘルプには表示されていない隠し機能であることを表す。
*/
const auto Hidden = Visibility::Hidden;

/**
    @brief 非公開だが、内部用ツールではヘルプにも表示され使用することを表す。
*/
const auto Private = Visibility::Private;

/**
    @brief 非公開で、内部用ツールでは使用することができるが、ヘルプには表示されていない隠し機能であることを表す。
*/
const auto HiddenPrivate = Visibility::HiddenPrivate;

/**
    @brief 値の設定後に(起動していれば)アプリケーションの再起動が必要となることを表す。
*/
const auto NeedsRelaunchApplication = ReflectionTime::NeedsRelaunchApplication;

/**
    @brief 値の設定後にシステムの再起動が必要となることを表す。
*/
const auto NeedsRebootSystem = ReflectionTime::NeedsRebootSystem;

/**
    @brief 互換性のために残されており、ヘルプに表示されないことを表す。
*/
const auto Deprecated = DeprecatedT{true};

}

}}
