﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/util/util_Optional.h>
#include <utility>
#include <tuple>
#include <algorithm>
#include <string>
#include <type_traits>
#include <sstream>

namespace nn { namespace devmenucommand {

namespace detail {

template <class F, class Tuple, std::size_t... I>
inline decltype(auto) ApplyTupleImpl(F&& f, Tuple&& t, std::index_sequence<I...>) NN_NOEXCEPT
{
    return (std::forward<F>(f))(std::get<I>(std::forward<Tuple>(t))...);
}

}

// std::apply() と同等
template <class F, class Tuple>
inline decltype(auto) ApplyTuple(F&& f, Tuple&& t) NN_NOEXCEPT
{
    return detail::ApplyTupleImpl(std::forward<F>(f), std::forward<Tuple>(t), std::make_index_sequence<std::tuple_size<std::remove_reference_t<Tuple>>::value>{});
}

template <typename F>
void ForeachArgs(F&)
{
}

template <typename F, typename Head, typename... Tail>
void ForeachArgs(F& f, Head&& head, Tail&&... tail)
{
    f(std::forward<Head>(head));
    ForeachArgs(f, std::forward<Tail>(tail)...);
}

// args の各要素 arg に対し f(arg) を呼び出す。
// ただし、arg が std::tuple<TArgs...>{targs...} だった場合には、再帰的に ApplyRecursively(f, targs...) を呼び出す。
template <typename F, typename... Args>
inline void ApplyRecursively(F&& f, Args&&... args) NN_NOEXCEPT;

namespace detail {

template <typename F, typename T>
inline void ApplyRecursivelyOne(F&& f, T&& arg) NN_NOEXCEPT
{
    std::forward<F>(f)(std::forward<T>(arg));
}

template <typename F, typename Tuple, std::size_t... I>
inline void ApplyRecursivelyForTuple(F&& f, Tuple&& tuple, std::index_sequence<I...>) NN_NOEXCEPT
{
    ApplyRecursively(std::forward<F>(f), std::get<I>(std::forward<Tuple>(tuple))...);
}

template <typename F, typename... Args>
inline void ApplyRecursivelyOne(F&& f, std::tuple<Args...>&& tuple) NN_NOEXCEPT
{
    ApplyRecursivelyForTuple(std::forward<F>(f), std::forward<std::tuple<Args...>>(tuple), std::make_index_sequence<sizeof...(Args)>{});
}

template <typename F, typename... Args>
inline void ApplyRecursivelyOne(F&& f, const std::tuple<Args...>& tuple) NN_NOEXCEPT
{
    ApplyRecursivelyForTuple(std::forward<F>(f), tuple, std::make_index_sequence<sizeof...(Args)>{});
}

} // detail

template <typename F>
inline void ApplyRecursively(F&&) NN_NOEXCEPT
{
    // nop
}

template <typename F, typename Arg, typename... Args>
inline void ApplyRecursively(F&& f, Arg&& arg, Args&&... args) NN_NOEXCEPT
{
    detail::ApplyRecursivelyOne(f, std::forward<Arg>(arg));
    ApplyRecursively(f, std::forward<Args>(args)...);
}

// s が suffix で終わっているかどうかを調べ、
// そうであれば suffix を取り除いた文字列を返し、そうでなければ nullopt を返す。
inline util::optional<std::string> RemoveSuffix(const std::string& s, const std::string& suffix) NN_NOEXCEPT
{
    if (!(s.size() > suffix.size() && std::equal(std::rbegin(suffix), std::rend(suffix), std::rbegin(s))))
    {
        return util::nullopt;
    }
    return s.substr(0, s.size() - suffix.size());
}

template <typename T>
inline typename std::enable_if<std::is_integral<T>::value, std::string>::type ToString(const T& x) NN_NOEXCEPT
{
    return std::to_string(x);
}

template <typename T>
inline typename std::enable_if<std::is_enum<T>::value, std::string>::type ToString(const T& x) NN_NOEXCEPT
{
    return ToString(static_cast<typename std::underlying_type<T>::type>(x));
}

inline std::string ToString(const char* s) NN_NOEXCEPT
{
    return s;
}

inline std::string ToString(const std::string& s) NN_NOEXCEPT
{
    return s;
}

template <typename Iterator, typename Convert>
inline std::string JoinString(Iterator begin, Iterator end, const char* separator, Convert&& convert) NN_NOEXCEPT
{
    std::string s;
    auto first = true;
    for (auto it = begin; it != end; ++it)
    {
        if (!std::exchange(first, false))
        {
            s.append(separator);
        }
        s.append(ToString(convert(*it)));
    }
    return s;
}

template <typename Iterator>
inline std::string JoinString(Iterator begin, Iterator end, const char* separator) NN_NOEXCEPT
{
    return JoinString(begin ,end, separator, [](auto&& x){ return x; });
}

inline std::vector<std::string> SplitString(const std::string& s, char separator) NN_NOEXCEPT
{
    std::vector<std::string> v;
    std::istringstream stream{s};
    std::string buffer;
    while (std::getline(stream, buffer, separator))
    {
        v.push_back(buffer);
    }
    return v;
}

template <typename T, typename... Properties>
inline void SetProperties(T& x, Properties&&... properties)
{
    ApplyRecursively([&](auto&& f)
    {
        x.SetProperty(f);
    }, std::forward<Properties>(properties)...);
}

template <typename T, typename Tag>
struct Wrapper
{
    T value;
};

template <typename Tag, typename T>
inline auto Wrap(T&& x)
{
    return Wrapper<std::decay_t<T>, Tag>{std::forward<T>(x)};
}

}}
