﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenuCommandAlbum_ReduceImage.h"

#include <algorithm>

namespace album {

namespace {

    std::vector<float> ReduceHorizontal(
        const std::vector<uint8_t>& srcData,
        int channels,
        int srcWidth,
        int srcHeight,
        int dstWidth
    ) NN_NOEXCEPT
    {
        std::vector<float> dstData;
        dstData.assign(channels * dstWidth * srcHeight, 0);
        if (srcWidth > dstWidth)
        {
            std::vector<int> work;
            for (int y = 0; y < srcHeight; y++)
            {
                work.assign(dstWidth, 0);
                const uint8_t*const pSrcLine = srcData.data() + channels * srcWidth * y;
                float*const         pDstLine = dstData.data() + channels * dstWidth * y;
                for (int srcX = 0; srcX < srcWidth; srcX++)
                {
                    int dstX = static_cast<int>(static_cast<float>(srcX) * static_cast<float>(dstWidth - 1) / static_cast<float>(srcWidth - 1));
                    dstX = std::min(dstX, dstWidth - 1);
                    dstX = std::max(dstX, 0);
                    const uint8_t* p = pSrcLine + channels * srcX;
                    float*         q = pDstLine + channels * dstX;
                    for (int c = 0; c < channels; c++)
                    {
                        q[c] += p[c];
                    }
                    work[dstX]++;
                }
                for (int dstX = 0; dstX < dstWidth; dstX++)
                {
                    float* q = pDstLine + channels * dstX;
                    int    n = work[dstX];
                    for (int c = 0; c < channels; c++)
                    {
                        q[c] /= static_cast<float>(n);
                        //q[c] = std::min(q[c], 255.f);
                        //q[c] = std::max(q[c], 0.f);
                    }
                }
            }
        }
        else
        {
            // src の方が小さいときは Dot by Dot コピー
            for (int y = 0; y < srcHeight; y++)
            {
                const uint8_t* p = srcData.data() + channels * srcWidth * y;
                float*         q = dstData.data() + channels * dstWidth * y;
                for (int i = 0; i < channels * srcWidth; i++)
                {
                    *(q++) = static_cast<float>(*(p++));
                }
                for (int i = channels * srcWidth; i < channels * dstWidth; i++)
                {
                    *(q++) = 255.f;
                }
            }
        }

        return dstData;
    }

    std::vector<float> ReduceVertical(
        const std::vector<float>& srcData,
        int channels,
        int srcHeight,
        int dstWidth,
        int dstHeight
    ) NN_NOEXCEPT
    {
        std::vector<float> dstData;
        dstData.resize(channels * dstWidth * dstHeight, 0);

        int lineElemCount = channels * dstWidth;

        if (srcHeight > dstHeight)
        {
            std::vector<int> work;
            work.resize(dstHeight, 0);
            for(int srcY = 0; srcY < srcHeight; srcY++)
            {
                int dstY = static_cast<int>(static_cast<float>(srcY) * static_cast<float>(dstHeight - 1) / static_cast<float>(srcHeight - 1));
                dstY = std::min(dstY, dstHeight - 1);
                dstY = std::max(dstY, 0);
                const float* p = srcData.data() + lineElemCount * srcY;
                float*       q = dstData.data() + lineElemCount * dstY;
                for (int i = 0; i < lineElemCount; i++)
                {
                    *(q++) += *(p++);
                }
                work[dstY]++;
            }
            for (int dstY = 0; dstY < dstHeight; dstY++)
            {
                float n = static_cast<float>(work[dstY]);
                float* q = dstData.data() + lineElemCount * dstY;
                for(int i = 0; i < lineElemCount; i++)
                {
                    q[i] /= n;
                    //q[i] = std::min(q[i], 255.f);
                    //q[i] = std::max(q[i], 0.f);
                }
            }
        }
        else
        {
            // src の方が小さいときは Dot by Dot コピー
            for (int y = 0; y < srcHeight; y++)
            {
                const float* p = srcData.data() + lineElemCount * y;
                float*       q = dstData.data() + lineElemCount * y;
                std::memcpy(q, p, channels * dstWidth * sizeof(float));
            }
            for (int y = srcHeight; y < dstHeight; y++)
            {
                float*       q = dstData.data() + lineElemCount * y;
                std::memset(q, 0, lineElemCount * sizeof(float));
            }
        }

        return dstData;
    }

}   // namespace

std::vector<uint8_t> ReduceImage(
    const std::vector<uint8_t>& srcData,
    int channels,
    int srcWidth,
    int srcHeight,
    int dstWidth,
    int dstHeight
) NN_NOEXCEPT
{
    // sWxsH -> dWxsH
    std::vector<float> img1 = ReduceHorizontal(srcData, channels, srcWidth, srcHeight, dstWidth);

    // dWxsH -> sWxdH
    std::vector<float> img2 = ReduceVertical(img1, channels, srcHeight, dstWidth, dstHeight);

    std::vector<uint8_t> img;
    img.resize(channels * dstWidth * dstHeight);
    for (int i = 0; i < channels * dstWidth * dstHeight; i++)
    {
        float value = img2[i];
        value = std::min(value, 255.f);
        value = std::max(value, 0.f);
        img[i] = static_cast<uint8_t>(value);
    }
    return img;
}

}   // namespace album
